package org.keycloak.authentication.authenticators.resetcred;

import org.jboss.logging.Logger;
import org.keycloak.ClientConnection;
import org.keycloak.Config;
import org.keycloak.authentication.AuthenticationFlowContext;
import org.keycloak.authentication.AuthenticationFlowError;
import org.keycloak.authentication.Authenticator;
import org.keycloak.authentication.AuthenticatorFactory;
import org.keycloak.email.EmailException;
import org.keycloak.email.EmailProvider;
import org.keycloak.events.Details;
import org.keycloak.events.Errors;
import org.keycloak.events.EventBuilder;
import org.keycloak.jose.jws.JWSBuilder;
import org.keycloak.jose.jws.JWSInput;
import org.keycloak.login.LoginFormsProvider;
import org.keycloak.models.AuthenticationExecutionModel;
import org.keycloak.models.ClientSessionModel;
import org.keycloak.models.KeycloakSession;
import org.keycloak.models.KeycloakSessionFactory;
import org.keycloak.models.RealmModel;
import org.keycloak.models.UserModel;
import org.keycloak.models.UserSessionModel;
import org.keycloak.models.utils.HmacOTP;
import org.keycloak.protocol.oidc.TokenManager;
import org.keycloak.provider.ProviderConfigProperty;
import org.keycloak.services.Urls;
import org.keycloak.services.messages.Messages;

import javax.crypto.SecretKey;
import javax.ws.rs.core.MultivaluedMap;
import javax.ws.rs.core.Response;
import javax.ws.rs.core.UriBuilder;
import java.util.List;
import java.util.concurrent.TimeUnit;

/**
 * @author <a href="mailto:bill@burkecentral.com">Bill Burke</a>
 * @version $Revision: 1 $
 */
public class ResetCredentialEmail implements Authenticator, AuthenticatorFactory {
    public static final String RESET_CREDENTIAL_SECRET = "RESET_CREDENTIAL_SECRET";
    public static final String KEY = "key";
    protected static Logger logger = Logger.getLogger(ResetCredentialEmail.class);

    public static final String PROVIDER_ID = "reset-credential-email";

    @Override
    public void authenticate(AuthenticationFlowContext context) {
        UserModel user = context.getUser();
        EventBuilder event = context.getEvent();
        if (user.getEmail() == null || user.getEmail().trim().length() == 0) {
            event.user(user).error(Errors.INVALID_EMAIL);
            Response challenge = context.form()
                    .setError(Messages.INVALID_EMAIL)
                    .createPasswordReset();
            context.failureChallenge(AuthenticationFlowError.INVALID_USER, challenge);
            return;
        }

        // We send the secret in the email in a link as a query param.  We don't need to sign it or anything because
        // it can only be guessed once, and it must match watch is stored in the client session.
        String secret = HmacOTP.generateSecret(10);
        context.getClientSession().setNote(RESET_CREDENTIAL_SECRET, secret);
        String link = UriBuilder.fromUri(context.getActionUrl()).queryParam(KEY, secret).build().toString();
        long expiration = TimeUnit.SECONDS.toMinutes(context.getRealm().getAccessCodeLifespanUserAction());
        try {

            context.getSession().getProvider(EmailProvider.class).setRealm(context.getRealm()).setUser(user).sendPasswordReset(secret, link, expiration);

            event.detail(Details.EMAIL, user.getEmail()).detail(Details.CODE_ID, context.getClientSession().getId()).success();
            Response challenge = context.form()
                    .setSuccess(Messages.EMAIL_SENT)
                    .createForm("validate-reset-email.ftl");
            context.challenge(challenge);
        } catch (EmailException e) {
            event.error(Errors.EMAIL_SEND_FAILED);
            logger.error("Failed to send password reset email", e);
            Response challenge = context.form()
                    .setError(Messages.EMAIL_SENT_ERROR)
                    .createErrorPage();
            context.failure(AuthenticationFlowError.INTERNAL_ERROR, challenge);
        }
    }

    @Override
    public void action(AuthenticationFlowContext context) {
        String secret = context.getClientSession().getNote(RESET_CREDENTIAL_SECRET);
        String key = null;
        if (context.getHttpRequest().getHttpMethod().equalsIgnoreCase("GET")) {
            key =context.getUriInfo().getQueryParameters().getFirst(KEY);

        } else if (context.getHttpRequest().getHttpMethod().equalsIgnoreCase("POST")) {
            key = context.getHttpRequest().getDecodedFormParameters().getFirst(KEY);
        }

        // Can only guess once!  We remove the note so another guess can't happen
        context.getClientSession().removeNote(RESET_CREDENTIAL_SECRET);
        if (secret == null || key == null || !secret.equals(key)) {
            context.getEvent().error(Errors.INVALID_USER_CREDENTIALS);
            Response challenge = context.form()
                    .setError(Messages.INVALID_ACCESS_CODE)
                    .createErrorPage();
            context.failure(AuthenticationFlowError.INTERNAL_ERROR, challenge);
            return;
        }
        context.success();
    }

    @Override
    public boolean requiresUser() {
        return true;
    }

    @Override
    public boolean configuredFor(KeycloakSession session, RealmModel realm, UserModel user) {
        return true;
    }

    @Override
    public void setRequiredActions(KeycloakSession session, RealmModel realm, UserModel user) {

    }

    @Override
    public String getDisplayType() {
        return "Reset Via Email";
    }

    @Override
    public String getReferenceCategory() {
        return null;
    }

    @Override
    public boolean isConfigurable() {
        return false;
    }

    public static final AuthenticationExecutionModel.Requirement[] REQUIREMENT_CHOICES = {
            AuthenticationExecutionModel.Requirement.REQUIRED
    };

    @Override
    public AuthenticationExecutionModel.Requirement[] getRequirementChoices() {
        return REQUIREMENT_CHOICES;
    }

    @Override
    public boolean isUserSetupAllowed() {
        return false;
    }

    @Override
    public String getHelpText() {
        return "Send email to user and wait for response.";
    }

    @Override
    public List<ProviderConfigProperty> getConfigProperties() {
        return null;
    }

    @Override
    public void close() {

    }

    @Override
    public Authenticator create(KeycloakSession session) {
        return this;
    }

    @Override
    public void init(Config.Scope config) {

    }

    @Override
    public void postInit(KeycloakSessionFactory factory) {

    }

    @Override
    public String getId() {
        return PROVIDER_ID;
    }
}
