/*
 * Copyright 2014-2015 Groupon, Inc
 * Copyright 2014-2015 The Billing Project, LLC
 *
 * The Billing Project licenses this file to you under the Apache License, version 2.0
 * (the "License"); you may not use this file except in compliance with the
 * License.  You may obtain a copy of the License at:
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.  See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

package org.killbill.billing.jaxrs.json;

import java.util.Set;

import org.killbill.billing.osgi.api.PluginInfo;
import org.killbill.billing.osgi.api.PluginServiceInfo;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonProperty;
import com.google.common.base.Function;
import com.google.common.collect.ImmutableSet;
import com.google.common.collect.Iterables;

public class PluginInfoJson {

    private final String bundleSymbolicName;

    private final String pluginName;

    private final String version;

    private final boolean running;

    private final Set<PluginServiceInfoJson> services;

    @JsonCreator
    public PluginInfoJson(@JsonProperty("bundleSymbolicName") final String bundleSymbolicName,
                          @JsonProperty("pluginName") final String pluginName,
                          @JsonProperty("version") final String version,
                          @JsonProperty("running") final boolean running,
                          @JsonProperty("services") final Set<PluginServiceInfoJson> services) {
        this.bundleSymbolicName = bundleSymbolicName;
        this.pluginName = pluginName;
        this.version = version;
        this.running = running;
        this.services = services;
    }

    public PluginInfoJson(final PluginInfo input) {
        this(input.getBundleSymbolicName(),
             input.getPluginName(),
             input.getVersion(),
             input.isRunning(),
             ImmutableSet.copyOf(Iterables.transform(input.getServices(), new Function<PluginServiceInfo, PluginServiceInfoJson>() {
                 @Override
                 public PluginServiceInfoJson apply(final PluginServiceInfo input) {
                     return new PluginServiceInfoJson(input.getServiceTypeName(), input.getRegistrationName());
                 }
             })));
    }

    public String getBundleSymbolicName() {
        return bundleSymbolicName;
    }

    public String getPluginName() {
        return pluginName;
    }

    public String getVersion() {
        return version;
    }

    public boolean isRunning() {
        return running;
    }

    public Set<PluginServiceInfoJson> getServices() {
        return services;
    }

    public static class PluginServiceInfoJson {

        private final String serviceTypeName;
        private final String registrationName;

        @JsonCreator
        public PluginServiceInfoJson(@JsonProperty("serviceTypeName") final String serviceTypeName,
                                     @JsonProperty("registrationName") final String registrationName) {
            this.serviceTypeName = serviceTypeName;
            this.registrationName = registrationName;
        }

        public String getServiceTypeName() {
            return serviceTypeName;
        }

        public String getRegistrationName() {
            return registrationName;
        }
    }
}
