/*
 * Copyright 2014 Groupon, Inc
 * Copyright 2014 The Billing Project, LLC
 *
 * The Billing Project licenses this file to you under the Apache License, version 2.0
 * (the "License"); you may not use this file except in compliance with the
 * License.  You may obtain a copy of the License at:
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.  See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

package org.killbill.billing.tenant.api;

import java.util.List;
import java.util.Locale;

import javax.inject.Inject;
import javax.inject.Named;

import org.killbill.billing.callcontext.InternalTenantContext;
import org.killbill.billing.tenant.api.TenantCacheInvalidation.CacheInvalidationKey;
import org.killbill.billing.tenant.api.TenantKV.TenantKey;
import org.killbill.billing.tenant.dao.TenantDao;
import org.killbill.billing.tenant.glue.DefaultTenantModule;
import org.killbill.billing.util.LocaleUtils;

/**
 * This is the private API which is used to extract per tenant objects (catalog, overdue, invoice templates, ..)
 * <p>
 * Some of these per tenant objects are cached at a higher level in their respective modules (catalog, overdue) to
 * avoid reconstructing the object state from the xml definition each time. As a result, the module also registers
 * a callback which is used for the cache invalidation when the state changes and the operation occurred on a remote node.
 * For those objects, the private api is called from the module.
 * <p>
 * Some others (invoice templates,...) are not cached (yet) and so the logic is simpler.
 * <p>
 * The api can only be used to retrieve objects where no caching is required.
 */
public class DefaultTenantInternalApi implements TenantInternalApi {

    private final TenantDao tenantDao;
    private final TenantCacheInvalidation tenantCacheInvalidation;


    @Inject
    public DefaultTenantInternalApi(@Named(DefaultTenantModule.NO_CACHING_TENANT) final TenantDao tenantDao,
                                    final TenantCacheInvalidation tenantCacheInvalidation) {
        this.tenantDao = tenantDao;
        this.tenantCacheInvalidation = tenantCacheInvalidation;
    }

    @Override
    public List<String> getTenantCatalogs(final InternalTenantContext tenantContext, final CacheInvalidationCallback cacheInvalidationCallback) {
        tenantCacheInvalidation.registerCallback(new CacheInvalidationKey(tenantContext.getTenantRecordId(), TenantKey.CATALOG), cacheInvalidationCallback);
        return tenantDao.getTenantValueForKey(TenantKey.CATALOG.toString(), tenantContext);
    }

    @Override
    public String getTenantOverdueConfig(final InternalTenantContext tenantContext, final CacheInvalidationCallback cacheInvalidationCallback) {
        tenantCacheInvalidation.registerCallback(new CacheInvalidationKey(tenantContext.getTenantRecordId(), TenantKey.OVERDUE_CONFIG), cacheInvalidationCallback);
        final List<String> values = tenantDao.getTenantValueForKey(TenantKey.OVERDUE_CONFIG.toString(), tenantContext);
        return getUniqueValue(values, "overdue config", tenantContext);
    }

    @Override
    public String getInvoiceTemplate(final Locale locale, final InternalTenantContext tenantContext) {
        final List<String> values = tenantDao.getTenantValueForKey(LocaleUtils.localeString(locale, TenantKey.INVOICE_TEMPLATE.toString()), tenantContext);
        return getUniqueValue(values, "invoice template", tenantContext);
    }

    @Override
    public String getManualPayInvoiceTemplate(final Locale locale, final InternalTenantContext tenantContext) {
        final List<String> values = tenantDao.getTenantValueForKey(LocaleUtils.localeString(locale, TenantKey.INVOICE_MP_TEMPLATE.toString()), tenantContext);
        return getUniqueValue(values, "manual pay invoice template", tenantContext);
    }

    @Override
    public String getInvoiceTranslation(final Locale locale, final InternalTenantContext tenantContext) {
        final List<String> values = tenantDao.getTenantValueForKey(LocaleUtils.localeString(locale, TenantKey.INVOICE_TRANSLATION_.toString()), tenantContext);
        return getUniqueValue(values, "invoice translation", tenantContext);
    }

    @Override
    public String getCatalogTranslation(final Locale locale, final InternalTenantContext tenantContext) {
        final List<String> values = tenantDao.getTenantValueForKey(LocaleUtils.localeString(locale, TenantKey.CATALOG_TRANSLATION_.toString()), tenantContext);
        return getUniqueValue(values, "catalog translation", tenantContext);
    }

    private String getUniqueValue(final List<String> values, final String msg, final InternalTenantContext tenantContext) {
        if (values.isEmpty()) {
            return null;
        }
        if (values.size() > 1) {
            throw new IllegalStateException(String.format("Unexpected number of values %d for %s and tenant %d",
                                                          values.size(), msg, tenantContext.getTenantRecordId()));
        }
        return values.get(0);
    }
}
