/*
 * Copyright 2010-2011 Ning, Inc.
 *
 * Ning licenses this file to you under the Apache License, version 2.0
 * (the "License"); you may not use this file except in compliance with the
 * License.  You may obtain a copy of the License at:
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.  See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

package com.ning.billing.invoice.dao;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.Collection;
import java.util.LinkedHashMap;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.UUID;

import com.ning.billing.invoice.api.InvoicePayment;
import com.ning.billing.util.eventbus.Bus;
import org.joda.time.DateTime;

import com.google.inject.Inject;
import com.ning.billing.invoice.api.Invoice;
import com.ning.billing.invoice.api.InvoiceItem;
import com.ning.billing.invoice.api.user.DefaultInvoiceCreationNotification;

public class MockInvoiceDao implements InvoiceDao {
    private final Bus eventBus;
    private final Object monitor = new Object();
    private final Map<UUID, Invoice> invoices = new LinkedHashMap<UUID, Invoice>();

    @Inject
    public MockInvoiceDao(Bus eventBus) {
        this.eventBus = eventBus;
    }

    @Override
    public void create(Invoice invoice) {
        synchronized (monitor) {
            invoices.put(invoice.getId(), invoice);
        }
        try {
            eventBus.post(new DefaultInvoiceCreationNotification(invoice.getId(), invoice.getAccountId(),
                                                                 invoice.getBalance(), invoice.getCurrency(),
                                                                 invoice.getInvoiceDate()));
        }
        catch (Bus.EventBusException ex) {
            throw new RuntimeException(ex);
        }
    }

//    private Invoice munge(Invoice invoice) {
//        if (invoice == null) {
//            return null;
//        }
//
//        DateTime lastPaymentDate = null;
//        BigDecimal amountPaid = new BigDecimal("0");
//
//        for (InvoicePayment invoicePayment : invoicePayments.values()) {
//            if (invoicePayment.getInvoiceId().equals(invoice.getId())) {
//                if (lastPaymentDate == null || lastPaymentDate.isBefore(invoicePayment.getPaymentAttemptDate())) {
//                    lastPaymentDate = invoicePayment.getPaymentAttemptDate();
//                }
//                if (invoicePayment.getAmount() != null) {
//                    amountPaid = amountPaid.add(invoicePayment.getAmount());
//                }
//            }
//        }
//
//        Invoice newInvoice = new DefaultInvoice(invoice.getId(),
//                                                invoice.getAccountId(),
//                                                invoice.getInvoiceDate(),
//                                                invoice.getTargetDate(),
//                                                invoice.getCurrency());
//        newInvoice.addInvoiceItems(invoice.getInvoiceItems());
//        newInvoice.addPayments(invoice.getPayments());
//
//        return newInvoice;
//    }
//
//    private List<Invoice> munge(Collection<Invoice> invoices) {
//        List<Invoice> result = new ArrayList<Invoice>();
//        for (Invoice invoice : invoices) {
//            result.add(munge(invoice));
//        }
//        return result;
//    }

    @Override
    public Invoice getById(UUID id) {
        synchronized (monitor) {
            return invoices.get(id);
        }
    }

    @Override
    public List<Invoice> get() {
        synchronized (monitor) {
            return new ArrayList<Invoice>(invoices.values());
        }
    }

    @Override
    public List<Invoice> getInvoicesByAccount(UUID accountId) {
        List<Invoice> result = new ArrayList<Invoice>();

        synchronized (monitor) {
            for (Invoice invoice : invoices.values()) {
                if (accountId.equals(invoice.getAccountId())) {
                    result.add(invoice);
                }
            }
        }
        return result;
    }

    @Override
    public List<Invoice> getInvoicesByAccount(UUID accountId, DateTime fromDate) {
        List<Invoice> invoicesForAccount = new ArrayList<Invoice>();

        synchronized (monitor) {
            for (Invoice invoice : get()) {
                if (accountId.equals(invoice.getAccountId()) && !invoice.getTargetDate().isBefore(fromDate)) {
                    invoicesForAccount.add(invoice);
                }
            }
        }

        return invoicesForAccount;
    }

    @Override
    public List<InvoiceItem> getInvoiceItemsByAccount(UUID accountId) {
        List<InvoiceItem> invoiceItemsForAccount = new ArrayList<InvoiceItem>();

        synchronized (monitor) {
            for (Invoice invoice : get()) {
                if (accountId.equals(invoice.getAccountId())) {
                    invoiceItemsForAccount.addAll(invoice.getInvoiceItems());
                }
            }
        }

        return invoiceItemsForAccount;
    }

    @Override
    public List<Invoice> getInvoicesBySubscription(UUID subscriptionId) {
        List<Invoice> result = new ArrayList<Invoice>();

        synchronized (monitor) {
            for (Invoice invoice : invoices.values()) {
                for (InvoiceItem item : invoice.getInvoiceItems()) {
                    if (subscriptionId.equals(item.getSubscriptionId())) {
                        result.add(invoice);
                        break;
                    }
                }
            }
        }
        return result;
    }

    @Override
    public List<UUID> getInvoicesForPayment(DateTime targetDate, int numberOfDays) {
        List<UUID> result = new ArrayList<UUID>();

        synchronized (monitor) {
            for (Invoice invoice : invoices.values()) {
                if (invoice.isDueForPayment(targetDate, numberOfDays)) {
                    result.add(invoice.getId());
                }
            }
        }

        return result;
    }

    @Override
    public void test() {
    }

    @Override
    public UUID getInvoiceIdByPaymentAttemptId(UUID paymentAttemptId) {
        synchronized(monitor) {
            for (Invoice invoice : invoices.values()) {
                for (InvoicePayment payment : invoice.getPayments()) {
                    if (paymentAttemptId.equals(payment.getPaymentAttemptId())) {
                        return invoice.getId();
                    }
                }
            }
        }
        return null;
    }

    @Override
    public InvoicePayment getInvoicePayment(UUID paymentAttemptId) {
        synchronized(monitor) {
            for (Invoice invoice : invoices.values()) {
                for (InvoicePayment payment : invoice.getPayments()) {
                    if (paymentAttemptId.equals(payment.getPaymentAttemptId())) {
                        return payment;
                    }
                }
            }
        }

        return null;
    }

    @Override
    public void notifyOfPaymentAttempt(InvoicePayment invoicePayment) {
        synchronized (monitor) {
            Invoice invoice = invoices.get(invoicePayment.getInvoiceId());
            if (invoice != null) {
                invoice.addPayment(invoicePayment);
            }
        }
    }

    @Override
    public BigDecimal getAccountBalance(UUID accountId) {
        BigDecimal balance = BigDecimal.ZERO;

        for (Invoice invoice : get()) {
            if (accountId.equals(invoice.getAccountId())) {
                balance = balance.add(invoice.getBalance());
            }
        }

        return balance;
    }

    @Override
    public List<Invoice> getUnpaidInvoicesByAccountId(UUID accountId, DateTime upToDate) {
        List<Invoice> unpaidInvoices = new ArrayList<Invoice>();

        for (Invoice invoice : get()) {
            if (accountId.equals(invoice.getAccountId()) && (invoice.getBalance().compareTo(BigDecimal.ZERO) > 0)) {
                unpaidInvoices.add(invoice);
            }
        }

        return unpaidInvoices;
    }
}
