/*
 * Copyright 2010-2014 Ning, Inc.
 *
 * Ning licenses this file to you under the Apache License, version 2.0
 * (the "License"); you may not use this file except in compliance with the
 * License.  You may obtain a copy of the License at:
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.  See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

package com.ning.billing.invoice.tree;

import java.util.LinkedList;
import java.util.List;
import java.util.UUID;

import org.joda.time.LocalDate;
import org.testng.annotations.Test;

import com.ning.billing.invoice.tree.NodeInterval.AddNodeCallback;
import com.ning.billing.invoice.tree.NodeInterval.BuildNodeCallback;
import com.ning.billing.invoice.tree.NodeInterval.SearchCallback;
import com.ning.billing.invoice.tree.NodeInterval.WalkCallback;

import static org.testng.Assert.assertEquals;
import static org.testng.Assert.assertNull;
import static org.testng.Assert.assertTrue;

public class TestNodeInterval /* extends InvoiceTestSuiteNoDB  */ {

    private AddNodeCallback CALLBACK = new DummyAddNodeCallback();

    public class DummyNodeInterval extends NodeInterval {

        private final UUID id;

        public DummyNodeInterval() {
            this.id = UUID.randomUUID();
        }

        public DummyNodeInterval(final NodeInterval parent, final LocalDate startDate, final LocalDate endDate) {
            super(parent, startDate, endDate);
            this.id = UUID.randomUUID();
        }

        public UUID getId() {
            return id;
        }
    }

    public class DummyAddNodeCallback implements AddNodeCallback {

        @Override
        public boolean onExistingNode(final NodeInterval existingNode) {
            return false;
        }

        @Override
        public boolean shouldInsertNode(final NodeInterval insertionNode) {
            return true;
        }
    }

    @Test(groups = "fast")
    public void testAddExistingItemSimple() {
        final DummyNodeInterval root = new DummyNodeInterval();

        final DummyNodeInterval top = createNodeInterval("2014-01-01", "2014-02-01");
        root.addNode(top, CALLBACK);

        final DummyNodeInterval firstChildLevel1 = createNodeInterval("2014-01-01", "2014-01-07");
        final DummyNodeInterval secondChildLevel1 = createNodeInterval("2014-01-08", "2014-01-15");
        final DummyNodeInterval thirdChildLevel1 = createNodeInterval("2014-01-16", "2014-02-01");
        root.addNode(firstChildLevel1, CALLBACK);
        root.addNode(secondChildLevel1, CALLBACK);
        root.addNode(thirdChildLevel1, CALLBACK);

        final DummyNodeInterval firstChildLevel2 = createNodeInterval("2014-01-01", "2014-01-03");
        final DummyNodeInterval secondChildLevel2 = createNodeInterval("2014-01-03", "2014-01-5");
        final DummyNodeInterval thirdChildLevel2 = createNodeInterval("2014-01-16", "2014-01-17");
        root.addNode(firstChildLevel2, CALLBACK);
        root.addNode(secondChildLevel2, CALLBACK);
        root.addNode(thirdChildLevel2, CALLBACK);

        checkNode(top, 3, root, firstChildLevel1, null);
        checkNode(firstChildLevel1, 2, top, firstChildLevel2, secondChildLevel1);
        checkNode(secondChildLevel1, 0, top, null, thirdChildLevel1);
        checkNode(thirdChildLevel1, 1, top, thirdChildLevel2, null);

        checkNode(firstChildLevel2, 0, firstChildLevel1, null, secondChildLevel2);
        checkNode(secondChildLevel2, 0, firstChildLevel1, null, null);
        checkNode(thirdChildLevel2, 0, thirdChildLevel1, null, null);
    }

    @Test(groups = "fast")
    public void testAddExistingItemWithRebalance() {
        final DummyNodeInterval root = new DummyNodeInterval();

        final DummyNodeInterval top = createNodeInterval("2014-01-01", "2014-02-01");
        root.addNode(top, CALLBACK);

        final DummyNodeInterval firstChildLevel2 = createNodeInterval("2014-01-01", "2014-01-03");
        final DummyNodeInterval secondChildLevel2 = createNodeInterval("2014-01-03", "2014-01-5");
        final DummyNodeInterval thirdChildLevel2 = createNodeInterval("2014-01-16", "2014-01-17");
        root.addNode(firstChildLevel2, CALLBACK);
        root.addNode(secondChildLevel2, CALLBACK);
        root.addNode(thirdChildLevel2, CALLBACK);

        final DummyNodeInterval firstChildLevel1 = createNodeInterval("2014-01-01", "2014-01-07");
        final DummyNodeInterval secondChildLevel1 = createNodeInterval("2014-01-08", "2014-01-15");
        final DummyNodeInterval thirdChildLevel1 = createNodeInterval("2014-01-16", "2014-02-01");
        root.addNode(firstChildLevel1, CALLBACK);
        root.addNode(secondChildLevel1, CALLBACK);
        root.addNode(thirdChildLevel1, CALLBACK);

        checkNode(top, 3, root, firstChildLevel1, null);
        checkNode(firstChildLevel1, 2, top, firstChildLevel2, secondChildLevel1);
        checkNode(secondChildLevel1, 0, top, null, thirdChildLevel1);
        checkNode(thirdChildLevel1, 1, top, thirdChildLevel2, null);

        checkNode(firstChildLevel2, 0, firstChildLevel1, null, secondChildLevel2);
        checkNode(secondChildLevel2, 0, firstChildLevel1, null, null);
        checkNode(thirdChildLevel2, 0, thirdChildLevel1, null, null);
    }

    @Test(groups = "fast")
    public void testBuild() {
        final DummyNodeInterval root = new DummyNodeInterval();

        final DummyNodeInterval top = createNodeInterval("2014-01-01", "2014-02-01");
        root.addNode(top, CALLBACK);

        final DummyNodeInterval firstChildLevel1 = createNodeInterval("2014-01-01", "2014-01-07");
        final DummyNodeInterval secondChildLevel1 = createNodeInterval("2014-01-08", "2014-01-15");
        final DummyNodeInterval thirdChildLevel1 = createNodeInterval("2014-01-16", "2014-02-01");
        root.addNode(firstChildLevel1, CALLBACK);
        root.addNode(secondChildLevel1, CALLBACK);
        root.addNode(thirdChildLevel1, CALLBACK);

        final DummyNodeInterval firstChildLevel2 = createNodeInterval("2014-01-01", "2014-01-03");
        final DummyNodeInterval secondChildLevel2 = createNodeInterval("2014-01-03", "2014-01-5");
        final DummyNodeInterval thirdChildLevel2 = createNodeInterval("2014-01-16", "2014-01-17");
        root.addNode(firstChildLevel2, CALLBACK);
        root.addNode(secondChildLevel2, CALLBACK);
        root.addNode(thirdChildLevel2, CALLBACK);

        final List<NodeInterval> output = new LinkedList<NodeInterval>();

        // Just build the missing pieces.
        root.build(new BuildNodeCallback() {
            @Override
            public void onMissingInterval(final NodeInterval curNode, final LocalDate startDate, final LocalDate endDate) {
                output.add(createNodeInterval(startDate, endDate));
            }

            @Override
            public void onLastNode(final NodeInterval curNode) {
                // Nothing
            }
        });

        final List<NodeInterval> expected = new LinkedList<NodeInterval>();
        expected.add(createNodeInterval("2014-01-05", "2014-01-07"));
        expected.add(createNodeInterval("2014-01-07", "2014-01-08"));
        expected.add(createNodeInterval("2014-01-15", "2014-01-16"));
        expected.add(createNodeInterval("2014-01-17", "2014-02-01"));

        assertEquals(output.size(), expected.size());
        checkInterval(output.get(0), expected.get(0));
        checkInterval(output.get(1), expected.get(1));
    }

    @Test(groups = "fast")
    public void testSearch() {
        final DummyNodeInterval root = new DummyNodeInterval();

        final DummyNodeInterval top = createNodeInterval("2014-01-01", "2014-02-01");
        root.addNode(top, CALLBACK);

        final DummyNodeInterval firstChildLevel1 = createNodeInterval("2014-01-01", "2014-01-07");
        final DummyNodeInterval secondChildLevel1 = createNodeInterval("2014-01-08", "2014-01-15");
        final DummyNodeInterval thirdChildLevel1 = createNodeInterval("2014-01-16", "2014-02-01");
        root.addNode(firstChildLevel1, CALLBACK);
        root.addNode(secondChildLevel1, CALLBACK);
        root.addNode(thirdChildLevel1, CALLBACK);

        final DummyNodeInterval firstChildLevel2 = createNodeInterval("2014-01-01", "2014-01-03");
        final DummyNodeInterval secondChildLevel2 = createNodeInterval("2014-01-03", "2014-01-5");
        final DummyNodeInterval thirdChildLevel2 = createNodeInterval("2014-01-16", "2014-01-17");
        root.addNode(firstChildLevel2, CALLBACK);
        root.addNode(secondChildLevel2, CALLBACK);
        root.addNode(thirdChildLevel2, CALLBACK);

        final DummyNodeInterval firstChildLevel3 = createNodeInterval("2014-01-01", "2014-01-02");
        final DummyNodeInterval secondChildLevel3 = createNodeInterval("2014-01-03", "2014-01-04");
        root.addNode(firstChildLevel3, CALLBACK);
        root.addNode(secondChildLevel3, CALLBACK);

        final NodeInterval search1 = root.findNode(new LocalDate("2014-01-04"), new SearchCallback() {
            @Override
            public boolean isMatch(final NodeInterval curNode) {
                return ((DummyNodeInterval) curNode).getId().equals(secondChildLevel3.getId());
            }
        });
        checkInterval(search1, secondChildLevel3);

        final NodeInterval search2 = root.findNode(new SearchCallback() {
            @Override
            public boolean isMatch(final NodeInterval curNode) {
                return ((DummyNodeInterval) curNode).getId().equals(thirdChildLevel2.getId());
            }
        });
        checkInterval(search2, thirdChildLevel2);

        final NodeInterval nullSearch = root.findNode(new SearchCallback() {
            @Override
            public boolean isMatch(final NodeInterval curNode) {
                return ((DummyNodeInterval) curNode).getId().equals("foo");
            }
        });
        assertNull(nullSearch);
    }

    @Test(groups = "fast")
    public void testWalkTree() {
        final DummyNodeInterval root = new DummyNodeInterval();

        final DummyNodeInterval firstChildLevel0 = createNodeInterval("2014-01-01", "2014-02-01");
        root.addNode(firstChildLevel0, CALLBACK);

        final DummyNodeInterval secondChildLevel0 = createNodeInterval("2014-02-01", "2014-03-01");
        root.addNode(secondChildLevel0, CALLBACK);

        final DummyNodeInterval firstChildLevel1 = createNodeInterval("2014-01-01", "2014-01-07");
        final DummyNodeInterval secondChildLevel1 = createNodeInterval("2014-01-08", "2014-01-15");
        final DummyNodeInterval thirdChildLevel1 = createNodeInterval("2014-01-16", "2014-02-01");
        root.addNode(firstChildLevel1, CALLBACK);
        root.addNode(secondChildLevel1, CALLBACK);
        root.addNode(thirdChildLevel1, CALLBACK);

        final DummyNodeInterval firstChildLevel2 = createNodeInterval("2014-01-01", "2014-01-03");
        final DummyNodeInterval secondChildLevel2 = createNodeInterval("2014-01-03", "2014-01-05");
        final DummyNodeInterval thirdChildLevel2 = createNodeInterval("2014-01-16", "2014-01-17");
        root.addNode(firstChildLevel2, CALLBACK);
        root.addNode(secondChildLevel2, CALLBACK);
        root.addNode(thirdChildLevel2, CALLBACK);

        final DummyNodeInterval firstChildLevel3 = createNodeInterval("2014-01-01", "2014-01-02");
        final DummyNodeInterval secondChildLevel3 = createNodeInterval("2014-01-03", "2014-01-04");
        root.addNode(firstChildLevel3, CALLBACK);
        root.addNode(secondChildLevel3, CALLBACK);

        final List<NodeInterval> expected = new LinkedList<NodeInterval>();
        expected.add(root);
        expected.add(firstChildLevel0);
        expected.add(firstChildLevel1);
        expected.add(firstChildLevel2);
        expected.add(firstChildLevel3);
        expected.add(secondChildLevel2);
        expected.add(secondChildLevel3);
        expected.add(secondChildLevel1);
        expected.add(thirdChildLevel1);
        expected.add(thirdChildLevel2);
        expected.add(secondChildLevel0);

        final List<NodeInterval> result = new LinkedList<NodeInterval>();
        root.walkTree(new WalkCallback() {
            @Override
            public void onCurrentNode(final int depth, final NodeInterval curNode, final NodeInterval parent) {
                result.add(curNode);
            }
        });

        assertEquals(result.size(), expected.size());
        for (int i = 0; i < result.size(); i++) {
            if (i == 0) {
                assertTrue(result.get(0).isRoot());
                checkInterval(result.get(0), createNodeInterval("2014-01-01", "2014-03-01"));
            } else {
                checkInterval(result.get(i), expected.get(i));
            }
        }
    }

    private void checkInterval(final NodeInterval real, final NodeInterval expected) {
        assertEquals(real.getStart(), expected.getStart());
        assertEquals(real.getEnd(), expected.getEnd());
    }

    private void checkNode(final NodeInterval node, final int expectedChildren, final DummyNodeInterval expectedParent, final DummyNodeInterval expectedLeftChild, final DummyNodeInterval expectedRightSibling) {
        assertEquals(node.getNbChildren(), expectedChildren);
        assertEquals(node.getParent(), expectedParent);
        assertEquals(node.getRightSibling(), expectedRightSibling);
        assertEquals(node.getLeftChild(), expectedLeftChild);
        assertEquals(node.getLeftChild(), expectedLeftChild);
    }

    private DummyNodeInterval createNodeInterval(final LocalDate startDate, final LocalDate endDate) {
        return new DummyNodeInterval(null, startDate, endDate);
    }

    private DummyNodeInterval createNodeInterval(final String startDate, final String endDate) {
        return createNodeInterval(new LocalDate(startDate), new LocalDate(endDate));
    }

}
