/*
 * Copyright 2010-2012 Ning, Inc.
 *
 * Ning licenses this file to you under the Apache License, version 2.0
 * (the "License"); you may not use this file except in compliance with the
 * License.  You may obtain a copy of the License at:
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.  See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

package com.ning.billing.jaxrs.json;

import java.math.BigDecimal;
import java.util.UUID;

import org.joda.time.DateTime;
import org.testng.Assert;
import org.testng.annotations.Test;

import com.ning.billing.jaxrs.JaxrsTestSuite;
import com.ning.billing.util.clock.Clock;
import com.ning.billing.util.clock.DefaultClock;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.SerializationFeature;
import com.fasterxml.jackson.datatype.joda.JodaModule;

public class TestCreditJson extends JaxrsTestSuite {

    private static final ObjectMapper mapper = new ObjectMapper();

    private final Clock clock = new DefaultClock();

    static {
        mapper.registerModule(new JodaModule());
        mapper.disable(SerializationFeature.WRITE_DATES_AS_TIMESTAMPS);
    }

    @Test(groups = "fast")
    public void testJson() throws Exception {
        final BigDecimal creditAmount = BigDecimal.TEN;
        final UUID invoiceId = UUID.randomUUID();
        final String invoiceNumber = UUID.randomUUID().toString();
        final DateTime requestedDate = clock.getUTCNow();
        final DateTime effectiveDate = clock.getUTCNow();
        final String reason = UUID.randomUUID().toString();
        final UUID accountId = UUID.randomUUID();

        final CreditJson creditJson = new CreditJson(creditAmount, invoiceId, invoiceNumber, requestedDate, effectiveDate, reason, accountId);
        Assert.assertEquals(creditJson.getRequestedDate(), requestedDate);
        Assert.assertEquals(creditJson.getEffectiveDate(), effectiveDate);
        Assert.assertEquals(creditJson.getCreditAmount(), creditAmount);
        Assert.assertEquals(creditJson.getInvoiceId(), invoiceId);
        Assert.assertEquals(creditJson.getInvoiceNumber(), invoiceNumber);
        Assert.assertEquals(creditJson.getReason(), reason);
        Assert.assertEquals(creditJson.getAccountId(), accountId);

        final String asJson = mapper.writeValueAsString(creditJson);
        Assert.assertEquals(asJson, "{\"creditAmount\":" + creditJson.getCreditAmount() + "," +
                                    "\"invoiceId\":\"" + creditJson.getInvoiceId().toString() + "\"," +
                                    "\"invoiceNumber\":\"" + creditJson.getInvoiceNumber() + "\"," +
                                    "\"requestedDate\":\"" + creditJson.getRequestedDate().toDateTimeISO().toString() + "\"," +
                                    "\"effectiveDate\":\"" + creditJson.getEffectiveDate().toDateTimeISO().toString() + "\"," +
                                    "\"reason\":\"" + creditJson.getReason() + "\"," +
                                    "\"accountId\":\"" + creditJson.getAccountId().toString() + "\"}");

        final CreditJson fromJson = mapper.readValue(asJson, CreditJson.class);
        Assert.assertEquals(fromJson, creditJson);
    }
}
