package bdi4jade.extension.remediation;

import bdi4jade.core.MetadataElement;
import bdi4jade.core.MetadataElementImpl;

/**
 * This class provides a default implementation for a resource, representing it
 * just with a given name.
 * 
 * It implements the {@link MetadataElement} interface, allowing to associate
 * metadata with resources.
 * 
 * @author Ingrid Nunes
 * 
 */
public class NamedResource extends MetadataElementImpl implements Resource, MetadataElement {

	private static final long serialVersionUID = 3958189054716876043L;

	private String name;

	/**
	 * The default constructor. It should be only used if persistence frameworks
	 * are used.
	 */
	protected NamedResource() {

	}

	/**
	 * Initializes a resource with its name.
	 * 
	 * @param name
	 *            the resource name.
	 */
	public NamedResource(String name) {
		this.name = name;
	}

	/**
	 * Returns true of the object is a named resource and has the same name of
	 * this named resource.
	 * 
	 * @param obj
	 *            to object to be tested if it is equal to this named resource.
	 * 
	 * @see Object#equals(Object)
	 */
	@Override
	public final boolean equals(Object obj) {
		if (obj instanceof NamedResource) {
			NamedResource sg = (NamedResource) obj;
			return this.name.equals(sg.name);
		}
		return false;
	}

	/**
	 * Returns the name of this resource.
	 * 
	 * @return the name.
	 */
	public String getName() {
		return name;
	}

	/**
	 * Returns the hash code of this named resource.
	 * 
	 * @return the hash code of the name of this resource.
	 * 
	 * @see java.lang.Object#hashCode()
	 */
	@Override
	public final int hashCode() {
		return name == null ? 0 : this.name.hashCode();
	}

	/**
	 * Sets the name of this resource. Ideally, the name should be final and
	 * initialized in the constructor. This method should be only used if
	 * persistence frameworks are used.
	 * 
	 * @param name
	 *            the name to set.
	 */
	protected void setName(String name) {
		this.name = name;
	}

	/**
	 * Returns the string representation of this resource, which is its name.
	 * 
	 * @see java.lang.Object#toString()
	 */
	@Override
	public String toString() {
		return name;
	}

}