//----------------------------------------------------------------------------
// Copyright (C) 2011  Ingrid Nunes
// 
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later version.
// 
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.
// 
// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
// 
// To contact the authors:
// http://inf.ufrgs.br/prosoft/bdi4jade/
//
//----------------------------------------------------------------------------

package bdi4jade.examples.interactionprotocol;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.Map;

import bdi4jade.annotation.Belief;
import bdi4jade.core.BDIAgent;
import bdi4jade.examples.interactionprotocol.domain.Component;
import bdi4jade.examples.interactionprotocol.domain.Service;
import bdi4jade.examples.interactionprotocol.domain.predicate.Abnormal;
import bdi4jade.examples.interactionprotocol.domain.predicate.AnomalousCommunication;
import bdi4jade.examples.interactionprotocol.domain.predicate.AnomalousComponent;
import bdi4jade.examples.interactionprotocol.domain.predicate.AnomalousLink;
import bdi4jade.examples.interactionprotocol.goal.ChangeServiceProviderGoal;
import bdi4jade.examples.interactionprotocol.goal.ProvideAnomalousProbabilityGoal;
import bdi4jade.examples.interactionprotocol.goal.ProvideServiceGoal;
import bdi4jade.examples.interactionprotocol.goal.RequestServiceGoal;
import bdi4jade.examples.interactionprotocol.goal.VerifyAbnormalBehaviourGoal;
import bdi4jade.examples.interactionprotocol.goal.VerifySuspiciousComponentGoal;
import bdi4jade.examples.interactionprotocol.plan.ChangeServiceProviderPlanBody;
import bdi4jade.examples.interactionprotocol.plan.NormalizeCommunicationPlanBody;
import bdi4jade.examples.interactionprotocol.plan.NotifyAnomalousComponentPlanBody;
import bdi4jade.examples.interactionprotocol.plan.ProvideAnomalousProbabilityPlanBody;
import bdi4jade.examples.interactionprotocol.plan.ProvideServicePlanBody;
import bdi4jade.examples.interactionprotocol.plan.RepairLinkPlanBody;
import bdi4jade.examples.interactionprotocol.plan.RequestServicePlanBody;
import bdi4jade.examples.interactionprotocol.plan.SelfHealingPlanBody;
import bdi4jade.examples.interactionprotocol.plan.VerifyInternalOrExternalCausePlanBody;
import bdi4jade.examples.interactionprotocol.plan.VerifySuspiciousComponentPlanBody;
import bdi4jade.extension.remediation.RemediationCapability;
import bdi4jade.goal.GoalTemplateFactory;
import bdi4jade.plan.DefaultPlan;
import bdi4jade.plan.Plan;
import bdi4jade.reasoning.DefaultBeliefRevisionStrategy;
import jade.lang.acl.ACLMessage;
import jade.lang.acl.MessageTemplate;

/**
 * @author jgfaccin
 *
 */
public class CauseAnalysisCapability extends RemediationCapability {

	private static final long serialVersionUID = 1643352181891539109L;

	// Beliefs
	@Belief
	private Map<Service, ArrayList<Service>> myServices;
	@Belief
	private Map<Service, Integer> serviceCost;
	@Belief
	private Map<Service, Component> currentProviders;
	@Belief
	private Map<String, Integer> knownAgents;
	@Belief
	private ArrayList<Component> unavailableProviders;
	@Belief
	private Boolean failure;

	// Plans
	@bdi4jade.annotation.Plan
	private Plan verifyInternalOrExternalCausePlan;
	@bdi4jade.annotation.Plan
	private Plan verifySuspiciousComponentPlan;
	@bdi4jade.annotation.Plan
	private Plan selfHealingPlan;
	@bdi4jade.annotation.Plan
	private Plan normalizeCommunicationPlan;
	@bdi4jade.annotation.Plan
	private Plan changeServiceProviderPlan;
	@bdi4jade.annotation.Plan
	private Plan notifyAnomalyPlan;
	@bdi4jade.annotation.Plan
	private Plan repairLinkPlan;
	@bdi4jade.annotation.Plan
	private Plan provideServicePlan;
	@bdi4jade.annotation.Plan
	private Plan requestServicePlan;
	@bdi4jade.annotation.Plan
	private Plan provideAnomalousProbabilityPlan;

	public CauseAnalysisCapability(BDIAgent agent, Map<Service, ArrayList<Service>> myServices,
			HashMap<String, Integer> knownAgents) {
		super(agent);
		this.myServices = myServices;
		this.serviceCost = new HashMap<Service, Integer>();
		this.knownAgents = knownAgents;
		this.unavailableProviders = new ArrayList<>();
		this.failure = false;
		this.currentProviders = new HashMap<Service, Component>();
		initializePlans();
		setBeliefRevisionStrategy(new MyBeliefRevisionStrategy());
	}

	public ArrayList<Service> getMyServices() {
		ArrayList<Service> services = new ArrayList<Service>(this.myServices.keySet());
		return services;
	}

	public ArrayList<Service> getMyServiceDependences(Service service) {
		if (this.myServices.containsKey(service)) {
			return this.myServices.get(service);
		}
		return new ArrayList<Service>();
	}

	public void setCurrentProviders(Map<Service, Component> currentProviders) {
		this.currentProviders = currentProviders;
	}

	public void setServiceCost(Service service, Integer cost) {
		this.serviceCost.put(service, cost);
	}

	public Map<Service, Integer> getServiceCost() {
		return this.serviceCost;
	}

	private void initializePlans() {
		this.verifyInternalOrExternalCausePlan = new DefaultPlan(VerifyAbnormalBehaviourGoal.class,
				VerifyInternalOrExternalCausePlanBody.class);
		this.selfHealingPlan = new DefaultPlan(GoalTemplateFactory.hasBeliefOfTypeWithValue(Abnormal.class, false),
				SelfHealingPlanBody.class);
		this.normalizeCommunicationPlan = new DefaultPlan(
				GoalTemplateFactory.hasBeliefOfTypeWithValue(AnomalousCommunication.class, false),
				NormalizeCommunicationPlanBody.class);
		this.changeServiceProviderPlan = new DefaultPlan(ChangeServiceProviderGoal.class,
				ChangeServiceProviderPlanBody.class);
		this.verifySuspiciousComponentPlan = new DefaultPlan(VerifySuspiciousComponentGoal.class,
				VerifySuspiciousComponentPlanBody.class);
		this.repairLinkPlan = new DefaultPlan(GoalTemplateFactory.hasBeliefOfTypeWithValue(AnomalousLink.class, false),
				RepairLinkPlanBody.class);
		this.notifyAnomalyPlan = new DefaultPlan(
				GoalTemplateFactory.hasBeliefOfTypeWithValue(AnomalousComponent.class, false),
				NotifyAnomalousComponentPlanBody.class);
		this.provideServicePlan = new DefaultPlan(ProvideServiceGoal.class, ProvideServicePlanBody.class);
		this.requestServicePlan = new DefaultPlan(GoalTemplateFactory.goalOfType(RequestServiceGoal.class),
				RequestServicePlanBody.class);
		this.provideAnomalousProbabilityPlan = new DefaultPlan(ProvideAnomalousProbabilityGoal.class,
				ProvideAnomalousProbabilityPlanBody.class);
	}

	class MyBeliefRevisionStrategy extends DefaultBeliefRevisionStrategy {

		private MessageTemplate mtNotification;
		private MessageTemplate mtServiceRequest;
		private MessageTemplate mtProbabilityRequest;

		@Override
		public void reviewBeliefs() {
			super.reviewBeliefs();

			this.mtNotification = MessageTemplate.MatchPerformative(ACLMessage.INFORM_REF);
			ACLMessage notificationMsg = getMyAgent().receive(mtNotification);
			if (notificationMsg != null) {
				getMyAgent().addGoal(new VerifyAbnormalBehaviourGoal(notificationMsg));
			}

			this.mtServiceRequest = MessageTemplate.MatchPerformative(ACLMessage.REQUEST);
			ACLMessage serviceRequestMsg = getMyAgent().receive(mtServiceRequest);
			if (serviceRequestMsg != null) {
				Service service = new Service(serviceRequestMsg.getContent());
				if (myServices.containsKey(service)) {
					getMyAgent().addGoal(new ProvideServiceGoal(serviceRequestMsg));
				}
			}

			this.mtProbabilityRequest = MessageTemplate.MatchPerformative(ACLMessage.QUERY_IF);
			ACLMessage probabilityRequestMsg = getMyAgent().receive(mtProbabilityRequest);
			if (probabilityRequestMsg != null) {
				getMyAgent().addGoal(new ProvideAnomalousProbabilityGoal(probabilityRequestMsg));
			}
		}

	}

}
