//----------------------------------------------------------------------------
// Copyright (C) 2011  Ingrid Nunes
// 
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later version.
// 
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.
// 
// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
// 
// To contact the authors:
// http://inf.ufrgs.br/prosoft/bdi4jade/
//
//----------------------------------------------------------------------------

package bdi4jade.examples.interactionprotocol.dao;

import java.util.ArrayList;

import org.bson.Document;

import com.mongodb.client.FindIterable;
import com.mongodb.client.MongoCollection;
import com.mongodb.client.MongoDatabase;
import com.mongodb.client.model.Filters;
import com.mongodb.client.model.Sorts;

import bdi4jade.examples.interactionprotocol.Trace;
import bdi4jade.examples.interactionprotocol.TraceDatabase;

/**
 * @author jgfaccin
 *
 */
public class TraceDAO {

	private static final String SENDER = "sender";
	private static final String RECEIVER = "receiver";
	private static final String SERVICE = "service";
	private static final String CID = "cid";
	private static final String PARENT_CID = "parentCid";
	private static final String SENT_AT = "sentAt";
	private static final String RECEIVED_AT = "receivedAt";
	private static final String VALUE = "value";
	private static final String TRACES_COLLECTION = "traces";

	private MongoDatabase db;

	public TraceDAO(String dbName) {
		this.db = TraceDatabase.getInstance().getDatabase(dbName);
	}

	public void dropCollection() {
		MongoCollection<Document> traces = getCollection();
		traces.drop();
	}

	public void addTrace(Trace trace) {
		MongoCollection<Document> traces = getCollection();
		Document item = new Document(TraceDAO.SENDER, trace.getSender()).append(TraceDAO.RECEIVER, trace.getReceiver())
				.append(TraceDAO.SERVICE, trace.getService()).append(TraceDAO.CID, trace.getCid())
				.append(TraceDAO.PARENT_CID, trace.getParentCid()).append(TraceDAO.SENT_AT, trace.getSentAt());
		traces.insertOne(item);
	}

	public void updateTrace(Trace trace) {
		MongoCollection<Document> traces = getCollection();
		traces.updateOne(
				Filters.and(Filters.eq(TraceDAO.SENDER, trace.getSender()),
						Filters.eq(TraceDAO.RECEIVER, trace.getReceiver()), Filters.eq(TraceDAO.CID, trace.getCid())),
				new Document("$set", new Document(TraceDAO.RECEIVED_AT, trace.getReceivedAt()).append(TraceDAO.VALUE,
						trace.getValue())));
	}

	public ArrayList<Trace> getTracesOfProvidedByBefore(String service, String provider, String parentCid) {
		MongoCollection<Document> traces = getCollection();
		Document doc = traces.find(Filters.and(Filters.eq(TraceDAO.PARENT_CID, parentCid), Filters.eq(TraceDAO.SERVICE, service),
				Filters.eq(TraceDAO.RECEIVER, provider))).first();
		Long receivedAt = doc.getLong(TraceDAO.RECEIVED_AT);

		FindIterable<Document> findIterable = traces.find(Filters.and(Filters.eq(TraceDAO.SERVICE, service),
				Filters.eq(TraceDAO.RECEIVER, provider), Filters.lte(TraceDAO.RECEIVED_AT, receivedAt)));
		
		ArrayList<Trace> traceList = new ArrayList<Trace>();
		for (Document document : findIterable) {
			traceList.add(getTraceInstance(document));
		}
		
		return traceList;
	}

	public ArrayList<Trace> getTracesByCID(String cid) {
		MongoCollection<Document> traces = getCollection();
		FindIterable<Document> findIterable = traces.find(Filters.eq(TraceDAO.CID, cid));
		ArrayList<Trace> traceList = new ArrayList<Trace>();
		for (Document document : findIterable) {
			traceList.add(getTraceInstance(document));
		}
		return traceList;
	}

	public ArrayList<Trace> getTracesByParentCID(String parentCid) {
		MongoCollection<Document> traces = getCollection();
		FindIterable<Document> findIterable = traces.find(Filters.eq(TraceDAO.PARENT_CID, parentCid));
		ArrayList<Trace> traceList = new ArrayList<Trace>();
		for (Document document : findIterable) {
			traceList.add(getTraceInstance(document));
		}
		return traceList;
	}

	public ArrayList<Trace> getTracesByServiceAndProvider(String service, String provider) {
		MongoCollection<Document> traces = getCollection();
		FindIterable<Document> findIterable = traces.find(Filters.and(Filters.eq(TraceDAO.SERVICE, service),
				Filters.eq(TraceDAO.RECEIVER, provider), Filters.exists(TraceDAO.RECEIVED_AT)))
				.sort(Sorts.ascending(TraceDAO.RECEIVED_AT));
		ArrayList<Trace> traceList = new ArrayList<Trace>();
		for (Document document : findIterable) {
			traceList.add(getTraceInstance(document));
		}
		return traceList;
	}

	public ArrayList<Trace> getTracesByCIDAndProvider(String cid, String provider) {
		MongoCollection<Document> traces = getCollection();
		FindIterable<Document> findIterable = traces
				.find(Filters.and(Filters.eq(TraceDAO.CID, cid), Filters.eq(TraceDAO.RECEIVER, provider)));
		ArrayList<Trace> traceList = new ArrayList<Trace>();
		for (Document document : findIterable) {
			traceList.add(getTraceInstance(document));
		}
		return traceList;
	}

	public ArrayList<Trace> getTracesByParentCIDAndProvider(String parentCid, String provider) {
		MongoCollection<Document> traces = getCollection();
		FindIterable<Document> findIterable = traces
				.find(Filters.and(Filters.eq(TraceDAO.PARENT_CID, parentCid), Filters.eq(TraceDAO.RECEIVER, provider)));
		ArrayList<Trace> traceList = new ArrayList<Trace>();
		for (Document document : findIterable) {
			traceList.add(getTraceInstance(document));
		}
		return traceList;
	}

	public String getCIDByParentCIDAndProviderAndService(String parentCid, String provider, String service) {
		MongoCollection<Document> traces = getCollection();
		FindIterable<Document> findIterable = traces.find(Filters.and(Filters.eq(TraceDAO.PARENT_CID, parentCid),
				Filters.eq(TraceDAO.RECEIVER, provider), Filters.eq(TraceDAO.SERVICE, service)));
		Document document = findIterable.first();
		return (String) document.get(TraceDAO.CID);
	}

	private MongoCollection<Document> getCollection() {
		return this.db.getCollection(TraceDAO.TRACES_COLLECTION);
	}

	private Trace getTraceInstance(Document document) {
		Trace trace = new Trace(document.getString(TraceDAO.SENDER), document.getString(TraceDAO.RECEIVER),
				document.getString(TraceDAO.SERVICE), document.getString(TraceDAO.CID),
				document.getString(TraceDAO.PARENT_CID));
		trace.setSentAt(document.getLong(TraceDAO.SENT_AT));
		trace.setReceivedAt(document.getLong(TraceDAO.RECEIVED_AT));
		trace.setValue(document.getString(TraceDAO.VALUE));
		return trace;
	}

}
