//----------------------------------------------------------------------------
// Copyright (C) 2011  Ingrid Nunes
// 
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later version.
// 
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.
// 
// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
// 
// To contact the authors:
// http://inf.ufrgs.br/prosoft/bdi4jade/
//
//----------------------------------------------------------------------------

package bdi4jade.examples.interactionprotocol.experiment;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Random;

import org.apache.log4j.PropertyConfigurator;

import bdi4jade.core.SingleCapabilityAgent;
import bdi4jade.event.GoalEvent;
import bdi4jade.event.GoalListener;
import bdi4jade.examples.interactionprotocol.CauseAnalysisCapability;
import bdi4jade.examples.interactionprotocol.MonitorMetricsCapability;
import bdi4jade.examples.interactionprotocol.ServiceProviderAgent;
import bdi4jade.examples.interactionprotocol.dao.TraceDAO;
import bdi4jade.examples.interactionprotocol.domain.Component;
import bdi4jade.examples.interactionprotocol.domain.Link;
import bdi4jade.examples.interactionprotocol.domain.LinkMonitor;
import bdi4jade.examples.interactionprotocol.domain.Service;
import bdi4jade.examples.interactionprotocol.goal.MonitorMetricsGoal;
import bdi4jade.examples.interactionprotocol.goal.RequestServiceGoal;
import jade.BootProfileImpl;
import jade.core.ProfileImpl;
import jade.wrapper.AgentContainer;
import jade.wrapper.AgentController;
import jade.wrapper.PlatformController;

/**
 * @author jgfaccin
 *
 */
public class ExperimentRunner1 implements GoalListener {

	public static final int ITERATIONS = 100;

	private ProfileImpl bootProfile;
	private jade.core.Runtime runtime;
	private PlatformController controller;

	private ArrayList<ServiceProviderAgent> providerAgents;
	private ArrayList<ServiceProviderAgent> supportingAgents;
	private SingleCapabilityAgent client;

	private int failurePoint;
	private int iteration;

	public ExperimentRunner1() {
		this.failurePoint = 50;
		this.iteration = 0;
		List<String> params = new ArrayList<String>();
		params.add("-gui");
		params.add("-detect-main:false");

		this.bootProfile = new BootProfileImpl(params.toArray(new String[0]));
		this.runtime = jade.core.Runtime.instance();
		this.controller = runtime.createMainContainer(bootProfile);

		this.providerAgents = initializeProviderAgents();
		this.supportingAgents = initializeSupportingAgents();
		initializeClientAgent();

		HashMap<Service, ArrayList<Service>> serviceDependences = initializeServiceDependenceList();

		setAgentCapabilitiesAndServices(this.providerAgents, this.supportingAgents, serviceDependences);

		for (ServiceProviderAgent providers : this.providerAgents) {
			providers.register();
		}

		setInitialCurrentProviders(this.providerAgents);
	}

	public static void main(String[] args) {
		PropertyConfigurator.configure(ExperimentRunner1.class.getResource("log4j.properties"));
		ExperimentRunner1 runner = new ExperimentRunner1();
		runner.run();
	}

	public void run() {
		try {
			Thread.sleep(2000);
			for (ServiceProviderAgent supporter : this.supportingAgents) {
				supporter.addGoal(new RequestServiceGoal(new Service("SERVICE2"), null), this);
				Thread.sleep(1000);
			}
			client.addGoal(new MonitorMetricsGoal(), this);

			this.iteration++;
		} catch (Exception e) {
			e.printStackTrace();
		}
	}

	@SuppressWarnings("unchecked")
	@Override
	public void goalPerformed(GoalEvent event) {
		if (event.getStatus().isFinished() && event.getGoal() instanceof MonitorMetricsGoal) {
			if (iteration < ITERATIONS) {
				if (iteration == failurePoint) {

					/*
					 * ServiceProviderAgent agent = providerAgents.get(4); Belief<String, Boolean>
					 * failure = (Belief<String, Boolean>) agent.getCapability().getBeliefBase()
					 * .getBelief("failure"); failure.setValue(true);
					 * System.out.println("Setting failure to " + agent.getLocalName());
					 */

					Link link = new Link(new Component("AGENT0"), new Component("AGENT4"), new Service("SERVICE2"));
					LinkMonitor.getInstance().getAnomalousLinks().add(link);
					System.out.println("Setting failure to link: " + link);

				}

				run();
			} else {
				System.out.println("Iterations finished!!");
			}
		}
	}

	private ArrayList<ServiceProviderAgent> initializeProviderAgents() {
		System.out.println("Initializing 31 service providers...");
		ArrayList<ServiceProviderAgent> agents = new ArrayList<>();

		try {
			for (int i = 0; i < 31; i++) {
				ServiceProviderAgent agent = new ServiceProviderAgent("AGENT" + i);
				agents.add(agent);
				AgentController ac = ((AgentContainer) controller).acceptNewAgent(agent.getCustomName(), agent);
				ac.start();
				setupDB(agent);
			}
		} catch (Exception e) {
			e.printStackTrace();
		}
		System.out.println("Provider agents initialized!");
		return agents;
	}

	private void initializeClientAgent() {
		System.out.println("Initializing client agent...");
		this.client = new SingleCapabilityAgent();
		try {
			AgentController ac = ((AgentContainer) controller).acceptNewAgent("CLIENT", this.client);
			ac.start();
		} catch (Exception e) {
			e.printStackTrace();
		}
		System.out.println("Client agent initalized!");
	}

	private ArrayList<ServiceProviderAgent> initializeSupportingAgents() {
		System.out.println("Initializing 5 supporting agents...");
		ArrayList<ServiceProviderAgent> agents = new ArrayList<>();
		try {
			for (int i = 1; i <= 5; i++) {
				ServiceProviderAgent agent = new ServiceProviderAgent("SUPPORT" + i);
				agents.add(agent);
				AgentController ac = ((AgentContainer) controller).acceptNewAgent(agent.getCustomName(), agent);
				ac.start();
				setupDB(agent);
			}
		} catch (Exception e) {
			e.printStackTrace();
		}
		System.out.println("Supporting agents initialized!");
		return agents;
	}

	private HashMap<String, HashMap<String, Integer>> initializeAdjacencyList(List<ServiceProviderAgent> providers,
			List<ServiceProviderAgent> supporters) {
		System.out.println("Initializing adjacency list for " + (providers.size() + supporters.size()) + " agents...");
		Random random = new Random(10);
		List<ServiceProviderAgent> agents = new ArrayList<ServiceProviderAgent>();
		agents.addAll(providers);
		agents.addAll(supporters);

		HashMap<String, HashMap<String, Integer>> adjacencyList = new HashMap<>();

		for (SingleCapabilityAgent agent : agents) {

			HashMap<String, Integer> neighbours = new HashMap<>();
			String agentLocalName = agent.getLocalName();

			for (SingleCapabilityAgent neighbour : agents) {
				String neighbourLocalName = neighbour.getLocalName();
				if (!neighbourLocalName.equals(agentLocalName)) {
					if (adjacencyList.containsKey(neighbourLocalName)) {
						neighbours.put(neighbourLocalName, adjacencyList.get(neighbourLocalName).get(agentLocalName));
					} else {
						neighbours.put(neighbourLocalName, random.nextInt(10));
					}
				}
			}
			adjacencyList.put(agentLocalName, neighbours);
		}
		System.out.println("Adjacency list initialization DONE!");
		return adjacencyList;
	}

	private HashMap<Service, ArrayList<Service>> initializeServiceDependenceList() {
		System.out.println("Creating service dependence list for 10 services...");
		HashMap<Service, ArrayList<Service>> dependenceList = new HashMap<>();
		for (int i = 0; i < 3; i++) {
			Service service = new Service("SERVICE" + i);
			ArrayList<Service> serviceDependences = new ArrayList<Service>();
			serviceDependences.add(new Service("SERVICE" + (2 * i + 1)));
			serviceDependences.add(new Service("SERVICE" + (2 * i + 2)));
			dependenceList.put(service, serviceDependences);
		}

		for (int i = 3, j = 1; i < 7; i++, j--) {
			Service service = new Service("SERVICE" + i);
			ArrayList<Service> serviceDependences = new ArrayList<Service>();
			serviceDependences.add(new Service("SERVICE" + (2 * i + j)));
			dependenceList.put(service, serviceDependences);
		}
		/*
		 * for (Service service : dependenceList.keySet()) {
		 * System.out.println(service.getId()); System.out.println("\t" +
		 * dependenceList.get(service)); }
		 */

		return dependenceList;
	}

	private void setAgentCapabilitiesAndServices(ArrayList<ServiceProviderAgent> providers,
			ArrayList<ServiceProviderAgent> supporters, HashMap<Service, ArrayList<Service>> serviceDependences) {

		setClientAgentCapability();

		HashMap<String, HashMap<String, Integer>> adjacencyList = initializeAdjacencyList(providers, supporters);

		System.out.println("Setting provider agent capabilities...");

		ServiceProviderAgent a0 = providers.get(0);
		HashMap<Service, ArrayList<Service>> myServices = new HashMap<Service, ArrayList<Service>>();
		Service s0 = new Service("SERVICE0");
		myServices.put(s0, serviceDependences.get(s0));
		CauseAnalysisCapability capability = new CauseAnalysisCapability(a0, myServices,
				adjacencyList.get(a0.getLocalName()));
		capability.setServiceCost(s0, 0);
		a0.setCapability(capability);

		int agentCounter = 1;

		for (int i = 1; i < 11; i++) {
			for (int j = 0; j < 3; j++) {
				ServiceProviderAgent provider = providers.get(agentCounter);
				Service service = new Service("SERVICE" + i);

				myServices = new HashMap<Service, ArrayList<Service>>();
				ArrayList<Service> dependences = new ArrayList<Service>();
				if (serviceDependences.get(service) != null) {
					dependences = serviceDependences.get(service);
				}
				myServices.put(service, dependences);
				capability = new CauseAnalysisCapability(provider, myServices,
						adjacencyList.get(provider.getLocalName()));
				capability.setServiceCost(service, agentCounter);

				provider.setCapability(capability);
				agentCounter++;
			}
		}
		System.out.println("Provider agent capabilities initialized!");
		System.out.println("Setting supporting agent capabilities...");

		myServices = new HashMap<Service, ArrayList<Service>>();

		myServices.put(s0, serviceDependences.get(s0));

		HashMap<Service, Component> currentProviders = new HashMap<Service, Component>();
		currentProviders.put(new Service("SERVICE1"), new Component("AGENT1"));
		currentProviders.put(new Service("SERVICE2"), new Component("AGENT4"));

		for (ServiceProviderAgent supporter : supporters) {
			CauseAnalysisCapability cap = new CauseAnalysisCapability(supporter, myServices,
					adjacencyList.get(supporter.getLocalName()));
			cap.setCurrentProviders(currentProviders);
			supporter.setCapability(cap);
		}

		System.out.println("Supporting agent capabilities initialized!");

	}

	private void setClientAgentCapability() {
		System.out.println("Setting client agent capability...");
		this.client.setCapability(new MonitorMetricsCapability());
		System.out.println("Client agent capability initialized!");
	}

	private void setInitialCurrentProviders(ArrayList<ServiceProviderAgent> providers) {
		ServiceProviderAgent agent = providers.get(0);
		CauseAnalysisCapability capability = (CauseAnalysisCapability) agent.getCapability();
		HashMap<Service, Component> currentProviders = new HashMap<Service, Component>();
		currentProviders.put(new Service("SERVICE1"), new Component("AGENT1"));
		currentProviders.put(new Service("SERVICE2"), new Component("AGENT4"));
		capability.setCurrentProviders(currentProviders);

		agent = providers.get(1);
		capability = (CauseAnalysisCapability) agent.getCapability();
		currentProviders = new HashMap<Service, Component>();
		currentProviders.put(new Service("SERVICE3"), new Component("AGENT7"));
		currentProviders.put(new Service("SERVICE4"), new Component("AGENT10"));
		capability.setCurrentProviders(currentProviders);

		agent = providers.get(2);
		capability = (CauseAnalysisCapability) agent.getCapability();
		currentProviders = new HashMap<Service, Component>();
		currentProviders.put(new Service("SERVICE3"), new Component("AGENT8"));
		currentProviders.put(new Service("SERVICE4"), new Component("AGENT11"));
		capability.setCurrentProviders(currentProviders);

		agent = providers.get(3);
		capability = (CauseAnalysisCapability) agent.getCapability();
		currentProviders = new HashMap<Service, Component>();
		currentProviders.put(new Service("SERVICE3"), new Component("AGENT9"));
		currentProviders.put(new Service("SERVICE4"), new Component("AGENT12"));
		capability.setCurrentProviders(currentProviders);

		agent = providers.get(4);
		capability = (CauseAnalysisCapability) agent.getCapability();
		currentProviders = new HashMap<Service, Component>();
		currentProviders.put(new Service("SERVICE5"), new Component("AGENT13"));
		currentProviders.put(new Service("SERVICE6"), new Component("AGENT16"));
		capability.setCurrentProviders(currentProviders);

		agent = providers.get(5);
		capability = (CauseAnalysisCapability) agent.getCapability();
		currentProviders = new HashMap<Service, Component>();
		currentProviders.put(new Service("SERVICE5"), new Component("AGENT14"));
		currentProviders.put(new Service("SERVICE6"), new Component("AGENT17"));
		capability.setCurrentProviders(currentProviders);

		agent = providers.get(6);
		capability = (CauseAnalysisCapability) agent.getCapability();
		currentProviders = new HashMap<Service, Component>();
		currentProviders.put(new Service("SERVICE5"), new Component("AGENT15"));
		currentProviders.put(new Service("SERVICE6"), new Component("AGENT18"));
		capability.setCurrentProviders(currentProviders);

		agent = providers.get(7);
		capability = (CauseAnalysisCapability) agent.getCapability();
		currentProviders = new HashMap<Service, Component>();
		currentProviders.put(new Service("SERVICE7"), new Component("AGENT19"));
		capability.setCurrentProviders(currentProviders);

		agent = providers.get(8);
		capability = (CauseAnalysisCapability) agent.getCapability();
		currentProviders = new HashMap<Service, Component>();
		currentProviders.put(new Service("SERVICE7"), new Component("AGENT20"));
		capability.setCurrentProviders(currentProviders);

		agent = providers.get(9);
		capability = (CauseAnalysisCapability) agent.getCapability();
		currentProviders = new HashMap<Service, Component>();
		currentProviders.put(new Service("SERVICE7"), new Component("AGENT21"));
		capability.setCurrentProviders(currentProviders);

		agent = providers.get(10);
		capability = (CauseAnalysisCapability) agent.getCapability();
		currentProviders = new HashMap<Service, Component>();
		currentProviders.put(new Service("SERVICE8"), new Component("AGENT22"));
		capability.setCurrentProviders(currentProviders);

		agent = providers.get(11);
		capability = (CauseAnalysisCapability) agent.getCapability();
		currentProviders = new HashMap<Service, Component>();
		currentProviders.put(new Service("SERVICE8"), new Component("AGENT23"));
		capability.setCurrentProviders(currentProviders);

		agent = providers.get(12);
		capability = (CauseAnalysisCapability) agent.getCapability();
		currentProviders = new HashMap<Service, Component>();
		currentProviders.put(new Service("SERVICE8"), new Component("AGENT24"));
		capability.setCurrentProviders(currentProviders);

		agent = providers.get(13);
		capability = (CauseAnalysisCapability) agent.getCapability();
		currentProviders = new HashMap<Service, Component>();
		currentProviders.put(new Service("SERVICE9"), new Component("AGENT25"));
		capability.setCurrentProviders(currentProviders);

		agent = providers.get(14);
		capability = (CauseAnalysisCapability) agent.getCapability();
		currentProviders = new HashMap<Service, Component>();
		currentProviders.put(new Service("SERVICE9"), new Component("AGENT26"));
		capability.setCurrentProviders(currentProviders);

		agent = providers.get(15);
		capability = (CauseAnalysisCapability) agent.getCapability();
		currentProviders = new HashMap<Service, Component>();
		currentProviders.put(new Service("SERVICE9"), new Component("AGENT27"));
		capability.setCurrentProviders(currentProviders);

		agent = providers.get(16);
		capability = (CauseAnalysisCapability) agent.getCapability();
		currentProviders = new HashMap<Service, Component>();
		currentProviders.put(new Service("SERVICE10"), new Component("AGENT28"));
		capability.setCurrentProviders(currentProviders);

		agent = providers.get(17);
		capability = (CauseAnalysisCapability) agent.getCapability();
		currentProviders = new HashMap<Service, Component>();
		currentProviders.put(new Service("SERVICE10"), new Component("AGENT29"));
		capability.setCurrentProviders(currentProviders);

		agent = providers.get(18);
		capability = (CauseAnalysisCapability) agent.getCapability();
		currentProviders = new HashMap<Service, Component>();
		currentProviders.put(new Service("SERVICE10"), new Component("AGENT30"));
		capability.setCurrentProviders(currentProviders);
	}

	private void setupDB(ServiceProviderAgent agent) {
		TraceDAO dao = new TraceDAO(agent.getAID().getLocalName());
		dao.dropCollection();
	}

}
