//----------------------------------------------------------------------------
// Copyright (C) 2011  Ingrid Nunes
// 
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later version.
// 
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.
// 
// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
// 
// To contact the authors:
// http://inf.ufrgs.br/prosoft/bdi4jade/
//
//----------------------------------------------------------------------------

package bdi4jade.examples.interactionprotocol.experiment;

import java.time.Duration;
import java.time.Instant;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Random;

import org.apache.log4j.PropertyConfigurator;

import bdi4jade.belief.Belief;
import bdi4jade.core.SingleCapabilityAgent;
import bdi4jade.event.GoalEvent;
import bdi4jade.event.GoalListener;
import bdi4jade.examples.interactionprotocol.CauseAnalysisCapability;
import bdi4jade.examples.interactionprotocol.MonitorMetricsCapability;
import bdi4jade.examples.interactionprotocol.ServiceProviderAgent;
import bdi4jade.examples.interactionprotocol.dao.TraceDAO;
import bdi4jade.examples.interactionprotocol.domain.Component;
import bdi4jade.examples.interactionprotocol.domain.Link;
import bdi4jade.examples.interactionprotocol.domain.LinkMonitor;
import bdi4jade.examples.interactionprotocol.domain.Service;
import bdi4jade.examples.interactionprotocol.goal.MonitorMetricsGoal;
import bdi4jade.examples.interactionprotocol.goal.RequestServiceGoal;
import jade.BootProfileImpl;
import jade.core.ProfileImpl;
import jade.wrapper.AgentContainer;
import jade.wrapper.AgentController;
import jade.wrapper.PlatformController;

/**
 * @author jgfaccin
 *
 */
public class InteractiveRunner implements GoalListener {

	public static final int ITERATIONS = 120;

	private ProfileImpl bootProfile;
	private jade.core.Runtime runtime;
	private PlatformController controller;

	private ArrayList<ServiceProviderAgent> providerAgents;
	private ArrayList<ServiceProviderAgent> supportingAgents;
	private SingleCapabilityAgent client;

	private int providerFailure;
	private int linkFailure;
	private int providerAndLinkFailure;
	private int iteration;
	private Instant startedAt;

	public InteractiveRunner() {
		this.startedAt = Instant.now();
		this.providerFailure = 30;
		this.linkFailure = 60;
		this.providerAndLinkFailure = 90;
		this.iteration = 0;
		List<String> params = new ArrayList<String>();
		params.add("-gui");
		params.add("-detect-main:false");

		this.bootProfile = new BootProfileImpl(params.toArray(new String[0]));
		this.runtime = jade.core.Runtime.instance();
		this.controller = runtime.createMainContainer(bootProfile);

		this.providerAgents = createProviderAgents();
		this.supportingAgents = createSupportingAgents();
		this.client = new SingleCapabilityAgent(new MonitorMetricsCapability());

		try {
			AgentController ac = ((AgentContainer) controller).acceptNewAgent("CLIENT", this.client);
			ac.start();

			ArrayList<ServiceProviderAgent> agents = new ArrayList<>();
			agents.addAll(providerAgents);
			agents.addAll(supportingAgents);
			for (ServiceProviderAgent agent : agents) {
				ac = ((AgentContainer) controller).acceptNewAgent(agent.getCustomName(), agent);
				ac.start();
				setupDB(agent);
			}
		} catch (Exception e) {
			e.printStackTrace();
		}

		for (ServiceProviderAgent providers : this.providerAgents) {
			providers.register();
		}
	}

	public static void main(String[] args) {
		PropertyConfigurator.configure(InteractiveRunner.class.getResource("log4j.properties"));
		InteractiveRunner runner = new InteractiveRunner();
		runner.run();
	}

	public void run() {
		try {
			Thread.sleep(2000);
			System.out.println("#ITERATION: " + this.iteration);
			for (int i = 0; i < 3; i++) {
				ServiceProviderAgent supporter = this.supportingAgents.get(i);
				supporter.addGoal(new RequestServiceGoal(new Service("B"), null), this);
				supporter.addGoal(new RequestServiceGoal(new Service("D"), null), this);
				supporter.addGoal(new RequestServiceGoal(new Service("E"), null), this);
				supporter.addGoal(new RequestServiceGoal(new Service("J"), null), this);
				Thread.sleep(500);
			}

			for (int i = 3; i < 6; i++) {
				ServiceProviderAgent supporter = this.supportingAgents.get(i);
				supporter.addGoal(new RequestServiceGoal(new Service("C"), null), this);
				supporter.addGoal(new RequestServiceGoal(new Service("G"), null), this);
				supporter.addGoal(new RequestServiceGoal(new Service("N"), null), this);
				supporter.addGoal(new RequestServiceGoal(new Service("X"), null), this);
				Thread.sleep(500);
			}
			client.addGoal(new MonitorMetricsGoal(), this);

			this.iteration++;
		} catch (Exception e) {
			e.printStackTrace();
		}
	}

	@Override
	public void goalPerformed(GoalEvent event) {
		if (event.getStatus().isFinished() && event.getGoal() instanceof MonitorMetricsGoal) {
			if (iteration < ITERATIONS) {
				if (iteration == providerFailure) {
					ServiceProviderAgent agent = this.providerAgents.get(15);
					@SuppressWarnings("unchecked")
					Belief<String, Boolean> failure = (Belief<String, Boolean>) agent.getCapability().getBeliefBase()
							.getBelief("failure");
					failure.setValue(true);
					System.out.println("Setting failure to " + agent.getLocalName());
				} else if (iteration == linkFailure) {
					Link link = new Link(new Component("AGENT21"), new Component("AGENT30"), new Service("X"));
					LinkMonitor.getInstance().getAnomalousLinks().add(link);
					System.out.println("Setting failure to link: " + link);
				} else if (iteration == providerAndLinkFailure) {
					ServiceProviderAgent agent = this.providerAgents.get(6);
					@SuppressWarnings("unchecked")
					Belief<String, Boolean> failure = (Belief<String, Boolean>) agent.getCapability().getBeliefBase()
							.getBelief("failure");
					failure.setValue(true);
					Link link = new Link(new Component("AGENT02"), new Component("AGENT07"), new Service("D"));
					LinkMonitor.getInstance().getAnomalousLinks().add(link);
					System.out.println("Setting failure to " + agent.getLocalName() + " and link " + link);
				}
				run();
			} else {
				Instant finishedAt = Instant.now();
				System.out.println("Iterations finished in " + Duration.between(startedAt, finishedAt).getSeconds());
			}
		}
	}

	public HashMap<String, Integer> getAdjacencyList() {
		HashMap<String, Integer> adjacencyList = new HashMap<>();
		Random random = new Random(System.currentTimeMillis());

		for (int i = 1; i <= 6; i++) {
			adjacencyList.put("SUPPORT" + i, random.nextInt(10));
		}
		return adjacencyList;
	}

	public ArrayList<ServiceProviderAgent> createProviderAgents() {
		ArrayList<ServiceProviderAgent> agents = new ArrayList<>();

		ServiceProviderAgent agent = new ServiceProviderAgent("AGENT01");
		HashMap<Service, ArrayList<Service>> agentServices = new HashMap<>();
		ArrayList<Service> serviceDependences = new ArrayList<>();
		serviceDependences.add(new Service("B"));
		serviceDependences.add(new Service("C"));
		agentServices.put(new Service("A"), serviceDependences);
		HashMap<Service, Component> currentProviders = new HashMap<>();
		currentProviders.put(new Service("B"), new Component("AGENT02"));
		currentProviders.put(new Service("C"), new Component("AGENT04"));
		HashMap<String, Integer> knownNeighbours = getAdjacencyList();
		CauseAnalysisCapability capability = new CauseAnalysisCapability(agent, agentServices, knownNeighbours);
		capability.setCurrentProviders(currentProviders);
		capability.setServiceCost(new Service("A"), 1);
		agent.setCapability(capability);
		agents.add(agent);

		agent = new ServiceProviderAgent("AGENT02");
		agentServices = new HashMap<>();
		serviceDependences = new ArrayList<>();
		serviceDependences.add(new Service("D"));
		serviceDependences.add(new Service("E"));
		agentServices.put(new Service("B"), serviceDependences);
		currentProviders = new HashMap<>();
		currentProviders.put(new Service("D"), new Component("AGENT07"));
		currentProviders.put(new Service("E"), new Component("AGENT08"));
		knownNeighbours = getAdjacencyList();
		capability = new CauseAnalysisCapability(agent, agentServices, knownNeighbours);
		capability.setCurrentProviders(currentProviders);
		capability.setServiceCost(new Service("B"), 2);
		agent.setCapability(capability);
		agents.add(agent);

		agent = new ServiceProviderAgent("AGENT03");
		agentServices = new HashMap<>();
		serviceDependences = new ArrayList<>();
		serviceDependences.add(new Service("F"));
		agentServices.put(new Service("B"), serviceDependences);
		currentProviders = new HashMap<>();
		currentProviders.put(new Service("F"), new Component("AGENT10"));
		knownNeighbours = getAdjacencyList();
		capability = new CauseAnalysisCapability(agent, agentServices, knownNeighbours);
		capability.setCurrentProviders(currentProviders);
		capability.setServiceCost(new Service("B"), 5);
		agent.setCapability(capability);
		agents.add(agent);

		agent = new ServiceProviderAgent("AGENT04");
		agentServices = new HashMap<>();
		serviceDependences = new ArrayList<>();
		serviceDependences.add(new Service("G"));
		agentServices.put(new Service("C"), serviceDependences);
		currentProviders = new HashMap<>();
		currentProviders.put(new Service("G"), new Component("AGENT11"));
		knownNeighbours = getAdjacencyList();
		capability = new CauseAnalysisCapability(agent, agentServices, knownNeighbours);
		capability.setCurrentProviders(currentProviders);
		capability.setServiceCost(new Service("C"), 2);
		agent.setCapability(capability);
		agents.add(agent);

		agent = new ServiceProviderAgent("AGENT05");
		agentServices = new HashMap<>();
		serviceDependences = new ArrayList<>();
		serviceDependences.add(new Service("H"));
		agentServices.put(new Service("C"), serviceDependences);
		currentProviders = new HashMap<>();
		currentProviders.put(new Service("H"), new Component("AGENT13"));
		knownNeighbours = getAdjacencyList();
		capability = new CauseAnalysisCapability(agent, agentServices, knownNeighbours);
		capability.setCurrentProviders(currentProviders);
		capability.setServiceCost(new Service("C"), 5);
		agent.setCapability(capability);
		agents.add(agent);

		agent = new ServiceProviderAgent("AGENT06");
		agentServices = new HashMap<>();
		serviceDependences = new ArrayList<>();
		serviceDependences.add(new Service("I"));
		agentServices.put(new Service("D"), serviceDependences);
		currentProviders = new HashMap<>();
		currentProviders.put(new Service("I"), new Component("AGENT14"));
		knownNeighbours = getAdjacencyList();
		capability = new CauseAnalysisCapability(agent, agentServices, knownNeighbours);
		capability.setCurrentProviders(currentProviders);
		capability.setServiceCost(new Service("D"), 5);
		agent.setCapability(capability);
		agents.add(agent);

		agent = new ServiceProviderAgent("AGENT07");
		agentServices = new HashMap<>();
		serviceDependences = new ArrayList<>();
		agentServices.put(new Service("D"), serviceDependences);
		knownNeighbours = getAdjacencyList();
		capability = new CauseAnalysisCapability(agent, agentServices, knownNeighbours);
		capability.setServiceCost(new Service("D"), 2);
		agent.setCapability(capability);
		agents.add(agent);

		agent = new ServiceProviderAgent("AGENT08");
		agentServices = new HashMap<>();
		serviceDependences = new ArrayList<>();
		serviceDependences.add(new Service("J"));
		agentServices.put(new Service("E"), serviceDependences);
		currentProviders = new HashMap<>();
		currentProviders.put(new Service("J"), new Component("AGENT16"));
		knownNeighbours = getAdjacencyList();
		capability = new CauseAnalysisCapability(agent, agentServices, knownNeighbours);
		capability.setCurrentProviders(currentProviders);
		capability.setServiceCost(new Service("E"), 2);
		agent.setCapability(capability);
		agents.add(agent);

		agent = new ServiceProviderAgent("AGENT09");
		agentServices = new HashMap<>();
		serviceDependences = new ArrayList<>();
		serviceDependences.add(new Service("K"));
		agentServices.put(new Service("E"), serviceDependences);
		currentProviders = new HashMap<>();
		currentProviders.put(new Service("K"), new Component("AGENT17"));
		knownNeighbours = getAdjacencyList();
		capability = new CauseAnalysisCapability(agent, agentServices, knownNeighbours);
		capability.setCurrentProviders(currentProviders);
		capability.setServiceCost(new Service("E"), 5);
		agent.setCapability(capability);
		agents.add(agent);

		agent = new ServiceProviderAgent("AGENT10");
		agentServices = new HashMap<>();
		serviceDependences = new ArrayList<>();
		serviceDependences.add(new Service("L"));
		serviceDependences.add(new Service("M"));
		agentServices.put(new Service("F"), serviceDependences);
		currentProviders = new HashMap<>();
		currentProviders.put(new Service("L"), new Component("AGENT18"));
		currentProviders.put(new Service("M"), new Component("AGENT19"));
		knownNeighbours = getAdjacencyList();
		capability = new CauseAnalysisCapability(agent, agentServices, knownNeighbours);
		capability.setCurrentProviders(currentProviders);
		capability.setServiceCost(new Service("F"), 4);
		agent.setCapability(capability);
		agents.add(agent);

		agent = new ServiceProviderAgent("AGENT11");
		agentServices = new HashMap<>();
		serviceDependences = new ArrayList<>();
		serviceDependences.add(new Service("N"));
		agentServices.put(new Service("G"), serviceDependences);
		currentProviders = new HashMap<>();
		currentProviders.put(new Service("N"), new Component("AGENT21"));
		knownNeighbours = getAdjacencyList();
		capability = new CauseAnalysisCapability(agent, agentServices, knownNeighbours);
		capability.setCurrentProviders(currentProviders);
		capability.setServiceCost(new Service("G"), 2);
		agent.setCapability(capability);
		agents.add(agent);

		agent = new ServiceProviderAgent("AGENT12");
		agentServices = new HashMap<>();
		serviceDependences = new ArrayList<>();
		serviceDependences.add(new Service("O"));
		serviceDependences.add(new Service("P"));
		agentServices.put(new Service("G"), serviceDependences);
		currentProviders = new HashMap<>();
		currentProviders.put(new Service("O"), new Component("AGENT22"));
		currentProviders.put(new Service("P"), new Component("AGENT23"));
		knownNeighbours = getAdjacencyList();
		capability = new CauseAnalysisCapability(agent, agentServices, knownNeighbours);
		capability.setCurrentProviders(currentProviders);
		capability.setServiceCost(new Service("G"), 5);
		agent.setCapability(capability);
		agents.add(agent);

		agent = new ServiceProviderAgent("AGENT13");
		agentServices = new HashMap<>();
		serviceDependences = new ArrayList<>();
		serviceDependences.add(new Service("Q"));
		agentServices.put(new Service("H"), serviceDependences);
		currentProviders = new HashMap<>();
		currentProviders.put(new Service("Q"), new Component("AGENT24"));
		knownNeighbours = getAdjacencyList();
		capability = new CauseAnalysisCapability(agent, agentServices, knownNeighbours);
		capability.setCurrentProviders(currentProviders);
		capability.setServiceCost(new Service("H"), 4);
		agent.setCapability(capability);
		agents.add(agent);

		agent = new ServiceProviderAgent("AGENT14");
		agentServices = new HashMap<>();
		serviceDependences = new ArrayList<>();
		agentServices.put(new Service("I"), serviceDependences);
		knownNeighbours = getAdjacencyList();
		capability = new CauseAnalysisCapability(agent, agentServices, knownNeighbours);
		capability.setServiceCost(new Service("I"), 4);
		agent.setCapability(capability);
		agents.add(agent);

		agent = new ServiceProviderAgent("AGENT15");
		agentServices = new HashMap<>();
		serviceDependences = new ArrayList<>();
		serviceDependences.add(new Service("R"));
		serviceDependences.add(new Service("S"));
		agentServices.put(new Service("J"), serviceDependences);
		currentProviders = new HashMap<>();
		currentProviders.put(new Service("R"), new Component("AGENT25"));
		currentProviders.put(new Service("S"), new Component("AGENT26"));
		knownNeighbours = getAdjacencyList();
		capability = new CauseAnalysisCapability(agent, agentServices, knownNeighbours);
		capability.setCurrentProviders(currentProviders);
		capability.setServiceCost(new Service("J"), 5);
		agent.setCapability(capability);
		agents.add(agent);

		agent = new ServiceProviderAgent("AGENT16");
		agentServices = new HashMap<>();
		serviceDependences = new ArrayList<>();
		agentServices.put(new Service("J"), serviceDependences);
		knownNeighbours = getAdjacencyList();
		capability = new CauseAnalysisCapability(agent, agentServices, knownNeighbours);
		capability.setServiceCost(new Service("J"), 2);
		agent.setCapability(capability);
		agents.add(agent);

		agent = new ServiceProviderAgent("AGENT17");
		agentServices = new HashMap<>();
		serviceDependences = new ArrayList<>();
		serviceDependences.add(new Service("T"));
		agentServices.put(new Service("K"), serviceDependences);
		currentProviders = new HashMap<>();
		currentProviders.put(new Service("T"), new Component("AGENT27"));
		knownNeighbours = getAdjacencyList();
		capability = new CauseAnalysisCapability(agent, agentServices, knownNeighbours);
		capability.setCurrentProviders(currentProviders);
		capability.setServiceCost(new Service("K"), 4);
		agent.setCapability(capability);
		agents.add(agent);

		agent = new ServiceProviderAgent("AGENT18");
		agentServices = new HashMap<>();
		serviceDependences = new ArrayList<>();
		agentServices.put(new Service("L"), serviceDependences);
		knownNeighbours = getAdjacencyList();
		capability = new CauseAnalysisCapability(agent, agentServices, knownNeighbours);
		capability.setServiceCost(new Service("L"), 4);
		agent.setCapability(capability);
		agents.add(agent);

		agent = new ServiceProviderAgent("AGENT19");
		agentServices = new HashMap<>();
		serviceDependences = new ArrayList<>();
		agentServices.put(new Service("M"), serviceDependences);
		knownNeighbours = getAdjacencyList();
		capability = new CauseAnalysisCapability(agent, agentServices, knownNeighbours);
		capability.setServiceCost(new Service("M"), 4);
		agent.setCapability(capability);
		agents.add(agent);

		agent = new ServiceProviderAgent("AGENT20");
		agentServices = new HashMap<>();
		serviceDependences = new ArrayList<>();
		serviceDependences.add(new Service("U"));
		serviceDependences.add(new Service("V"));
		agentServices.put(new Service("N"), serviceDependences);
		currentProviders = new HashMap<>();
		currentProviders.put(new Service("U"), new Component("AGENT28"));
		currentProviders.put(new Service("V"), new Component("AGENT29"));
		knownNeighbours = getAdjacencyList();
		capability = new CauseAnalysisCapability(agent, agentServices, knownNeighbours);
		capability.setCurrentProviders(currentProviders);
		capability.setServiceCost(new Service("N"), 5);
		agent.setCapability(capability);
		agents.add(agent);

		agent = new ServiceProviderAgent("AGENT21");
		agentServices = new HashMap<>();
		serviceDependences = new ArrayList<>();
		serviceDependences.add(new Service("X"));
		agentServices.put(new Service("N"), serviceDependences);
		currentProviders = new HashMap<>();
		currentProviders.put(new Service("X"), new Component("AGENT30"));
		knownNeighbours = getAdjacencyList();
		capability = new CauseAnalysisCapability(agent, agentServices, knownNeighbours);
		capability.setCurrentProviders(currentProviders);
		capability.setServiceCost(new Service("N"), 2);
		agent.setCapability(capability);
		agents.add(agent);

		agent = new ServiceProviderAgent("AGENT22");
		agentServices = new HashMap<>();
		serviceDependences = new ArrayList<>();
		agentServices.put(new Service("O"), serviceDependences);
		knownNeighbours = getAdjacencyList();
		capability = new CauseAnalysisCapability(agent, agentServices, knownNeighbours);
		capability.setServiceCost(new Service("O"), 4);
		agent.setCapability(capability);
		agents.add(agent);

		agent = new ServiceProviderAgent("AGENT23");
		agentServices = new HashMap<>();
		serviceDependences = new ArrayList<>();
		agentServices.put(new Service("P"), serviceDependences);
		knownNeighbours = getAdjacencyList();
		capability = new CauseAnalysisCapability(agent, agentServices, knownNeighbours);
		capability.setServiceCost(new Service("P"), 4);
		agent.setCapability(capability);
		agents.add(agent);

		agent = new ServiceProviderAgent("AGENT24");
		agentServices = new HashMap<>();
		serviceDependences = new ArrayList<>();
		agentServices.put(new Service("Q"), serviceDependences);
		knownNeighbours = getAdjacencyList();
		capability = new CauseAnalysisCapability(agent, agentServices, knownNeighbours);
		capability.setServiceCost(new Service("Q"), 4);
		agent.setCapability(capability);
		agents.add(agent);

		agent = new ServiceProviderAgent("AGENT25");
		agentServices = new HashMap<>();
		serviceDependences = new ArrayList<>();
		agentServices.put(new Service("R"), serviceDependences);
		knownNeighbours = getAdjacencyList();
		capability = new CauseAnalysisCapability(agent, agentServices, knownNeighbours);
		capability.setServiceCost(new Service("R"), 4);
		agent.setCapability(capability);
		agents.add(agent);

		agent = new ServiceProviderAgent("AGENT26");
		agentServices = new HashMap<>();
		serviceDependences = new ArrayList<>();
		agentServices.put(new Service("S"), serviceDependences);
		knownNeighbours = getAdjacencyList();
		capability = new CauseAnalysisCapability(agent, agentServices, knownNeighbours);
		capability.setServiceCost(new Service("S"), 4);
		agent.setCapability(capability);
		agents.add(agent);

		agent = new ServiceProviderAgent("AGENT27");
		agentServices = new HashMap<>();
		serviceDependences = new ArrayList<>();
		agentServices.put(new Service("T"), serviceDependences);
		knownNeighbours = getAdjacencyList();
		capability = new CauseAnalysisCapability(agent, agentServices, knownNeighbours);
		capability.setServiceCost(new Service("T"), 4);
		agent.setCapability(capability);
		agents.add(agent);

		agent = new ServiceProviderAgent("AGENT28");
		agentServices = new HashMap<>();
		serviceDependences = new ArrayList<>();
		agentServices.put(new Service("U"), serviceDependences);
		knownNeighbours = getAdjacencyList();
		capability = new CauseAnalysisCapability(agent, agentServices, knownNeighbours);
		capability.setServiceCost(new Service("U"), 4);
		agent.setCapability(capability);
		agents.add(agent);

		agent = new ServiceProviderAgent("AGENT29");
		agentServices = new HashMap<>();
		serviceDependences = new ArrayList<>();
		agentServices.put(new Service("V"), serviceDependences);
		knownNeighbours = getAdjacencyList();
		capability = new CauseAnalysisCapability(agent, agentServices, knownNeighbours);
		capability.setServiceCost(new Service("V"), 4);
		agent.setCapability(capability);
		agents.add(agent);

		agent = new ServiceProviderAgent("AGENT30");
		agentServices = new HashMap<>();
		serviceDependences = new ArrayList<>();
		agentServices.put(new Service("X"), serviceDependences);
		knownNeighbours = getAdjacencyList();
		capability = new CauseAnalysisCapability(agent, agentServices, knownNeighbours);
		capability.setServiceCost(new Service("X"), 2);
		agent.setCapability(capability);
		agents.add(agent);
		
		agent = new ServiceProviderAgent("AGENT31");
		agentServices = new HashMap<>();
		serviceDependences = new ArrayList<>();
		agentServices.put(new Service("X"), serviceDependences);
		knownNeighbours = getAdjacencyList();
		capability = new CauseAnalysisCapability(agent, agentServices, knownNeighbours);
		capability.setServiceCost(new Service("X"), 5);
		agent.setCapability(capability);
		agents.add(agent);

		return agents;
	}

	public ArrayList<ServiceProviderAgent> createSupportingAgents() {
		ArrayList<ServiceProviderAgent> agents = new ArrayList<>();

		HashMap<Service, ArrayList<Service>> agentServices = new HashMap<>();
		HashMap<String, Integer> knownNeighbours = new HashMap<>();
		HashMap<Service, Component> currentProviders = new HashMap<>();
		currentProviders.put(new Service("B"), new Component("AGENT02"));
		currentProviders.put(new Service("D"), new Component("AGENT07"));
		currentProviders.put(new Service("E"), new Component("AGENT08"));
		currentProviders.put(new Service("J"), new Component("AGENT16"));

		ServiceProviderAgent agent = new ServiceProviderAgent("SUPPORT1");
		CauseAnalysisCapability capability = new CauseAnalysisCapability(agent, agentServices, knownNeighbours);
		capability.setCurrentProviders(currentProviders);
		agent.setCapability(capability);
		agents.add(agent);

		agent = new ServiceProviderAgent("SUPPORT2");
		capability = new CauseAnalysisCapability(agent, agentServices, knownNeighbours);
		capability.setCurrentProviders(currentProviders);
		agent.setCapability(capability);
		agents.add(agent);

		agent = new ServiceProviderAgent("SUPPORT3");
		capability = new CauseAnalysisCapability(agent, agentServices, knownNeighbours);
		capability.setCurrentProviders(currentProviders);
		agent.setCapability(capability);
		agents.add(agent);

		currentProviders = new HashMap<>();
		currentProviders.put(new Service("C"), new Component("AGENT04"));
		currentProviders.put(new Service("G"), new Component("AGENT11"));
		currentProviders.put(new Service("N"), new Component("AGENT21"));
		currentProviders.put(new Service("X"), new Component("AGENT30"));

		agent = new ServiceProviderAgent("SUPPORT4");
		capability = new CauseAnalysisCapability(agent, agentServices, knownNeighbours);
		capability.setCurrentProviders(currentProviders);
		agent.setCapability(capability);
		agents.add(agent);

		agent = new ServiceProviderAgent("SUPPORT5");
		capability = new CauseAnalysisCapability(agent, agentServices, knownNeighbours);
		capability.setCurrentProviders(currentProviders);
		agent.setCapability(capability);
		agents.add(agent);

		agent = new ServiceProviderAgent("SUPPORT6");
		capability = new CauseAnalysisCapability(agent, agentServices, knownNeighbours);
		capability.setCurrentProviders(currentProviders);
		agent.setCapability(capability);
		agents.add(agent);

		return agents;
	}

	private void setupDB(ServiceProviderAgent agent) {
		TraceDAO dao = new TraceDAO(agent.getAID().getLocalName());
		dao.dropCollection();
	}

}
