//----------------------------------------------------------------------------
// Copyright (C) 2011  Ingrid Nunes
// 
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later version.
// 
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.
// 
// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
// 
// To contact the authors:
// http://inf.ufrgs.br/prosoft/bdi4jade/
//
//----------------------------------------------------------------------------

package bdi4jade.examples.interactionprotocol.plan;

import java.io.File;
import java.io.IOException;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.List;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import bdi4jade.examples.interactionprotocol.Trace;
import bdi4jade.examples.interactionprotocol.dao.TraceDAO;
import bdi4jade.examples.interactionprotocol.domain.Component;
import bdi4jade.examples.interactionprotocol.domain.Service;
import bdi4jade.examples.interactionprotocol.goal.ProvideAnomalousProbabilityGoal;
import bdi4jade.plan.Plan.EndState;
import bdi4jade.plan.planbody.AbstractPlanBody;
import jade.lang.acl.ACLMessage;

/**
 * @author jgfaccin
 *
 */
public class ProvideAnomalousProbabilityPlanBody extends AbstractPlanBody {

	private static final long serialVersionUID = 6587858865136342797L;
	private final Log log = LogFactory.getLog(this.getClass());

	private String homePath = "src/bdi4jade/examples/interactionprotocol/";

	@Override
	public void action() {
		try {
			ProvideAnomalousProbabilityGoal goal = (ProvideAnomalousProbabilityGoal) this.getGoal();
			Component provider = new Component(goal.getMessage().getUserDefinedParameter(Component.NAME));
			Service service = new Service(goal.getMessage().getUserDefinedParameter(Service.NAME));

			ArrayList<Trace> traces = getTracesByProviderAndService(provider, service);

			if (!traces.isEmpty()) {
				Double probability = computeAnomalousProbability(traces);
				System.out.println("[" + this.getAgent().getLocalName() + "] Providing anomaly probability " + probability + " for "
						+ provider.getId() + "[" + service.getId() + "]");
				sendReply(goal.getMessage(), probability);
			}
		} catch (Exception e) {
			e.printStackTrace();
		}
		setEndState(EndState.SUCCESSFUL);

	}

	private void sendReply(ACLMessage msg, Double probability) {
		ACLMessage reply = msg.createReply();
		reply.setContent(String.valueOf(probability));
		reply.setPerformative(ACLMessage.CONFIRM);
		//log.info(reply.getConversationId() + "\t" + myAgent.getLocalName() + "\t" + reply.getPerformative() + "\t - \t" + reply.getContent());
		this.getAgent().send(reply);
	}

	private Double computeAnomalousProbability(ArrayList<Trace> traces) {
		long currentTime = System.currentTimeMillis();
		ArrayList<Double> values = new ArrayList<>();
		ArrayList<Double> rawWeights = new ArrayList<>();

		for (Trace trace : traces) {
			values.add((double)trace.getReceivedAt() - trace.getSentAt());
			rawWeights.add((double)currentTime - trace.getReceivedAt());
		}
		ArrayList<Double> weights = computeWeights(rawWeights);

		File inputFile = new File(homePath + this.myAgent.getLocalName() + "_in.txt");
		File outputFile = new File(homePath + this.myAgent.getLocalName() + "_out.txt");
		try {
			inputFile.createNewFile();
			outputFile.createNewFile();
		} catch (IOException e) {
			e.printStackTrace();
		}

		populateInputFile(inputFile, values, weights);
		callRScript(inputFile, outputFile);
		String output = readOutputFile(outputFile);
		inputFile.delete();
		outputFile.delete();
		Double val = Double.valueOf(output);
		Double probability = 1 - (val > 1 ? 1 : val); 

		return probability;
	}

	private void populateInputFile(File inputFile, ArrayList<Double> values, ArrayList<Double> weights) {
		Charset utf8 = StandardCharsets.UTF_8;
		List<String> lines = new ArrayList<>();
		lines.add("time\tweights");
		for (int i = 0; i < values.size(); i++) {
			lines.add(values.get(i).toString() + "\t" + weights.get(i).toString());
		}
		try {
			Files.write(Paths.get(inputFile.getAbsolutePath()), lines, utf8);
		} catch (Exception e) {
			e.printStackTrace();
		}
	}

	private String readOutputFile(File outputFile) {
		List<String> lines = new ArrayList<String>();
		try {
			lines = Files.readAllLines(Paths.get(outputFile.getAbsolutePath()));
		} catch (Exception e) {
			e.printStackTrace();
		}
		if (!lines.isEmpty()) {
			return lines.get(0);
		}
		return "1";
	}

	private void callRScript(File inputFile, File outputFile) {
		File scriptFile = new File(homePath + "script.r");
		String command = "Rscript " + scriptFile.getAbsolutePath() + " " + inputFile.getAbsolutePath() + " "
				+ outputFile.getAbsolutePath();
		try {
			Runtime.getRuntime().exec(command);
			Thread.sleep(1000);
		} catch (Exception e) {
			e.printStackTrace();
		}
	}

	private ArrayList<Double> computeWeights(ArrayList<Double> rawWeights) {
		Double minWeight = rawWeights.get(rawWeights.size() - 1);
		ArrayList<Double> normalizedWeights = new ArrayList<Double>();
		Double sum = 0.0;
		for (Double weight : rawWeights) {
			Double normalizedWeight = minWeight / weight;
			sum += normalizedWeight;
			normalizedWeights.add(normalizedWeight);
		}
		Double multiplier = 1.0 / sum;
		ArrayList<Double> weights = new ArrayList<Double>();
		for (Double normalizedWeight : normalizedWeights) {
			weights.add(normalizedWeight * multiplier);
		}
		return weights;
	}

	private ArrayList<Trace> getTracesByProviderAndService(Component component, Service service) {
		TraceDAO dao = new TraceDAO(getAgent().getLocalName());
		ArrayList<Trace> traces = dao.getTracesByServiceAndProvider(service.getId(), component.getId());
		return traces;
	}

}
