//----------------------------------------------------------------------------

// Copyright (C) 2011  Ingrid Nunes
// 
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later version.
// 
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.
// 
// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
// 
// To contact the authors:
// http://inf.ufrgs.br/prosoft/bdi4jade/
//
//----------------------------------------------------------------------------

package bdi4jade.examples.interactionprotocol.plan;

import java.time.Duration;
import java.time.Instant;
import java.util.ArrayList;
import java.util.Map;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import bdi4jade.belief.Belief;
import bdi4jade.event.GoalEvent;
import bdi4jade.examples.interactionprotocol.domain.Component;
import bdi4jade.examples.interactionprotocol.domain.Link;
import bdi4jade.examples.interactionprotocol.domain.LinkMonitor;
import bdi4jade.examples.interactionprotocol.domain.Service;
import bdi4jade.examples.interactionprotocol.goal.ProvideServiceGoal;
import bdi4jade.examples.interactionprotocol.goal.RequestServiceGoal;
import bdi4jade.plan.Plan.EndState;
import bdi4jade.plan.planbody.AbstractPlanBody;
import jade.lang.acl.ACLMessage;

/**
 * @author jgfaccin
 *
 */
public class ProvideServicePlanBody extends AbstractPlanBody {

	private static final long serialVersionUID = -7820028025514943596L;
	public static final String SUCCEEDED = "SUCCEEDED";
	private final Log log = LogFactory.getLog(this.getClass());

	@bdi4jade.annotation.Belief
	private Belief<String, Map<Service, ArrayList<Service>>> myServices;
	@bdi4jade.annotation.Belief
	private Belief<String, Boolean> failure;
	@bdi4jade.annotation.Belief
	private Belief<String, Map<Service, Integer>> serviceCost;

	private boolean sent = false;
	private int counter;
	private Instant startedAt;
	private int cummulatedCost;

	public ProvideServicePlanBody() {
		this.counter = 0;
		this.startedAt = Instant.now();
		this.cummulatedCost = 0;
	}
	
	@Override
	public void action() {
		ProvideServiceGoal goal = (ProvideServiceGoal) this.getGoal();
		ACLMessage requestMsg = goal.getMessage();
		Service service = new Service(requestMsg.getContent());
		ArrayList<Service> subServices = myServices.getValue().get(service);
		String parentCid = requestMsg.getConversationId();

		if (!subServices.isEmpty()) {
			if (!sent) {

				for (Service subService : subServices) {
					RequestServiceGoal subgoal = new RequestServiceGoal(subService, parentCid);
					dispatchSubgoalAndListen(subgoal);
				}
				this.sent = true;
			} else {
				GoalEvent goalEvent = getGoalEvent();
				if (goalEvent != null) {
					if (goalEvent.getGoal() instanceof RequestServiceGoal) {
						RequestServiceGoal subgoal = (RequestServiceGoal) goalEvent.getGoal();
						this.cummulatedCost += subgoal.getCost();
					}
					
					counter++;
					if (counter == subServices.size()) {
						this.sent = false;
						simulateProcessingTime(requestMsg);
						replyRequest(requestMsg, service);
					}
				}
			}
		} else {
			simulateProcessingTime(requestMsg);
			replyRequest(requestMsg, service);
		}
	}

	private void replyRequest(ACLMessage requestMsg, Service service) {
		ACLMessage replyRequest = requestMsg.createReply();
		Integer cost = serviceCost.getValue().get(service);
		replyRequest.setContent(cost.toString());
		replyRequest.setPerformative(ACLMessage.CONFIRM);

		Instant finishedAt = Instant.now();
		long delta = Duration.between(startedAt, finishedAt).toMillis();
		log.info(requestMsg.getConversationId() + "\t" + myAgent.getLocalName() + "\t"
				+ requestMsg.getSender().getLocalName() + "\t" + requestMsg.getContent() + "\t" + this.cummulatedCost + "\t" + delta);
		
		myAgent.send(replyRequest);
		setEndState(EndState.SUCCESSFUL);
	}

	private void simulateProcessingTime(ACLMessage msg) {
		try {
			Thread.sleep(10);
			Link link = new Link(new Component(this.getAgent().getLocalName()),
					new Component(msg.getSender().getLocalName()), new Service(msg.getContent()));
			if (failure.getValue()) {
				Thread.sleep(250);
			}
			if (LinkMonitor.getInstance().getAnomalousLinks().contains(link)) {
				Thread.sleep(250);
			}
		} catch (Exception e) {
			e.printStackTrace();
		}
	}
}
