//----------------------------------------------------------------------------
// Copyright (C) 2011  Ingrid Nunes
// 
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later version.
// 
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.
// 
// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
// 
// To contact the authors:
// http://inf.ufrgs.br/prosoft/bdi4jade/
//
//----------------------------------------------------------------------------

package bdi4jade.examples.interactionprotocol.plan;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;

import bdi4jade.belief.TransientPredicate;
import bdi4jade.event.GoalEvent;
import bdi4jade.examples.interactionprotocol.Trace;
import bdi4jade.examples.interactionprotocol.dao.TraceDAO;
import bdi4jade.examples.interactionprotocol.domain.Communication;
import bdi4jade.examples.interactionprotocol.domain.Component;
import bdi4jade.examples.interactionprotocol.domain.Conversation;
import bdi4jade.examples.interactionprotocol.domain.Service;
import bdi4jade.examples.interactionprotocol.domain.predicate.Abnormal;
import bdi4jade.examples.interactionprotocol.domain.predicate.AnomalousCommunication;
import bdi4jade.examples.interactionprotocol.goal.VerifyAbnormalBehaviourGoal;
import bdi4jade.examples.interactionprotocol.goal.VerifySuspiciousComponentGoal;
import bdi4jade.extension.remediation.graph.TransientCauseEffectRelationship;
import bdi4jade.extension.remediation.logics.Fact;
import bdi4jade.goal.PredicateGoal;
import bdi4jade.plan.Plan.EndState;
import jade.lang.acl.ACLMessage;

/**
 * @author jgfaccin
 *
 */
public class VerifyInternalOrExternalCausePlanBody extends CauseEffectKnowledgeModelUpdater {

	private static final long serialVersionUID = 8287156375478059409L;

	private boolean sent = false;
	private int counter = 0;
	HashMap<String, HashSet<String>> suspiciousComponents;

	public VerifyInternalOrExternalCausePlanBody() {
		super();
		this.suspiciousComponents = new HashMap<String, HashSet<String>>();
	}

	@SuppressWarnings("unchecked")
	@Override
	public void action() {
		VerifyAbnormalBehaviourGoal goal = (VerifyAbnormalBehaviourGoal) this.getGoal();
		if (!sent) {
			Conversation abnormalConversation = new Conversation(goal.getMsg().getContent());
			Abnormal abnormalBehaviour = new Abnormal(abnormalConversation);

			HashSet<String> serviceProviders = getServiceProviders(abnormalConversation.getId());

			System.out.println("[" + this.myAgent.getLocalName() + "] Verifying internal or external cause...");
			for (String provider : serviceProviders) {
				HashSet<String> anomalousServices = new HashSet<String>();
				HashSet<String> providedServices = getProvidedServices(abnormalConversation.getId(), provider);
				for (String service : providedServices) {
					System.out.println("\t[" + this.myAgent.getLocalName() + "] Buscando dados sobre " + service + " e " + provider);
					ArrayList<Long> data = getFilteredData(service, provider, abnormalConversation.getId());
					Long lastMeasurement = data.remove(data.size() - 1);
					Collections.sort(data);
					if (isAnomalous(data, lastMeasurement)) {
						anomalousServices.add(service);
					}
				}
				if (!anomalousServices.isEmpty()) {
					suspiciousComponents.put(provider, anomalousServices);
				}
			}

			if (suspiciousComponents.isEmpty()) {
				System.out.println(
						"[" + this.getAgent().getLocalName() + "] Dispatching <PredicateGoal[Abnormal], false>");
				dispatchSubgoalAndListen(new PredicateGoal<Abnormal>(abnormalBehaviour, false));
			} else {
				for (String provider : suspiciousComponents.keySet()) {
					HashSet<String> anomalousServices = suspiciousComponents.get(provider);
					for (String service : anomalousServices) {
						String childCid = getChildCid(abnormalConversation.getId(), provider, service);

						AnomalousCommunication anomalousCommPredicate = new AnomalousCommunication(
								new Communication(new Component(provider), new Service(service)));

						TransientPredicate<AnomalousCommunication> anomalousCommBelief = (TransientPredicate<AnomalousCommunication>) this
								.getBeliefBase().getBelief(anomalousCommPredicate);
						if (anomalousCommBelief != null) {
							anomalousCommBelief.setValue(true);
						} else {
							anomalousCommBelief = new TransientPredicate<AnomalousCommunication>(anomalousCommPredicate,
									true);
							this.getBeliefBase().addBelief(anomalousCommBelief);
						}

						// Create Cause Effect Relationship
						TransientCauseEffectRelationship tcer = new TransientCauseEffectRelationship(
								new Fact(anomalousCommPredicate, true));
						updateKnowledgeModel(tcer);

						// Create goals
						// !¬Anomalous(Communication)
						System.out.println("[" + this.getAgent().getLocalName()
								+ "] Dispatching <PredicateGoal[AnomalousCommunication], false>");
						dispatchSubgoalAndListen(
								new PredicateGoal<AnomalousCommunication>(anomalousCommPredicate, false));
						// **?Anomalous(Component)** virou VerifySuspiciousComponentGoal
						System.out.println(
								"[" + this.getAgent().getLocalName() + "] Dispatching <VerifySuspiciousComponent>");
						dispatchGoal(new VerifySuspiciousComponentGoal(new Component(provider), new Service(service),
								new Conversation(childCid)));

						this.counter++;
					}
				}
			}
			this.sent = true;
		} else {
			GoalEvent goalEvent = getGoalEvent();
			if (goalEvent != null) {
				this.counter--;
				if (this.counter <= 0) {
					this.sent = false;
					replyNotifier(goal.getMsg());
				}
			}
		}
	}

	private void replyNotifier(ACLMessage msg) {
		ACLMessage replyMsg = msg.createReply();
		replyMsg.setPerformative(ACLMessage.DISCONFIRM);
		myAgent.send(replyMsg);
		setEndState(EndState.SUCCESSFUL);
	}

	private boolean isAnomalous(ArrayList<Long> data, Long value) {
		if (data.isEmpty()) {
			return false;
		}
		
		int medianIndex = 0;

		// reset basic stats
		Double lowerQuartile = 0.0;
		Double upperQuartile = 0.0;
		Long median = 0L;
		Double lowerBound = 0.0;
		Double upperBound = 0.0;

		// the boolean determining whether there are an even number of elements
		Boolean evenm = false;

		// set the quartile positions
		if ((data.size() % 2) == 0) {
			medianIndex = (int) Math.floor(data.size() / 2);
			evenm = true;
		} else {
			medianIndex = (data.size() + 1) / 2;
			medianIndex = medianIndex - 1;
		}

		// get the median position
		if (medianIndex > 0) {
			// lower quartile
			lowerQuartile += (data.get((int) Math.floor(medianIndex / 2)) + data.get((int) Math
					.ceil(medianIndex / 2))) / 2;

			if (evenm == false) {
				// median of an odd set
				median = data.get(medianIndex);
			} else if (evenm == true) {
				// median of an even set
				median += (data.get(medianIndex) + data.get(medianIndex + 1)) / 2;
			}

			// upper quartile
			upperQuartile += (data.get(((int) (medianIndex + Math.ceil(medianIndex / 2)))) + data
					.get(((int) (medianIndex + Math.floor(medianIndex / 2))))) / 2;

			upperBound = median + ((upperQuartile - lowerQuartile) * 1.5);
			lowerBound = median - ((upperQuartile - lowerQuartile) * 1.5);
		}
		System.out.println("\t\tUpperBound: " + upperBound + " LowerBound: " + lowerBound + "Value: " + value);
		
		//return value < lowerBound || value > upperBound;
		return value > upperBound;
	}

	private HashSet<String> getProvidedServices(String parentCid, String provider) {
		TraceDAO dao = new TraceDAO(this.getAgent().getLocalName());
		ArrayList<Trace> traces = dao.getTracesByParentCIDAndProvider(parentCid, provider);

		HashSet<String> services = new HashSet<String>();

		for (Trace trace : traces) {
			services.add(trace.getService());
		}
		return services;
	}

	private HashSet<String> getServiceProviders(String parentCid) {
		TraceDAO dao = new TraceDAO(this.getAgent().getLocalName());
		ArrayList<Trace> traces = dao.getTracesByParentCID(parentCid);

		HashSet<String> providers = new HashSet<String>();

		for (Trace trace : traces) {
			providers.add(trace.getReceiver());
		}
		return providers;
	}

	private String getChildCid(String parentCid, String provider, String service) {
		TraceDAO dao = new TraceDAO(this.getAgent().getLocalName());
		String childCid = dao.getCIDByParentCIDAndProviderAndService(parentCid, provider, service);
		return childCid;
	}

	// Later I need to add the quality feature as a parameter.
	private ArrayList<Long> getFilteredData(String service, String provider, String parentCid) {
		TraceDAO dao = new TraceDAO(this.getAgent().getLocalName());
		ArrayList<Trace> traces = dao.getTracesOfProvidedByBefore(service, provider, parentCid);

		ArrayList<Long> data = new ArrayList<Long>();

		for (Trace trace : traces) {
			Long receivedAt = trace.getReceivedAt();
			Long sentAt = trace.getSentAt();
			if (receivedAt != null) {
				Long measurement = receivedAt - sentAt;
				data.add(measurement);
			}
		}
		return data;
	}

}
