/*
 * Copyright 2010-2013 Ning, Inc.
 * Copyright 2014 Groupon, Inc
 * Copyright 2014 The Billing Project, LLC
 *
 * The Billing Project licenses this file to you under the Apache License, version 2.0
 * (the "License"); you may not use this file except in compliance with the
 * License.  You may obtain a copy of the License at:
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.  See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

package org.killbill.billing.jaxrs;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.util.List;
import java.util.UUID;

import org.killbill.billing.client.KillBillClientException;
import org.killbill.billing.client.model.Account;
import org.killbill.billing.client.model.Invoice;
import org.killbill.billing.client.model.InvoiceItem;
import org.killbill.billing.client.model.InvoicePayment;
import org.killbill.billing.client.model.InvoicePayments;
import org.killbill.billing.client.model.Payment;
import org.killbill.billing.client.model.PaymentMethod;
import org.killbill.billing.client.model.Payments;
import org.killbill.billing.client.model.Refund;
import org.killbill.billing.client.model.Transaction;
import org.killbill.billing.payment.api.TransactionType;
import org.testng.Assert;
import org.testng.annotations.Test;

import com.google.common.collect.ImmutableList;

public class TestInvoicePayment extends TestJaxrsBase {

    @Test(groups = "slow")
    public void testRetrievePayment() throws Exception {
        final InvoicePayment paymentJson = setupScenarioWithPayment();
        final Payment retrievedPaymentJson = killBillClient.getPayment(paymentJson.getPaymentId(), false);
        Assert.assertTrue(retrievedPaymentJson.equals((Payment) paymentJson));
    }

    @Test(groups = "slow", description = "Can create a full refund with no adjustment")
    public void testFullRefundWithNoAdjustment() throws Exception {
        final InvoicePayment invoicePaymentJson = setupScenarioWithPayment();

        // Issue a refund for the full amount
        final BigDecimal refundAmount = invoicePaymentJson.getPurchasedAmount();
        final BigDecimal expectedInvoiceBalance = refundAmount;

        // Post and verify the refund
        final Refund refund = new Refund();
        refund.setPaymentId(invoicePaymentJson.getPaymentId());
        refund.setAmount(refundAmount);
        final Payment paymentAfterRefundJson = killBillClient.createInvoicePaymentRefund(refund, createdBy, reason, comment);
        verifyRefund(invoicePaymentJson, paymentAfterRefundJson, refundAmount);

        // Verify the invoice balance
        verifyInvoice(invoicePaymentJson, expectedInvoiceBalance);
    }

    @Test(groups = "slow", description = "Can create a partial refund with no adjustment")
    public void testPartialRefundWithNoAdjustment() throws Exception {
        final InvoicePayment paymentJson = setupScenarioWithPayment();

        // Issue a refund for a fraction of the amount
        final BigDecimal refundAmount = getFractionOfAmount(paymentJson.getPurchasedAmount());
        final BigDecimal expectedInvoiceBalance = refundAmount;

        // Post and verify the refund
        final Refund refund = new Refund();
        refund.setPaymentId(paymentJson.getPaymentId());
        refund.setAmount(refundAmount);
        final Payment paymentAfterRefundJson = killBillClient.createInvoicePaymentRefund(refund, createdBy, reason, comment);
        verifyRefund(paymentJson, paymentAfterRefundJson, refundAmount);

        // Verify the invoice balance
        verifyInvoice(paymentJson, expectedInvoiceBalance);
    }

    @Test(groups = "slow", description = "Can create a full refund with invoice adjustment")
    public void testFullRefundWithInvoiceAdjustment() throws Exception {
        final InvoicePayment paymentJson = setupScenarioWithPayment();

        // Issue a refund for the full amount
        final BigDecimal refundAmount = paymentJson.getPurchasedAmount();
        final BigDecimal expectedInvoiceBalance = BigDecimal.ZERO;

        // Post and verify the refund
        final Refund refund = new Refund();
        refund.setPaymentId(paymentJson.getPaymentId());
        refund.setAmount(refundAmount);
        refund.setAdjusted(true);
        final Payment paymentAfterRefundJson = killBillClient.createInvoicePaymentRefund(refund, createdBy, reason, comment);
        verifyRefund(paymentJson, paymentAfterRefundJson, refundAmount);

        // Verify the invoice balance
        verifyInvoice(paymentJson, expectedInvoiceBalance);
    }

    @Test(groups = "slow", description = "Can create a partial refund with invoice adjustment")
    public void testPartialRefundWithInvoiceAdjustment() throws Exception {
        final InvoicePayment paymentJson = setupScenarioWithPayment();

        // Issue a refund for a fraction of the amount
        final BigDecimal refundAmount = getFractionOfAmount(paymentJson.getPurchasedAmount());
        final BigDecimal expectedInvoiceBalance = BigDecimal.ZERO;

        // Post and verify the refund
        final Refund refund = new Refund();
        refund.setPaymentId(paymentJson.getPaymentId());
        refund.setAmount(refundAmount);
        refund.setAdjusted(true);
        final Payment paymentAfterRefundJson = killBillClient.createInvoicePaymentRefund(refund, createdBy, reason, comment);
        verifyRefund(paymentJson, paymentAfterRefundJson, refundAmount);

        // Verify the invoice balance
        verifyInvoice(paymentJson, expectedInvoiceBalance);
    }

    @Test(groups = "slow", description = "Can create a full refund with invoice item adjustment")
    public void testRefundWithFullInvoiceItemAdjustment() throws Exception {
        final InvoicePayment paymentJson = setupScenarioWithPayment();

        // Get the individual items for the invoice
        final Invoice invoice = killBillClient.getInvoice(paymentJson.getTargetInvoiceId(), true);
        final InvoiceItem itemToAdjust = invoice.getItems().get(0);

        // Issue a refund for the full amount
        final BigDecimal refundAmount = itemToAdjust.getAmount();
        final BigDecimal expectedInvoiceBalance = BigDecimal.ZERO;

        // Post and verify the refund
        final Refund refund = new Refund();
        refund.setPaymentId(paymentJson.getPaymentId());
        refund.setAmount(refundAmount);
        refund.setAdjusted(true);
        final InvoiceItem adjustment = new InvoiceItem();
        adjustment.setInvoiceItemId(itemToAdjust.getInvoiceItemId());
        /* null amount means full adjustment for that item */
        refund.setAdjustments(ImmutableList.<InvoiceItem>of(adjustment));
        final Payment paymentAfterRefundJson = killBillClient.createInvoicePaymentRefund(refund, createdBy, reason, comment);
        verifyRefund(paymentJson, paymentAfterRefundJson, refundAmount);

        // Verify the invoice balance
        verifyInvoice(paymentJson, expectedInvoiceBalance);
    }

    @Test(groups = "slow", description = "Can create a partial refund with invoice item adjustment")
    public void testPartialRefundWithInvoiceItemAdjustment() throws Exception {
        final InvoicePayment paymentJson = setupScenarioWithPayment();

        // Get the individual items for the invoice
        final Invoice invoice = killBillClient.getInvoice(paymentJson.getTargetInvoiceId(), true);
        final InvoiceItem itemToAdjust = invoice.getItems().get(0);

        // Issue a refund for a fraction of the amount
        final BigDecimal refundAmount = getFractionOfAmount(itemToAdjust.getAmount());
        final BigDecimal expectedInvoiceBalance = BigDecimal.ZERO;

        // Post and verify the refund
        final Refund refund = new Refund();
        refund.setPaymentId(paymentJson.getPaymentId());
        refund.setAdjusted(true);
        final InvoiceItem adjustment = new InvoiceItem();
        adjustment.setInvoiceItemId(itemToAdjust.getInvoiceItemId());
        adjustment.setAmount(refundAmount);
        refund.setAdjustments(ImmutableList.<InvoiceItem>of(adjustment));
        final Payment paymentAfterRefundJson = killBillClient.createInvoicePaymentRefund(refund, createdBy, reason, comment);
        verifyRefund(paymentJson, paymentAfterRefundJson, refundAmount);

        // Verify the invoice balance
        verifyInvoice(paymentJson, expectedInvoiceBalance);
    }

    @Test(groups = "slow", description = "Can paginate through all payments and refunds")
    public void testPaymentsAndRefundsPagination() throws Exception {
        InvoicePayment lastPayment = setupScenarioWithPayment();

        for (int i = 0; i < 5; i++) {
            final Refund refund = new Refund();
            refund.setPaymentId(lastPayment.getPaymentId());
            refund.setAmount(lastPayment.getPurchasedAmount());
            killBillClient.createInvoicePaymentRefund(refund, createdBy, reason, comment);

            final InvoicePayment invoicePayment = new InvoicePayment();
            invoicePayment.setPurchasedAmount(lastPayment.getPurchasedAmount());
            invoicePayment.setAccountId(lastPayment.getAccountId());
            invoicePayment.setTargetInvoiceId(lastPayment.getTargetInvoiceId());
            final InvoicePayment payment = killBillClient.createInvoicePayment(invoicePayment, true, createdBy, reason, comment);
            lastPayment = payment;
        }

        final InvoicePayments allPayments = killBillClient.getInvoicePaymentsForAccount(lastPayment.getAccountId());
        Assert.assertEquals(allPayments.size(), 6);

        final List<Transaction> objRefundFromJson = getDirectPaymentTransactions(allPayments, TransactionType.REFUND.toString());
        Assert.assertEquals(objRefundFromJson.size(), 5);

        Payments paymentsPage = killBillClient.getPayments(0L, 1L);
        for (int i = 0; i < 6; i++) {
            Assert.assertNotNull(paymentsPage);
            Assert.assertEquals(paymentsPage.size(), 1);
            Assert.assertEquals(paymentsPage.get(0), allPayments.get(i));
            paymentsPage = paymentsPage.getNext();
        }
        Assert.assertNull(paymentsPage);
    }

    private BigDecimal getFractionOfAmount(final BigDecimal amount) {
        return amount.divide(BigDecimal.TEN).setScale(2, BigDecimal.ROUND_HALF_UP);
    }

    private InvoicePayment setupScenarioWithPayment() throws Exception {
        final Account accountJson = createAccountWithPMBundleAndSubscriptionAndWaitForFirstInvoice();

        final List<InvoicePayment> paymentsForAccount = killBillClient.getPaymentsForAccount(accountJson.getAccountId());
        Assert.assertEquals(paymentsForAccount.size(), 1);

        final InvoicePayment paymentJson = paymentsForAccount.get(0);

        // Check the PaymentMethod from paymentMethodId returned in the Payment object
        final UUID paymentMethodId = paymentJson.getPaymentMethodId();
        final PaymentMethod paymentMethodJson = killBillClient.getPaymentMethod(paymentMethodId, true);
        Assert.assertEquals(paymentMethodJson.getPaymentMethodId(), paymentMethodId);
        Assert.assertEquals(paymentMethodJson.getAccountId(), accountJson.getAccountId());

        // Verify the refunds
        final List<Transaction> objRefundFromJson = getDirectPaymentTransactions(paymentsForAccount, TransactionType.REFUND.toString());
        Assert.assertEquals(objRefundFromJson.size(), 0);
        return paymentJson;
    }

    private void verifyRefund(final InvoicePayment paymentJson, final Payment paymentAfterRefund, final BigDecimal refundAmount) throws KillBillClientException {

        final List<Transaction> transactions = getDirectPaymentTransactions(ImmutableList.of(paymentAfterRefund), TransactionType.REFUND.toString());
        Assert.assertEquals(transactions.size(), 1);

        final Transaction refund = transactions.get(0);
        Assert.assertEquals(refund.getPaymentId(), paymentJson.getPaymentId());
        Assert.assertEquals(refund.getAmount().setScale(2, RoundingMode.HALF_UP), refundAmount.setScale(2, RoundingMode.HALF_UP));
        Assert.assertEquals(refund.getCurrency(), DEFAULT_CURRENCY);
        Assert.assertEquals(refund.getStatus(), "SUCCESS");
        Assert.assertEquals(refund.getEffectiveDate().getYear(), clock.getUTCNow().getYear());
        Assert.assertEquals(refund.getEffectiveDate().getMonthOfYear(), clock.getUTCNow().getMonthOfYear());
        Assert.assertEquals(refund.getEffectiveDate().getDayOfMonth(), clock.getUTCNow().getDayOfMonth());

        // Verify the refund via the payment API
        final Payment retrievedPaymentJson = killBillClient.getPayment(paymentJson.getPaymentId(), true);
        Assert.assertEquals(retrievedPaymentJson.getPaymentId(), paymentJson.getPaymentId());
        Assert.assertEquals(retrievedPaymentJson.getPurchasedAmount().setScale(2, RoundingMode.HALF_UP), paymentJson.getPurchasedAmount().setScale(2, RoundingMode.HALF_UP));
        Assert.assertEquals(retrievedPaymentJson.getAccountId(), paymentJson.getAccountId());
        Assert.assertEquals(retrievedPaymentJson.getCurrency(), paymentJson.getCurrency());
        Assert.assertEquals(retrievedPaymentJson.getPaymentMethodId(), paymentJson.getPaymentMethodId());
    }

    private void verifyInvoice(final InvoicePayment paymentJson, final BigDecimal expectedInvoiceBalance) throws KillBillClientException {
        final Invoice invoiceJson = killBillClient.getInvoice(paymentJson.getTargetInvoiceId());
        Assert.assertEquals(invoiceJson.getBalance().setScale(2, BigDecimal.ROUND_HALF_UP),
                            expectedInvoiceBalance.setScale(2, BigDecimal.ROUND_HALF_UP));
    }
}
