/*
 * Copyright 2010-2013 Ning, Inc.
 * Copyright 2014 Groupon, Inc
 * Copyright 2014 The Billing Project, LLC
 *
 * The Billing Project licenses this file to you under the Apache License, version 2.0
 * (the "License"); you may not use this file except in compliance with the
 * License.  You may obtain a copy of the License at:
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.  See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

package org.killbill.billing.jaxrs;

import java.math.BigDecimal;
import java.sql.Date;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import org.joda.time.DateTime;
import org.killbill.billing.catalog.api.BillingPeriod;
import org.killbill.billing.catalog.api.Currency;
import org.killbill.billing.catalog.api.ProductCategory;
import org.killbill.billing.catalog.api.TimeUnit;
import org.killbill.billing.client.KillBillClientException;
import org.killbill.billing.client.model.Catalog;
import org.killbill.billing.client.model.Catalogs;
import org.killbill.billing.client.model.Plan;
import org.killbill.billing.client.model.PlanDetail;
import org.killbill.billing.client.model.Product;
import org.killbill.billing.client.model.SimplePlan;
import org.testng.Assert;
import org.testng.annotations.Test;

import com.google.common.collect.ImmutableList;
import com.google.common.io.Resources;

public class TestCatalog extends TestJaxrsBase {

    @Test(groups = "slow", description = "Upload and retrieve a per tenant catalog")
    public void testMultiTenantCatalog() throws Exception {
        final String catalogPath = Resources.getResource("SpyCarBasic.xml").getPath();
        killBillClient.uploadXMLCatalog(catalogPath, createdBy, reason, comment);

        final String catalog = killBillClient.getXMLCatalog();
        Assert.assertNotNull(catalog);
    }

    @Test(groups = "slow", description = "Can retrieve a json version of the catalog")
    public void testCatalog() throws Exception {
        final Set<String> allBasePlans = new HashSet<String>();

        final List<Catalog> catalogsJson = killBillClient.getJSONCatalog();

        Assert.assertEquals(catalogsJson.get(0).getName(), "Firearms");
        Assert.assertEquals(catalogsJson.get(0).getEffectiveDate(), Date.valueOf("2011-01-01"));
        Assert.assertEquals(catalogsJson.get(0).getCurrencies().size(), 3);
        Assert.assertEquals(catalogsJson.get(0).getProducts().size(), 11);
        Assert.assertEquals(catalogsJson.get(0).getPriceLists().size(), 4);

        for (final Product productJson : catalogsJson.get(0).getProducts()) {
            if (!"BASE".equals(productJson.getType())) {
                Assert.assertEquals(productJson.getIncluded().size(), 0);
                Assert.assertEquals(productJson.getAvailable().size(), 0);
                continue;
            }

            // Save all plans for later (see below)
            for (final Plan planJson : productJson.getPlans()) {
                allBasePlans.add(planJson.getName());
            }

            // Retrieve available products (addons) for that base product
            final List<PlanDetail> availableAddons = killBillClient.getAvailableAddons(productJson.getName());
            final Set<String> availableAddonsNames = new HashSet<String>();
            for (final PlanDetail planDetailJson : availableAddons) {
                availableAddonsNames.add(planDetailJson.getProduct());
            }
            Assert.assertEquals(availableAddonsNames, new HashSet<String>(productJson.getAvailable()));
        }

        // Verify base plans endpoint
        final List<PlanDetail> basePlans = killBillClient.getBasePlans();
        final Set<String> foundBasePlans = new HashSet<String>();
        for (final PlanDetail planDetailJson : basePlans) {
            foundBasePlans.add(planDetailJson.getPlan());
        }
        Assert.assertEquals(foundBasePlans, allBasePlans);
    }

    @Test(groups = "slow", description = "Try to retrieve a json version of the catalog with an invalid date",
            expectedExceptions = KillBillClientException.class,
            expectedExceptionsMessageRegExp = "There is no catalog version that applies for the given date.*")
    public void testCatalogInvalidDate() throws Exception {
        final List<Catalog> catalogsJson = killBillClient.getJSONCatalog(DateTime.parse("2008-01-01"));
        Assert.fail();
    }

    @Test(groups = "slow", description = "Can create a simple Plan into a per-tenant catalog")
    public void testAddSimplePlan() throws Exception {

        killBillClient.addSimplePan(new SimplePlan("foo-monthly", "Foo", ProductCategory.BASE, Currency.USD, BigDecimal.TEN, BillingPeriod.MONTHLY, 0, TimeUnit.UNLIMITED, ImmutableList.<String>of()), requestOptions);
        List<Catalog> catalogsJson = killBillClient.getJSONCatalog(requestOptions);
        Assert.assertEquals(catalogsJson.size(),1);
        Assert.assertEquals(catalogsJson.get(0).getProducts().size(),1);
        Assert.assertEquals(catalogsJson.get(0).getProducts().get(0).getName(),"Foo");
        Assert.assertEquals(catalogsJson.get(0).getPriceLists().size(),1);
        Assert.assertEquals(catalogsJson.get(0).getPriceLists().get(0).getName(), "DEFAULT");
        Assert.assertEquals(catalogsJson.get(0).getPriceLists().get(0).getPlans().size(), 1);
        Assert.assertEquals(catalogsJson.get(0).getPriceLists().get(0).getPlans().get(0), "foo-monthly");


        killBillClient.addSimplePan(new SimplePlan("foo-annual", "Foo", ProductCategory.BASE, Currency.USD, new BigDecimal("100.00"), BillingPeriod.ANNUAL, 0, TimeUnit.UNLIMITED, ImmutableList.<String>of()), requestOptions);

        catalogsJson = killBillClient.getJSONCatalog(requestOptions);
        Assert.assertEquals(catalogsJson.size(),1);
        Assert.assertEquals(catalogsJson.get(0).getProducts().size(),1);
        Assert.assertEquals(catalogsJson.get(0).getProducts().get(0).getName(),"Foo");
        Assert.assertEquals(catalogsJson.get(0).getPriceLists().size(),1);
        Assert.assertEquals(catalogsJson.get(0).getPriceLists().get(0).getName(), "DEFAULT");
        Assert.assertEquals(catalogsJson.get(0).getPriceLists().get(0).getPlans().size(), 2);

    }
}
