/*
 * Copyright 2010-2013 Ning, Inc.
 * Copyright 2014 Groupon, Inc
 * Copyright 2014 The Billing Project, LLC
 *
 * The Billing Project licenses this file to you under the Apache License, version 2.0
 * (the "License"); you may not use this file except in compliance with the
 * License.  You may obtain a copy of the License at:
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.  See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

package org.killbill.billing.catalog;

import java.util.List;

import org.killbill.billing.ErrorCode;
import org.killbill.billing.callcontext.InternalTenantContext;
import org.killbill.billing.catalog.api.Catalog;
import org.killbill.billing.catalog.api.CatalogApiException;
import org.killbill.billing.catalog.api.CatalogService;
import org.killbill.billing.catalog.api.StaticCatalog;
import org.killbill.billing.catalog.io.VersionedCatalogLoader;
import org.killbill.billing.platform.api.KillbillService;
import org.killbill.billing.platform.api.LifecycleHandlerType;
import org.killbill.billing.platform.api.LifecycleHandlerType.LifecycleLevel;
import org.killbill.billing.tenant.api.TenantApiException;
import org.killbill.billing.tenant.api.TenantInternalApi;
import org.killbill.billing.tenant.api.TenantKV.TenantKey;
import org.killbill.billing.tenant.api.TenantUserApi;
import org.killbill.billing.util.callcontext.InternalCallContextFactory;
import org.killbill.billing.util.callcontext.TenantContext;
import org.killbill.billing.util.config.CatalogConfig;

import com.google.inject.Inject;
import com.google.inject.Provider;

public class DefaultCatalogService implements KillbillService, CatalogService {

    private static final String CATALOG_SERVICE_NAME = "catalog-service";

    private static VersionedCatalog catalog;

    private final CatalogConfig config;
    private boolean isInitialized;

    private final VersionedCatalogLoader loader;
    private final TenantInternalApi tenantApi;

    @Inject
    public DefaultCatalogService(final CatalogConfig config, final TenantInternalApi tenantApi, final VersionedCatalogLoader loader) {
        this.config = config;
        this.isInitialized = false;
        this.loader = loader;
        this.tenantApi = tenantApi;

    }

    @LifecycleHandlerType(LifecycleLevel.LOAD_CATALOG)
    public synchronized void loadCatalog() throws ServiceException {
        if (!isInitialized) {
            try {
                final String url = config.getCatalogURI();
                catalog = loader.load(url);
                isInitialized = true;
            } catch (Exception e) {
                throw new ServiceException(e);
            }
        }
    }

    @Override
    public String getName() {
        return CATALOG_SERVICE_NAME;
    }

    @Override
    public Catalog getFullCatalog(final InternalTenantContext context) throws CatalogApiException {
        return getCatalog(context);
    }

    @Override
    public StaticCatalog getCurrentCatalog(final InternalTenantContext context) throws CatalogApiException {
        return getCatalog(context);
    }

    private VersionedCatalog getCatalog(final InternalTenantContext context) throws CatalogApiException {
        if (context.getTenantRecordId() == InternalCallContextFactory.INTERNAL_TENANT_RECORD_ID) {
            return catalog;
        }
        try {
            final List<String> catalogXMLs = tenantApi.getTenantCatalogs(context);
            if (catalogXMLs.isEmpty()) {
                return catalog;
            }
            return loader.load(catalogXMLs);
        } catch (TenantApiException e) {
            throw new CatalogApiException(e);
        } catch (ServiceException e) {
            throw new CatalogApiException(ErrorCode.CAT_INVALID_FOR_TENANT, "Failed to load catalog for tenant " + context.getTenantRecordId());
        }
    }

}
