/*
 * Decompiled with CFR 0.152.
 */
package voldemort.utils;

import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import java.io.File;
import java.io.IOException;
import java.io.StringReader;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Random;
import java.util.Set;
import java.util.TreeMap;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.TimeUnit;
import org.apache.commons.io.FileUtils;
import org.apache.log4j.Logger;
import voldemort.VoldemortException;
import voldemort.client.protocol.admin.AdminClient;
import voldemort.client.protocol.admin.AdminClientConfig;
import voldemort.client.rebalance.RebalanceClusterPlan;
import voldemort.client.rebalance.RebalanceNodePlan;
import voldemort.client.rebalance.RebalancePartitionsInfo;
import voldemort.cluster.Cluster;
import voldemort.cluster.Node;
import voldemort.routing.RoutingStrategy;
import voldemort.routing.RoutingStrategyFactory;
import voldemort.server.VoldemortConfig;
import voldemort.server.rebalance.VoldemortRebalancingException;
import voldemort.store.StoreDefinition;
import voldemort.store.metadata.MetadataStore;
import voldemort.store.readonly.ReadOnlyStorageFormat;
import voldemort.utils.ByteArray;
import voldemort.utils.KeyDistributionGenerator;
import voldemort.utils.Pair;
import voldemort.utils.Utils;
import voldemort.versioning.Occurred;
import voldemort.versioning.VectorClock;
import voldemort.versioning.Versioned;
import voldemort.xml.ClusterMapper;

/*
 * This class specifies class file version 49.0 but uses Java 6 signatures.  Assumed Java 6.
 */
public class RebalanceUtils {
    private static Logger logger = Logger.getLogger(RebalanceUtils.class);
    public static final List<String> canRebalanceList = Arrays.asList("bdb", "read-only");
    public static final String initialClusterFileName = "initial-cluster.xml";
    public static final String finalClusterFileName = "final-cluster.xml";

    public static HashMap<Integer, List<Integer>> getOptimizedReplicaToPartitionList(int stealerNodeId, Cluster cluster, StoreDefinition storeDef, HashMap<Integer, List<Integer>> currentReplicaToPartitionList) {
        HashMap optimizedReplicaToPartitionList = Maps.newHashMap();
        RoutingStrategy strategy = new RoutingStrategyFactory().updateRoutingStrategy(storeDef, cluster);
        for (Map.Entry<Integer, List<Integer>> tuple : currentReplicaToPartitionList.entrySet()) {
            ArrayList partitionList = Lists.newArrayList();
            for (int partition : tuple.getValue()) {
                List<Integer> preferenceList = strategy.getReplicatingPartitionList(partition);
                if (RebalanceUtils.containsPreferenceList(cluster, preferenceList, stealerNodeId)) continue;
                partitionList.add(partition);
            }
            if (partitionList.size() <= 0) continue;
            optimizedReplicaToPartitionList.put(tuple.getKey(), partitionList);
        }
        return optimizedReplicaToPartitionList;
    }

    public static Versioned<Cluster> getLatestCluster(List<Integer> requiredNodes, AdminClient adminClient) {
        Versioned<Cluster> latestCluster = new Versioned<Cluster>(adminClient.getAdminClientCluster());
        ArrayList<Versioned<Cluster>> clusterList = new ArrayList<Versioned<Cluster>>();
        clusterList.add(latestCluster);
        for (Node node : adminClient.getAdminClientCluster().getNodes()) {
            try {
                Versioned<Cluster> versionedCluster = adminClient.getRemoteCluster(node.getId());
                VectorClock newClock = (VectorClock)versionedCluster.getVersion();
                if (null == newClock || clusterList.contains(versionedCluster)) continue;
                RebalanceUtils.checkNotConcurrent(clusterList, newClock);
                clusterList.add(versionedCluster);
                Occurred occurred = newClock.compare(latestCluster.getVersion());
                if (!Occurred.AFTER.equals((Object)occurred)) continue;
                latestCluster = versionedCluster;
            }
            catch (Exception e) {
                if (null != requiredNodes && requiredNodes.contains(node.getId())) {
                    throw new VoldemortException("Failed on node " + node.getId(), e);
                }
                logger.info((Object)("Failed on node " + node.getId()), (Throwable)e);
            }
        }
        return latestCluster;
    }

    private static void checkNotConcurrent(ArrayList<Versioned<Cluster>> clockList, VectorClock newClock) {
        for (Versioned<Cluster> versionedCluster : clockList) {
            VectorClock clock = (VectorClock)versionedCluster.getVersion();
            if (!Occurred.CONCURRENTLY.equals((Object)clock.compare(newClock))) continue;
            throw new VoldemortException("Cluster is in inconsistent state because we got conflicting clocks " + clock + " and on current node " + newClock);
        }
    }

    public static int getCrossZoneMoves(Cluster targetCluster, RebalanceClusterPlan plan) {
        int crossZoneMoves = 0;
        for (RebalanceNodePlan nodePlan : plan.getRebalancingTaskQueue()) {
            List<RebalancePartitionsInfo> infos = nodePlan.getRebalanceTaskList();
            for (RebalancePartitionsInfo info : infos) {
                Node donorNode = targetCluster.getNodeById(info.getDonorId());
                Node stealerNode = targetCluster.getNodeById(info.getStealerId());
                if (donorNode.getZoneId() == stealerNode.getZoneId()) continue;
                ++crossZoneMoves;
            }
        }
        return crossZoneMoves;
    }

    public static int getTotalMoves(RebalanceClusterPlan plan) {
        int totalMoves = 0;
        for (RebalanceNodePlan nodePlan : plan.getRebalancingTaskQueue()) {
            totalMoves += nodePlan.getRebalanceTaskList().size();
        }
        return totalMoves;
    }

    public static void assertSameDonor(List<RebalancePartitionsInfo> partitionInfos, int expectedDonorId) {
        int donorId = expectedDonorId < 0 ? partitionInfos.get(0).getDonorId() : expectedDonorId;
        for (RebalancePartitionsInfo info : partitionInfos) {
            if (info.getDonorId() == donorId) continue;
            throw new VoldemortException("Found a stealer information " + info + " having a different donor node from others ( " + donorId + " )");
        }
    }

    public static void generateMinCluster(Cluster currentCluster, Cluster targetCluster, List<StoreDefinition> storeDefs, String outputDir, int tries) {
        HashMap<StoreDefinition, Integer> uniqueStores = KeyDistributionGenerator.getUniqueStoreDefinitionsWithCounts(storeDefs);
        List<ByteArray> keys = KeyDistributionGenerator.generateKeys(10000);
        Cluster minCluster = targetCluster;
        int minMoves = Integer.MAX_VALUE;
        double minStdDev = Double.MAX_VALUE;
        for (int numTries = 0; numTries < tries; ++numTries) {
            Pair<Cluster, Integer> minClusterMove = RebalanceUtils.generateMinCluster(currentCluster, targetCluster, storeDefs);
            double currentStdDev = KeyDistributionGenerator.getStdDeviation(KeyDistributionGenerator.generateOverallDistributionWithUniqueStores(minClusterMove.getFirst(), uniqueStores, keys));
            System.out.println("Optimization number " + numTries + ": " + minClusterMove.getSecond() + " moves, " + currentStdDev + " std dev");
            System.out.println("Current min moves: " + minMoves + "; current min std dev: " + minStdDev);
            if (!(currentStdDev <= minStdDev)) continue;
            if (minClusterMove.getSecond() > minMoves) {
                System.out.println("Warning: the newly chosen cluster requires " + (minClusterMove.getSecond() - minMoves) + " addition moves!");
            }
            minMoves = minClusterMove.getSecond();
            minStdDev = currentStdDev;
            minCluster = minClusterMove.getFirst();
            System.out.println("Current distribution");
            System.out.println(KeyDistributionGenerator.printOverallDistribution(currentCluster, storeDefs, keys));
            System.out.println("-------------------------\n");
            System.out.println("Target distribution");
            System.out.println(KeyDistributionGenerator.printOverallDistribution(minCluster, storeDefs, keys));
            System.out.println("=========================\n");
            if (outputDir == null) continue;
            try {
                FileUtils.writeStringToFile((File)new File(outputDir, finalClusterFileName + numTries), (String)new ClusterMapper().writeCluster(minCluster));
                continue;
            }
            catch (Exception e) {
                // empty catch block
            }
        }
        System.out.println("\n==========================");
        System.out.println("Final distribution");
        System.out.println(KeyDistributionGenerator.printOverallDistribution(minCluster, storeDefs, keys));
        System.out.println("=========================\n");
        if (outputDir != null) {
            try {
                FileUtils.writeStringToFile((File)new File(outputDir, finalClusterFileName), (String)new ClusterMapper().writeCluster(minCluster));
            }
            catch (Exception e) {
                // empty catch block
            }
        }
    }

    public static Pair<Cluster, Integer> generateMinCluster(Cluster currentCluster, Cluster targetCluster, List<StoreDefinition> storeDefs) {
        int currentNumNodes = currentCluster.getNumberOfNodes();
        int targetNumNodes = targetCluster.getNumberOfNodes();
        ArrayList newNodeIds = Lists.newArrayList();
        ArrayList donorNodeIds = Lists.newArrayList();
        ArrayList allNodes = Lists.newArrayList();
        HashMap numPartitionsPerZone = Maps.newHashMap();
        HashMap numNodesPerZone = Maps.newHashMap();
        HashMap numDonorNodesPerZone = Maps.newHashMap();
        for (Node node : targetCluster.getNodes()) {
            if (node.getPartitionIds().isEmpty()) {
                newNodeIds.add(node.getId());
            } else {
                donorNodeIds.add(node.getId());
                if (numDonorNodesPerZone.containsKey(node.getZoneId())) {
                    int currentNumDonorNodesInZone = (Integer)numNodesPerZone.get(node.getZoneId());
                    numDonorNodesPerZone.put(node.getZoneId(), ++currentNumDonorNodesInZone);
                } else {
                    numDonorNodesPerZone.put(node.getZoneId(), 1);
                }
            }
            allNodes.add(RebalanceUtils.updateNode(node, Lists.newArrayList(node.getPartitionIds())));
            if (numPartitionsPerZone.containsKey(node.getZoneId())) {
                int currentNumPartitionsInZone = (Integer)numPartitionsPerZone.get(node.getZoneId());
                numPartitionsPerZone.put(node.getZoneId(), currentNumPartitionsInZone += node.getNumberOfPartitions());
            } else {
                numPartitionsPerZone.put(node.getZoneId(), node.getNumberOfPartitions());
            }
            if (numNodesPerZone.containsKey(node.getZoneId())) {
                int currentNumNodesInZone = (Integer)numNodesPerZone.get(node.getZoneId());
                numNodesPerZone.put(node.getZoneId(), ++currentNumNodesInZone);
                continue;
            }
            numNodesPerZone.put(node.getZoneId(), 1);
        }
        Cluster returnCluster = RebalanceUtils.updateCluster(targetCluster, allNodes);
        int totalPrimaryPartitionsMoved = 0;
        if (currentNumNodes == targetNumNodes) {
            return Pair.create(returnCluster, totalPrimaryPartitionsMoved);
        }
        Iterator i$ = newNodeIds.iterator();
        block1: while (i$.hasNext()) {
            int newNodeId = (Integer)i$.next();
            Node newNode = targetCluster.getNodeById(newNodeId);
            int partitionsToSteal = (int)Math.floor((double)((Integer)numPartitionsPerZone.get(newNode.getZoneId())).intValue() * 1.0 / (double)((Integer)numNodesPerZone.get(newNode.getZoneId())).intValue());
            int nodesStolenFrom = 0;
            for (int index = 0; index < donorNodeIds.size(); ++index) {
                int donorNodeId = (Integer)donorNodeIds.get(index);
                Node donorNode = currentCluster.getNodeById(donorNodeId);
                if (donorNode.getZoneId() != newNode.getZoneId()) continue;
                if (partitionsToSteal <= 0) continue block1;
                int partitionsToDonate = Math.max((int)Math.floor(partitionsToSteal / ((Integer)numDonorNodesPerZone.get(newNode.getZoneId()) - nodesStolenFrom)), 1);
                ++nodesStolenFrom;
                if (returnCluster.getNodeById(donorNodeId).getNumberOfPartitions() <= partitionsToDonate) continue;
                ArrayList donorPartitions = Lists.newArrayList(returnCluster.getNodeById(donorNodeId).getPartitionIds());
                Collections.shuffle(donorPartitions, new Random(System.currentTimeMillis()));
                int partitionsDonated = 0;
                Iterator i$2 = donorPartitions.iterator();
                while (i$2.hasNext()) {
                    int donorPartition = (Integer)i$2.next();
                    if (partitionsDonated == partitionsToDonate) break;
                    Cluster intermediateCluster = RebalanceUtils.createUpdatedCluster(returnCluster, newNodeId, Lists.newArrayList((Object[])new Integer[]{donorPartition}));
                    if (RebalanceUtils.getCrossZoneMoves(intermediateCluster, new RebalanceClusterPlan(returnCluster, intermediateCluster, storeDefs, true)) != 0) continue;
                    returnCluster = intermediateCluster;
                    ++partitionsDonated;
                    ++totalPrimaryPartitionsMoved;
                }
                partitionsToSteal -= partitionsDonated;
            }
        }
        return Pair.create(returnCluster, totalPrimaryPartitionsMoved);
    }

    public static boolean checkKeyBelongsToPartition(int nodeId, byte[] key, HashMap<Integer, List<Integer>> replicaToPartitionList, Cluster cluster, StoreDefinition storeDef) {
        List<Integer> keyPartitions = new RoutingStrategyFactory().updateRoutingStrategy(storeDef, cluster).getPartitionList(key);
        List<Integer> nodePartitions = cluster.getNodeById(nodeId).getPartitionIds();
        return RebalanceUtils.checkKeyBelongsToPartition(keyPartitions, nodePartitions, replicaToPartitionList);
    }

    public static boolean checkKeyBelongsToPartition(List<Integer> keyPartitions, List<Integer> nodePartitions, HashMap<Integer, List<Integer>> replicaToPartitionList) {
        replicaToPartitionList = Utils.notNull(replicaToPartitionList);
        for (int replicaNum = 0; replicaNum < keyPartitions.size(); ++replicaNum) {
            List<Integer> partitionsToMove;
            if (!nodePartitions.contains(keyPartitions.get(replicaNum)) || (partitionsToMove = replicaToPartitionList.get(replicaNum)) == null || partitionsToMove.size() <= 0 || !partitionsToMove.contains(keyPartitions.get(0))) continue;
            return true;
        }
        return false;
    }

    public static List<Integer> checkKeyBelongsToPartition(byte[] key, Set<Pair<Integer, HashMap<Integer, List<Integer>>>> stealerNodeToMappingTuples, Cluster cluster, StoreDefinition storeDef) {
        List<Integer> keyPartitions = new RoutingStrategyFactory().updateRoutingStrategy(storeDef, cluster).getPartitionList(key);
        ArrayList nodesToPush = Lists.newArrayList();
        for (Pair<Integer, HashMap<Integer, List<Integer>>> stealNodeToMap : stealerNodeToMappingTuples) {
            List<Integer> nodePartitions = cluster.getNodeById(stealNodeToMap.getFirst()).getPartitionIds();
            if (!RebalanceUtils.checkKeyBelongsToPartition(keyPartitions, nodePartitions, stealNodeToMap.getSecond())) continue;
            nodesToPush.add(stealNodeToMap.getFirst());
        }
        return nodesToPush;
    }

    public static void validateClusterState(Cluster cluster, AdminClient adminClient) {
        for (Node node : cluster.getNodes()) {
            Versioned<MetadataStore.VoldemortState> versioned = adminClient.getRemoteServerState(node.getId());
            if (!MetadataStore.VoldemortState.NORMAL_SERVER.equals((Object)versioned.getValue())) {
                throw new VoldemortRebalancingException("Cannot rebalance since node " + node.getId() + " (" + node.getHost() + ") is not in normal state, but in " + (Object)((Object)versioned.getValue()));
            }
            if (!logger.isInfoEnabled()) continue;
            logger.info((Object)("Node " + node.getId() + " (" + node.getHost() + ") is ready for rebalance."));
        }
    }

    public static Cluster getClusterWithNewNodes(Cluster currentCluster, Cluster targetCluster) {
        ArrayList<Node> newNodes = new ArrayList<Node>();
        for (Node node : targetCluster.getNodes()) {
            if (RebalanceUtils.containsNode(currentCluster, node.getId())) continue;
            newNodes.add(RebalanceUtils.updateNode(node, new ArrayList<Integer>()));
        }
        return RebalanceUtils.updateCluster(currentCluster, newNodes);
    }

    public static Cluster updateCluster(Cluster currentCluster, List<Node> updatedNodeList) {
        ArrayList<Node> newNodeList = new ArrayList<Node>(updatedNodeList);
        for (Node currentNode : currentCluster.getNodes()) {
            if (updatedNodeList.contains(currentNode)) continue;
            newNodeList.add(currentNode);
        }
        Collections.sort(newNodeList);
        return new Cluster(currentCluster.getName(), newNodeList, Lists.newArrayList(currentCluster.getZones()));
    }

    public static boolean containsNode(Cluster cluster, int nodeId) {
        try {
            cluster.getNodeById(nodeId);
            return true;
        }
        catch (VoldemortException e) {
            return false;
        }
    }

    public static boolean containsPreferenceList(Cluster cluster, List<Integer> preferenceList, int nodeId) {
        for (int partition : preferenceList) {
            if (RebalanceUtils.getNodeByPartitionId(cluster, partition).getId() != nodeId) continue;
            return true;
        }
        return false;
    }

    public static Cluster createUpdatedCluster(Cluster currentCluster, int stealerNodeId, List<Integer> donatedPartitions) {
        ClusterMapper mapper = new ClusterMapper();
        Cluster updatedCluster = mapper.readCluster(new StringReader(mapper.writeCluster(currentCluster)));
        for (int donatedPartition : donatedPartitions) {
            Node stealerNode;
            Node donorNode = RebalanceUtils.getNodeByPartitionId(updatedCluster, donatedPartition);
            if (donorNode == (stealerNode = updatedCluster.getNodeById(stealerNodeId))) continue;
            donorNode = RebalanceUtils.removePartitionToNode(donorNode, donatedPartition);
            stealerNode = RebalanceUtils.addPartitionToNode(stealerNode, donatedPartition);
            updatedCluster = RebalanceUtils.updateCluster(updatedCluster, Lists.newArrayList((Object[])new Node[]{donorNode, stealerNode}));
        }
        return updatedCluster;
    }

    public static Node updateNode(Node node, List<Integer> partitionsList) {
        return new Node(node.getId(), node.getHost(), node.getHttpPort(), node.getSocketPort(), node.getAdminPort(), node.getZoneId(), partitionsList);
    }

    public static Node addPartitionToNode(Node node, Integer donatedPartition) {
        return RebalanceUtils.addPartitionToNode(node, Sets.newHashSet((Object[])new Integer[]{donatedPartition}));
    }

    public static Node removePartitionToNode(Node node, Integer donatedPartition) {
        return RebalanceUtils.removePartitionToNode(node, Sets.newHashSet((Object[])new Integer[]{donatedPartition}));
    }

    public static Node addPartitionToNode(Node node, Set<Integer> donatedPartitions) {
        ArrayList<Integer> deepCopy = new ArrayList<Integer>(node.getPartitionIds());
        deepCopy.addAll(donatedPartitions);
        Collections.sort(deepCopy);
        return RebalanceUtils.updateNode(node, deepCopy);
    }

    public static Node removePartitionToNode(Node node, Set<Integer> donatedPartitions) {
        ArrayList<Integer> deepCopy = new ArrayList<Integer>(node.getPartitionIds());
        deepCopy.removeAll(donatedPartitions);
        return RebalanceUtils.updateNode(node, deepCopy);
    }

    public static Map<Integer, Integer> getCurrentPartitionMapping(Cluster currentCluster) {
        LinkedHashMap<Integer, Integer> partitionToNode = new LinkedHashMap<Integer, Integer>();
        for (Node node : currentCluster.getNodes()) {
            for (Integer partition : node.getPartitionIds()) {
                Integer previousRegisteredNodeId = (Integer)partitionToNode.get(partition);
                if (previousRegisteredNodeId != null) {
                    throw new IllegalArgumentException("Partition id " + partition + " found on two nodes : " + node.getId() + " and " + previousRegisteredNodeId);
                }
                partitionToNode.put(partition, node.getId());
            }
        }
        return partitionToNode;
    }

    public static void propagateCluster(AdminClient adminClient, Cluster cluster) {
        HashMap currentClusters = Maps.newHashMap();
        Versioned<Cluster> latestCluster = new Versioned<Cluster>(cluster);
        ArrayList<Versioned<Cluster>> clusterList = new ArrayList<Versioned<Cluster>>();
        clusterList.add(latestCluster);
        for (Node node : cluster.getNodes()) {
            try {
                Versioned<Cluster> versionedCluster = adminClient.getRemoteCluster(node.getId());
                VectorClock newClock = (VectorClock)versionedCluster.getVersion();
                currentClusters.put(node.getId(), versionedCluster.getValue());
                if (null == newClock || clusterList.contains(versionedCluster)) continue;
                RebalanceUtils.checkNotConcurrent(clusterList, newClock);
                clusterList.add(versionedCluster);
                Occurred occurred = newClock.compare(latestCluster.getVersion());
                if (!Occurred.AFTER.equals((Object)occurred)) continue;
                latestCluster = versionedCluster;
            }
            catch (Exception e) {
                throw new VoldemortException("Failed to get cluster version from node " + node.getId(), e);
            }
        }
        VectorClock latestClock = ((VectorClock)latestCluster.getVersion()).incremented(0, System.currentTimeMillis());
        HashSet completedNodeIds = Sets.newHashSet();
        try {
            for (Node node : cluster.getNodes()) {
                logger.info((Object)("Updating cluster definition on remote node " + node));
                adminClient.updateRemoteCluster(node.getId(), cluster, latestClock);
                logger.info((Object)("Updated cluster definition " + cluster + " on remote node " + node.getId()));
                completedNodeIds.add(node.getId());
            }
        }
        catch (VoldemortException e) {
            for (Integer completedNodeId : completedNodeIds) {
                try {
                    adminClient.updateRemoteCluster(completedNodeId, (Cluster)currentClusters.get(completedNodeId), latestClock);
                }
                catch (VoldemortException exception) {
                    logger.error((Object)("Could not revert cluster metadata back on node " + completedNodeId));
                }
            }
            throw e;
        }
    }

    public static List<Integer> getStolenPrimaryPartitions(Cluster currentCluster, Cluster targetCluster, int stealNodeId) {
        ArrayList<Integer> targetList = new ArrayList<Integer>(targetCluster.getNodeById(stealNodeId).getPartitionIds());
        List<Object> currentList = new ArrayList();
        if (RebalanceUtils.containsNode(currentCluster, stealNodeId)) {
            currentList = currentCluster.getNodeById(stealNodeId).getPartitionIds();
        }
        targetList.removeAll(currentList);
        return targetList;
    }

    public static Map<Integer, Set<Pair<Integer, Integer>>> getStolenPartitionTuples(Cluster currentCluster, Cluster targetCluster, StoreDefinition storeDef) {
        Map<Integer, Set<Pair<Integer, Integer>>> currentNodeIdToReplicas = RebalanceUtils.getNodeIdToAllPartitions(currentCluster, storeDef, true);
        Map<Integer, Set<Pair<Integer, Integer>>> targetNodeIdToReplicas = RebalanceUtils.getNodeIdToAllPartitions(targetCluster, storeDef, true);
        HashMap stealerNodeToStolenPartitionTuples = Maps.newHashMap();
        for (int stealerId : RebalanceUtils.getNodeIds(Lists.newArrayList(targetCluster.getNodes()))) {
            Set<Pair<Integer, Integer>> targetStealerReplicas;
            Set<Pair<Integer, Integer>> clusterStealerReplicas = currentNodeIdToReplicas.get(stealerId);
            Set<Pair<Integer, Integer>> diff = Utils.getAddedInTarget(clusterStealerReplicas, targetStealerReplicas = targetNodeIdToReplicas.get(stealerId));
            if (diff == null || diff.size() <= 0) continue;
            stealerNodeToStolenPartitionTuples.put(stealerId, diff);
        }
        return stealerNodeToStolenPartitionTuples;
    }

    public static void combinePartitionTuples(Map<Integer, Set<Pair<Integer, Integer>>> existingPartitionTuples, Map<Integer, Set<Pair<Integer, Integer>>> newPartitionTuples) {
        for (int nodeId : newPartitionTuples.keySet()) {
            Set<Object> tuples = null;
            if (existingPartitionTuples.containsKey(nodeId)) {
                tuples = existingPartitionTuples.get(nodeId);
            } else {
                tuples = Sets.newHashSet();
                existingPartitionTuples.put(nodeId, (Set<Pair<Integer, Integer>>)tuples);
            }
            tuples.addAll((Collection)newPartitionTuples.get(nodeId));
        }
    }

    public static Map<Integer, Set<Pair<Integer, Integer>>> getNodeIdToAllPartitions(Cluster cluster, StoreDefinition storeDef, boolean includePrimary) {
        RoutingStrategy routingStrategy = new RoutingStrategyFactory().updateRoutingStrategy(storeDef, cluster);
        HashMap<Integer, Set<Pair<Integer, Integer>>> nodeIdToReplicas = new HashMap<Integer, Set<Pair<Integer, Integer>>>();
        Map<Integer, Integer> partitionToNodeIdMap = RebalanceUtils.getCurrentPartitionMapping(cluster);
        for (Node node : cluster.getNodes()) {
            nodeIdToReplicas.put(node.getId(), new HashSet());
        }
        for (Node node : cluster.getNodes()) {
            for (Integer primary : node.getPartitionIds()) {
                List<Integer> replicaPartitionList = routingStrategy.getReplicatingPartitionList(primary);
                if (replicaPartitionList.size() != storeDef.getReplicationFactor()) {
                    throw new VoldemortException("Number of replicas returned (" + replicaPartitionList.size() + ") is less than the required replication factor (" + storeDef.getReplicationFactor() + ")");
                }
                int replicaType = 0;
                if (!includePrimary) {
                    replicaPartitionList.remove(primary);
                    replicaType = 1;
                }
                for (Integer replicaPartition : replicaPartitionList) {
                    Integer replicaNodeId = partitionToNodeIdMap.get(replicaPartition);
                    ((Set)nodeIdToReplicas.get(replicaNodeId)).add(Pair.create(replicaType, primary));
                    ++replicaType;
                }
            }
        }
        return nodeIdToReplicas;
    }

    public static void dumpCluster(Cluster initialCluster, Cluster finalCluster, File outputDir) {
        if (!outputDir.exists()) {
            Utils.mkdirs(outputDir);
        }
        File initialClusterFile = new File(outputDir, initialClusterFileName);
        File finalClusterFile = new File(outputDir, finalClusterFileName);
        ClusterMapper mapper = new ClusterMapper();
        try {
            FileUtils.writeStringToFile((File)initialClusterFile, (String)mapper.writeCluster(initialCluster));
            FileUtils.writeStringToFile((File)finalClusterFile, (String)mapper.writeCluster(finalCluster));
        }
        catch (IOException e) {
            logger.error((Object)"Error writing cluster metadata to file");
        }
    }

    public static void printLog(int taskId, Logger logger, String message) {
        logger.info((Object)("Task id [" + Integer.toString(taskId) + "] " + message));
    }

    public static void printErrorLog(int taskId, Logger logger, String message, Exception e) {
        if (e == null) {
            logger.error((Object)("Task id " + Integer.toString(taskId) + "] " + message));
        } else {
            logger.error((Object)("Task id " + Integer.toString(taskId) + "] " + message), (Throwable)e);
        }
    }

    public static Node getNodeByPartitionId(Cluster cluster, int partitionId) {
        for (Node node : cluster.getNodes()) {
            if (!node.getPartitionIds().contains(partitionId)) continue;
            return node;
        }
        return null;
    }

    public static AdminClient createTempAdminClient(VoldemortConfig voldemortConfig, Cluster cluster, int numConnPerNode) {
        AdminClientConfig config = new AdminClientConfig().setMaxConnectionsPerNode(numConnPerNode).setAdminConnectionTimeoutSec(voldemortConfig.getAdminConnectionTimeout()).setAdminSocketTimeoutSec(voldemortConfig.getAdminSocketTimeout()).setAdminSocketBufferSize(voldemortConfig.getAdminSocketBufferSize());
        return new AdminClient(cluster, config);
    }

    public static List<StoreDefinition> getStoreDefinition(Cluster cluster, AdminClient adminClient) {
        List<StoreDefinition> storeDefs = null;
        for (Node node : cluster.getNodes()) {
            List<StoreDefinition> storeDefList = adminClient.getRemoteStoreDefList(node.getId()).getValue();
            if (storeDefs == null) {
                storeDefs = storeDefList;
                continue;
            }
            if (Utils.compareList(storeDefs, storeDefList)) continue;
            throw new VoldemortException("Store definitions on node " + node.getId() + " does not match those on other nodes");
        }
        if (storeDefs == null) {
            throw new VoldemortException("Could not retrieve list of store definitions correctly");
        }
        return storeDefs;
    }

    public static List<StoreDefinition> validateRebalanceStore(List<StoreDefinition> storeDefList) {
        ArrayList<StoreDefinition> returnList = new ArrayList<StoreDefinition>(storeDefList.size());
        for (StoreDefinition def : storeDefList) {
            if (!def.isView() && !canRebalanceList.contains(def.getType())) {
                throw new VoldemortException("Rebalance does not support rebalancing of stores of type " + def.getType() + " - " + def.getName());
            }
            if (!def.isView()) {
                returnList.add(def);
                continue;
            }
            logger.debug((Object)("Ignoring view " + def.getName() + " for rebalancing"));
        }
        return returnList;
    }

    public static void validateReadOnlyStores(Cluster cluster, List<StoreDefinition> storeDefs, AdminClient adminClient) {
        List<StoreDefinition> readOnlyStores = RebalanceUtils.filterStores(storeDefs, true);
        if (readOnlyStores.size() == 0) {
            return;
        }
        List<String> storeNames = RebalanceUtils.getStoreNames(readOnlyStores);
        for (Node node : cluster.getNodes()) {
            if (node.getNumberOfPartitions() == 0) continue;
            for (Map.Entry<String, String> storeToStorageFormat : adminClient.getROStorageFormat(node.getId(), storeNames).entrySet()) {
                if (storeToStorageFormat.getValue().compareTo(ReadOnlyStorageFormat.READONLY_V2.getCode()) == 0) continue;
                throw new VoldemortRebalancingException("Cannot rebalance since node " + node.getId() + " has store " + storeToStorageFormat.getKey() + " not using format " + (Object)((Object)ReadOnlyStorageFormat.READONLY_V2));
            }
        }
    }

    public static String printMap(Map<Integer, Set<Pair<Integer, Integer>>> nodeIdToAllPartitions) {
        StringBuilder sb = new StringBuilder();
        for (Map.Entry<Integer, Set<Pair<Integer, Integer>>> entry : nodeIdToAllPartitions.entrySet()) {
            Integer nodeId = entry.getKey();
            Set<Pair<Integer, Integer>> allPartitions = entry.getValue();
            HashMap<Integer, List<Integer>> replicaTypeToPartitions = RebalanceUtils.flattenPartitionTuples(allPartitions);
            TreeMap<Integer, List<Integer>> sortedReplicaTypeToPartitions = new TreeMap<Integer, List<Integer>>(replicaTypeToPartitions);
            sb.append(nodeId);
            if (replicaTypeToPartitions.size() > 0) {
                for (Map.Entry<Integer, List<Integer>> partitions : sortedReplicaTypeToPartitions.entrySet()) {
                    Collections.sort(partitions.getValue());
                    sb.append(" - " + partitions.getValue());
                }
            } else {
                sb.append(" - empty");
            }
            sb.append(Utils.NEWLINE);
        }
        return sb.toString();
    }

    public static HashMap<Integer, List<Integer>> flattenPartitionTuples(Set<Pair<Integer, Integer>> partitionTuples) {
        HashMap flattenedTuples = Maps.newHashMap();
        for (Pair<Integer, Integer> pair : partitionTuples) {
            if (flattenedTuples.containsKey(pair.getFirst())) {
                ((List)flattenedTuples.get(pair.getFirst())).add(pair.getSecond());
                continue;
            }
            ArrayList newPartitions = Lists.newArrayList();
            newPartitions.add(pair.getSecond());
            flattenedTuples.put(pair.getFirst(), newPartitions);
        }
        return flattenedTuples;
    }

    public static Set<Pair<Integer, Integer>> flattenPartitionTuples(HashMap<Integer, List<Integer>> replicaToPartitionList) {
        HashSet partitionTuples = Sets.newHashSet();
        for (Map.Entry<Integer, List<Integer>> entry : replicaToPartitionList.entrySet()) {
            Iterator<Integer> iter = entry.getValue().iterator();
            while (iter.hasNext()) {
                partitionTuples.add(new Pair<Integer, Integer>(entry.getKey(), iter.next()));
            }
        }
        return partitionTuples;
    }

    public static List<RebalancePartitionsInfo> flattenNodePlans(List<RebalanceNodePlan> rebalanceNodePlanList) {
        ArrayList<RebalancePartitionsInfo> list = new ArrayList<RebalancePartitionsInfo>();
        for (RebalanceNodePlan rebalanceNodePlan : rebalanceNodePlanList) {
            for (RebalancePartitionsInfo stealInfo : rebalanceNodePlan.getRebalanceTaskList()) {
                list.add(stealInfo);
            }
        }
        return list;
    }

    public static List<Integer> getPartitionsFromTuples(Set<Pair<Integer, Integer>> tuples) {
        ArrayList partitions = Lists.newArrayList();
        if (tuples != null) {
            for (Pair<Integer, Integer> tuple : tuples) {
                partitions.add(tuple.getSecond());
            }
        }
        return partitions;
    }

    public static List<RebalancePartitionsInfo> filterPartitionPlanWithStores(List<RebalancePartitionsInfo> existingPlanList, List<StoreDefinition> storeDefs) {
        ArrayList plans = Lists.newArrayList();
        List<String> storeNames = RebalanceUtils.getStoreNames(storeDefs);
        for (RebalancePartitionsInfo existingPlan : existingPlanList) {
            RebalancePartitionsInfo info = RebalancePartitionsInfo.create(existingPlan.toJsonString());
            HashMap<String, HashMap<Integer, List<Integer>>> storeToReplicaToAddPartitions = info.getStoreToReplicaToAddPartitionList();
            HashMap<String, HashMap<Integer, List<Integer>>> storeToReplicaToDeletePartitions = info.getStoreToReplicaToDeletePartitionList();
            HashMap newStoreToReplicaToAddPartitions = Maps.newHashMap();
            HashMap newStoreToReplicaToDeletePartitions = Maps.newHashMap();
            for (String storeName : storeNames) {
                if (storeToReplicaToAddPartitions.containsKey(storeName)) {
                    newStoreToReplicaToAddPartitions.put(storeName, storeToReplicaToAddPartitions.get(storeName));
                }
                if (!storeToReplicaToDeletePartitions.containsKey(storeName)) continue;
                newStoreToReplicaToDeletePartitions.put(storeName, storeToReplicaToDeletePartitions.get(storeName));
            }
            info.setStoreToReplicaToAddPartitionList(newStoreToReplicaToAddPartitions);
            info.setStoreToReplicaToDeletePartitionList(newStoreToReplicaToDeletePartitions);
            plans.add(info);
        }
        return plans;
    }

    public static HashMap<Integer, List<RebalancePartitionsInfo>> groupPartitionsInfoByNode(List<RebalancePartitionsInfo> rebalancePartitionPlanList, boolean groupByStealerNode) {
        HashMap nodeToPartitionsInfo = Maps.newHashMap();
        if (rebalancePartitionPlanList != null) {
            for (RebalancePartitionsInfo partitionInfo : rebalancePartitionPlanList) {
                int nodeId = groupByStealerNode ? partitionInfo.getStealerId() : partitionInfo.getDonorId();
                List partitionInfos = (List)nodeToPartitionsInfo.get(nodeId);
                if (partitionInfos == null) {
                    partitionInfos = Lists.newArrayList();
                    nodeToPartitionsInfo.put(nodeId, partitionInfos);
                }
                partitionInfos.add(partitionInfo);
            }
        }
        return nodeToPartitionsInfo;
    }

    public static StoreDefinition getStoreDefinitionWithName(List<StoreDefinition> storeDefs, String storeName) {
        StoreDefinition def = null;
        for (StoreDefinition storeDef : storeDefs) {
            if (storeDef.getName().compareTo(storeName) != 0) continue;
            def = storeDef;
        }
        if (def == null) {
            throw new VoldemortException("Could not find store " + storeName);
        }
        return def;
    }

    public static List<StoreDefinition> filterStores(List<StoreDefinition> storeDefs, boolean isReadOnly) {
        ArrayList filteredStores = Lists.newArrayList();
        for (StoreDefinition storeDef : storeDefs) {
            if (storeDef.getType().equals("read-only") != isReadOnly) continue;
            filteredStores.add(storeDef);
        }
        return filteredStores;
    }

    public static List<String> getStoreNames(List<StoreDefinition> storeDefList) {
        ArrayList<String> storeList = new ArrayList<String>();
        for (StoreDefinition def : storeDefList) {
            storeList.add(def.getName());
        }
        return storeList;
    }

    public static List<Integer> getNodeIds(List<Node> nodes) {
        ArrayList<Integer> nodeIds = new ArrayList<Integer>(nodes.size());
        for (Node node : nodes) {
            nodeIds.add(node.getId());
        }
        return nodeIds;
    }

    public static void executorShutDown(ExecutorService executorService, long timeOutSec) {
        try {
            executorService.shutdown();
            executorService.awaitTermination(timeOutSec, TimeUnit.SECONDS);
        }
        catch (Exception e) {
            logger.warn((Object)"Error while stoping executor service.", (Throwable)e);
        }
    }

    public static ExecutorService createExecutors(int numThreads) {
        return Executors.newFixedThreadPool(numThreads, new ThreadFactory(){

            public Thread newThread(Runnable r) {
                Thread thread = new Thread(r);
                thread.setName(r.getClass().getName());
                return thread;
            }
        });
    }
}

