/*
 * Copyright 2017 Red Hat, Inc. and/or its affiliates
 * and other contributors as indicated by the @author tags.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.keycloak.authorization.jpa.store;

import static org.keycloak.authorization.UserManagedPermissionUtil.updatePolicy;

import javax.persistence.EntityManager;

import org.keycloak.authorization.jpa.entities.PermissionTicketEntity;
import org.keycloak.authorization.jpa.entities.PolicyEntity;
import org.keycloak.authorization.jpa.entities.ScopeEntity;
import org.keycloak.authorization.model.PermissionTicket;
import org.keycloak.authorization.model.Policy;
import org.keycloak.authorization.model.Resource;
import org.keycloak.authorization.model.ResourceServer;
import org.keycloak.authorization.model.Scope;
import org.keycloak.authorization.store.StoreFactory;
import org.keycloak.models.jpa.JpaModel;

import br.ufrgs.inf.prosoft.cache.*;

/**
 * @author <a href="mailto:bill@burkecentral.com">Bill Burke</a>
 * @version $Revision: 1 $
 */
public class PermissionTicketAdapter implements PermissionTicket, JpaModel<PermissionTicketEntity> {

    private final EntityManager entityManager;
    private final PermissionTicketEntity entity;
    private final StoreFactory storeFactory;

    public PermissionTicketAdapter(PermissionTicketEntity entity, EntityManager entityManager, StoreFactory storeFactory) {
        this.entity = entity;
        this.entityManager = entityManager;
        this.storeFactory = storeFactory;
    }

public static GetterCache<PermissionTicketEntity> getEntity = new GetterCache<>("PermissionTicketAdapter.getEntity");

    @Override
    public PermissionTicketEntity getEntity() {
        return getEntity.computeIfAbsent(() -> {
        return entity;
        }, 120000);
    }

    @Override
    public String getId() {
        return getEntity().getId();
    }

    @Override
    public String getOwner() {
        return getEntity().getOwner();
    }

    @Override
    public String getRequester() {
        return getEntity().getRequester();
    }

    @Override
    public boolean isGranted() {
        return getEntity().isGranted();
    }

    @Override
    public Long getCreatedTimestamp() {
        return getEntity().getCreatedTimestamp();
    }

    @Override
    public Long getGrantedTimestamp() {
        return getEntity().getGrantedTimestamp();
    }

    @Override
    public void setGrantedTimestamp(Long millis) {
        getEntity.invalidate();
        entity.setGrantedTimestamp(millis);
        updatePolicy(this, storeFactory);
    }

    @Override
    public ResourceServer getResourceServer() {
        return storeFactory.getResourceServerStore().findById(getEntity().getResourceServer().getId());
    }

    @Override
    public Policy getPolicy() {
        PolicyEntity policy = getEntity().getPolicy();

        if (policy == null) {
            return null;
        }

        return storeFactory.getPolicyStore().findById(policy.getId(), getEntity().getResourceServer().getId());
    }

    @Override
    public void setPolicy(Policy policy) {
        if (policy != null) {
            getEntity.invalidate();
            entity.setPolicy(entityManager.getReference(PolicyEntity.class, policy.getId()));
        }
    }

    @Override
    public Resource getResource() {
        return storeFactory.getResourceStore().findById(getEntity().getResource().getId(), getResourceServer().getId());
    }

    @Override
    public Scope getScope() {
        ScopeEntity scope = getEntity().getScope();

        if (scope == null) {
            return null;
        }

        return storeFactory.getScopeStore().findById(scope.getId(), getResourceServer().getId());
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (o == null || !(o instanceof Policy)) return false;

        PermissionTicket that = (PermissionTicket) o;
        return that.getId().equals(getId());
    }

    @Override
    public int hashCode() {
        return getId().hashCode();
    }

    public static PermissionTicketEntity toEntity(EntityManager em, PermissionTicket permission) {
        if (permission instanceof PermissionTicketAdapter) {
            return ((PermissionTicketAdapter)permission).getEntity();
        } else {
            return em.getReference(PermissionTicketEntity.class, permission.getId());
        }
    }



}
