package br.ufrgs.inf.prosoft.adaptivecaching.analysis.decision.flowchart;

import br.ufrgs.inf.prosoft.adaptivecaching.analysis.decision.flowchart.stats.Thresholds;
import br.ufrgs.inf.prosoft.adaptivecaching.monitoring.application.metadata.Method;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;
import java.util.logging.Level;
import java.util.logging.Logger;

public final class FlowchartWorkFlow {

    private static final Logger logger = Logger.getLogger(FlowchartWorkFlow.class.getName());

    private List<Method> methods;

    public FlowchartWorkFlow setMethods(List<Method> methods) {
        this.methods = methods;
        return this;
    }

    private void countStats() {
        logger.log(Level.INFO, "Counting stats of {0} methods", this.methods.size());
        Collections.sort(this.methods, (m1, m2) -> Integer.compare(m1.getOccurrencesSize(), m2.getOccurrencesSize()));
        this.methods.stream().parallel().forEach(Method::calculateMetrics);
    }

    private void calculateThresholds() {
        logger.log(Level.INFO, "Calculating thresholds");
        Thresholds.reset();
        Thresholds.population = getPopulation();
        this.methods.stream().forEach(Method::calculateThresholds);

        logger.log(Level.INFO, "Average ExecutionTime: {0}", Thresholds.getAverageExecutionTime());
        logger.log(Level.INFO, "Average HitRatio: {0}", Thresholds.getAverageHitRatio());
        logger.log(Level.INFO, "Average MissRatio: {0}", Thresholds.getAverageMissRatio());
        logger.log(Level.INFO, "Average shareability: {0}", Thresholds.getAverageShareability());
        logger.log(Level.INFO, "StdDv ExecutionTime: {0}", Thresholds.getStdDevExecutionTimeRatio());
        logger.log(Level.INFO, "StdDv HitRatio: {0}", Thresholds.getStdDevHitRatio());
        logger.log(Level.INFO, "StdDv MissRatio: {0}", Thresholds.getStdDevMissRatio());
        logger.log(Level.INFO, "StdDv shareability: {0}", Thresholds.getStdDevShareability());
        logger.log(Level.INFO, "StdDv frequency: {0}", Thresholds.getStdDevFrequency());

        int k = 0;
        logger.log(Level.INFO, "Using {0} stdDev to calculate thresholds...", k);
        logger.log(Level.INFO, "Threshold ExecutionTime: {0}", Thresholds.expensivenessThreshold(k));
        logger.log(Level.INFO, "Threshold HitRatio: {0}", Thresholds.hitThreshold(k));
        logger.log(Level.INFO, "Threshold MissRatio: {0}", Thresholds.missThreshold(k));
        logger.log(Level.INFO, "Threshold Shareability: {0}", Thresholds.shareabilityThreshold(k));
        logger.log(Level.INFO, "Threshold frequency: {0}", Thresholds.frequencyThreshold(k));
    }

    public void filterCacheableInputs() {
        countStats();
        calculateThresholds();

        logger.log(Level.INFO, "Deciding if methods are cacheable...");

        Iterator<Method> iterator = this.methods.iterator();
        while (iterator.hasNext()) {
            Method method = iterator.next();
            method.filterCacheableInputs();
            if (method.getGroupsOfOccurrences().isEmpty()) {
                iterator.remove();
            }
        }

        logger.log(Level.INFO, "{0} cacheable methods detected", this.methods.size());
    }

    private long getPopulation() {
        return this.methods.stream().parallel()
                .map(Method::getOccurrencesSize)
                .reduce(Integer::sum)
                .get();
    }
}
