package br.ufrgs.inf.prosoft.adaptivecaching.analysis.decision.flowchart;

import br.ufrgs.inf.prosoft.adaptivecaching.analysis.decision.CacheDecider;
import br.ufrgs.inf.prosoft.adaptivecaching.analysis.decision.flowchart.model.MethodEntry;
import br.ufrgs.inf.prosoft.adaptivecaching.analysis.decision.flowchart.model.MethodStats;
import br.ufrgs.inf.prosoft.adaptivecaching.analysis.decision.flowchart.stats.CacheabilityMetrics;
import br.ufrgs.inf.prosoft.adaptivecaching.analysis.decision.flowchart.stats.CacheabilityPatternDecider;
import br.ufrgs.inf.prosoft.adaptivecaching.monitoring.application.metadata.LogTrace;
import br.ufrgs.inf.prosoft.adaptivecaching.monitoring.application.metadata.MethodInfo;
import org.apache.commons.lang3.builder.EqualsBuilder;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.PrintWriter;
import java.math.BigDecimal;
import java.math.RoundingMode;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Set;
import java.util.logging.Level;
import java.util.stream.Collectors;

public class FlowchartWorkFlow {

    Logger logger = LoggerFactory.getLogger(FlowchartWorkFlow.class);

    private final long population;
    protected HashMap<MethodInfo, MethodStats> methodsInfoMap;

    private CacheDecider decider;

    private double sumMissRatio;
    private double sumHitRatio;
    private double sumExecutionTime;
    private double sumShareability;
    private double sumFrequency;
    private List<Double> hitRatios = new ArrayList<>();
    private List<Double> missRatios = new ArrayList<>();
    private List<Long> executionTimes = new ArrayList<>();
    private List<Double> shareabilities = new ArrayList<>();
    private List<Long> frequencies = new ArrayList<>();

    public FlowchartWorkFlow(List<LogTrace> logList) {
        this.decider = new CacheabilityPatternDecider(logList.size(), this);
        this.methodsInfoMap = countOccurrences(logList);
        this.population = logList.size();

        logger.debug(methodsInfoMap.size() + " unique method calls identified from " + logList.size() + " original traces");

        logger.debug("Average ExecutionTime: " + getAverageExecutionTime());
        logger.debug("Average HitRatio: " + getAverageHitRatio());
        logger.debug("Average MissRatio: " + getAverageMissRatio());
        logger.debug("Average shareability: " + getAverageShareability());
        logger.debug("StdDv ExecutionTime: " + getStdDevExecutionTimeRatio());
        logger.debug("StdDv HitRatio: " + getStdDevHitRatio());
        logger.debug("StdDv MissRatio: " + getStdDevMissRatio());
        logger.debug("StdDv shareability: " + getStdDevShareability());
        logger.debug("StdDv frequency: " + getStdDevFrequency());

        int k = 0;
        logger.debug("Using " + k + " stdDev to calculate thresholds...");
        logger.debug("Threshold ExecutionTime: " + expensivenessThreshold(k));
        logger.debug("Threshold HitRatio: " + hitThreshold(k));
        logger.debug("Threshold MissRatio: " + missThreshold(k));
        logger.debug("Threshold Shareability: " + shareabilityThreshold(k));
        logger.debug("Threshold frequency: " + frequencyThreshold(k));
    }

    public Set<MethodEntry> filterCacheableMethods(long expiryTime) {
        logger.debug("Deciding if methods are cacheable...");

        Set<MethodEntry> cacheableMethods = getMethodsInfoMap().keySet().stream()
                .filter(mi -> decider.isCacheable(mi, getMethodsInfoMap().get(mi)))
                .map(mi -> new MethodEntry(mi, getMethodsInfoMap().get(mi), System.currentTimeMillis() + expiryTime))
                .collect(Collectors.toSet());

        logger.info(cacheableMethods.size() + " cacheable methods detected. Printing files...");

        //TODO remove: print all unique methods and metrics to csv file
        try (PrintWriter pw = new PrintWriter(new File("allmethods.csv"))) {
            pw.write("isStaticData,changeMoreThanUsed,usedByManyRequests,isUserSpecific,isCacheSizeLarge,isDataSizeLarge,isExpensive,signature,numberOfSameOccurrences,numberOfDifferentReturnOccurrences,totalOccurrences,sameOccurrencesAverageExecutionTime,sameOccurrencesTotalExecutionTime,hitRatio,missRatio\n");
            getMethodsInfoMap().keySet().stream().forEach(mi -> pw.write(CacheabilityMetrics.allMetricsToString(mi, getMethodsInfoMap().get(mi), this, getMethodsInfoMap().size()) + "," + new MethodEntry(mi, getMethodsInfoMap().get(mi), System.currentTimeMillis() + expiryTime).getMethodInfo().getSignature() + "," + new MethodEntry(mi, getMethodsInfoMap().get(mi), System.currentTimeMillis() + expiryTime).getMethodStats().toCSV() + '\n'));
        } catch (FileNotFoundException ex) {
        }

        //TODO remove: print cacheable methods to csv file
        try (PrintWriter pw = new PrintWriter(new File("cacheablemethods.csv"))) {
            pw.write("isStaticData,changeMoreThanUsed,usedByManyRequests,isUserSpecific,isCacheSizeLarge,isDataSizeLarge,isExpensive,signature,numberOfSameOccurrences,numberOfDifferentReturnOccurrences,totalOccurrences,sameOccurrencesAverageExecutionTime,sameOccurrencesTotalExecutionTime,hitRatio,missRatio\n");
            cacheableMethods.stream().forEach(ma -> pw.write(CacheabilityMetrics.allMetricsToString(ma.getMethodInfo(), ma.getMethodStats(), this, getMethodsInfoMap().size()) + "," + ma.getMethodInfo().getSignature() + "," + ma.getMethodStats().toCSV() + '\n'));
        } catch (FileNotFoundException ex) {
        }
        return cacheableMethods;
    }

    public HashMap<MethodInfo, MethodStats> countOccurrences(List<LogTrace> logs) {
        logger.info("Counting {0} occurrences", logs.size());
        sumExecutionTime = 0;
        sumHitRatio = 0;
        sumMissRatio = 0;
        sumShareability = 0;
        sumFrequency = 0;

        HashMap<MethodInfo, MethodStats> methodInfoMap = new HashMap<>();

        for (int i = 0; i < logs.size(); i++) {
            LogTrace logTrace = logs.get(i);
            System.out.print(".");
            System.out.flush();
            if (i != 0 && i % 100 == 0) {
                System.out.println();
            }
            if (methodInfoMap.containsKey(logTrace.getMethodInfo())) {
                continue;
            }

            MethodStats methodStats = new MethodStats(logTrace);

            for (int j = 0; j < logs.size(); j++) {
                LogTrace traceCompare = logs.get(j);

                if (i == j) {
                    continue;
                }

                //if similar methods: same signature and params, different return
                if (traceCompare.getMethodInfo().equalsWithoutReturnedValue(logTrace.getMethodInfo())) {
                    //if identical methods
                    if (EqualsBuilder.reflectionEquals(traceCompare.getMethodInfo().getReturnedValue(), logTrace.getMethodInfo().getReturnedValue())) {
                        methodStats.addSameOccurrence(traceCompare);
                    } else {
                        methodStats.addDifferentReturnOccurrence();
                    }
                }
            }

            methodInfoMap.put(logTrace.getMethodInfo(), methodStats);

            sumExecutionTime += methodStats.getSameOccurrencesTotalExecutionTime();
            executionTimes.add(methodStats.getSameOccurrencesTotalExecutionTime());

            sumHitRatio += methodStats.hitRatio();
            hitRatios.add(methodStats.hitRatio());

            sumMissRatio += methodStats.missRatio();
            missRatios.add(methodStats.missRatio());

            sumShareability += methodStats.shareability();
            shareabilities.add(methodStats.shareability());

            sumFrequency += methodStats.getNumberOfSameOccurrences();
            frequencies.add(methodStats.getNumberOfSameOccurrences());
        }

        return methodInfoMap;
    }

    /**
     * General mean hit ratio of all calls
     *
     * @return
     */
    public double getAverageHitRatio() {
        if (population == 0) {
            return 0;
        }
        return new BigDecimal(sumHitRatio).divide(new BigDecimal(population), 5, RoundingMode.HALF_UP).doubleValue();
    }

    public double getAverageMissRatio() {
        if (population == 0) {
            return 0;
        }
        return new BigDecimal(sumMissRatio).divide(new BigDecimal(population), 5, RoundingMode.HALF_UP).doubleValue();
    }

    public double getAverageExecutionTime() {
        if (population == 0) {
            return 0;
        }
        return new BigDecimal(sumExecutionTime).divide(new BigDecimal(population), 5, RoundingMode.HALF_UP).doubleValue();
    }

    public double getAverageShareability() {
        if (population == 0) {
            return 0;
        }
        return new BigDecimal(sumShareability).divide(new BigDecimal(population), 5, RoundingMode.HALF_UP).doubleValue();
    }

    private Double StdDevHitRatio;

    public double getStdDevHitRatio() {
        if (population == 0) {
            return 0;
        }
        if (StdDevHitRatio != null) {
            return StdDevHitRatio;
        }

        double mean = getAverageHitRatio();
        double temp = 0;
        for (double a : hitRatios) {
            temp += (a - mean) * (a - mean);
        }
        StdDevHitRatio = Math.sqrt(temp / population);
        return StdDevHitRatio;
    }

    private Double StdDevMissRatio;

    public double getStdDevMissRatio() {
        if (population == 0) {
            return 0;
        }
        if (StdDevMissRatio != null) {
            return StdDevMissRatio;
        }

        double mean = getAverageMissRatio();
        double temp = 0;
        for (double a : missRatios) {
            temp += (a - mean) * (a - mean);
        }
        StdDevMissRatio = Math.sqrt(temp / population);
        return StdDevMissRatio;
    }

    private Double StdDevExecutionTimeRatio;

    public double getStdDevExecutionTimeRatio() {
        if (population == 0) {
            return 0;
        }
        if (StdDevExecutionTimeRatio != null) {
            return StdDevExecutionTimeRatio;
        }

        double mean = getAverageExecutionTime();
        double temp = executionTimes.stream()
                .map(executionTime -> (executionTime - mean) * (executionTime - mean))
                .reduce(Double::sum).get();
        StdDevExecutionTimeRatio = Math.sqrt(temp / population);
        return StdDevExecutionTimeRatio;
    }

    private Double StdDevFrequency;

    public double getStdDevFrequency() {
        if (population == 0) {
            return 0;
        }
        if (StdDevFrequency != null) {
            return StdDevFrequency;
        }

        double mean = getAverageFrequency();
        double temp = frequencies.stream()
                .map(frequency -> (frequency - mean) * (frequency - mean))
                .reduce(Double::sum).get();
        StdDevFrequency = Math.sqrt(temp / population);
        return StdDevFrequency;
    }

    private Double StdDevShareability;

    public double getStdDevShareability() {
        if (population == 0) {
            return 0;
        }
        if (StdDevShareability != null) {
            return StdDevShareability;
        }

        double mean = getAverageShareability();
        double temp = shareabilities.stream()
                .map((shareability) -> (shareability - mean) * (shareability - mean))
                .reduce(Double::sum).get();
        StdDevShareability = Math.sqrt(temp / population);
        return StdDevShareability;
    }

    public HashMap<MethodInfo, MethodStats> getMethodsInfoMap() {
        return methodsInfoMap;
    }

    //getting X% with most hits
    public double hitThreshold(int kStdDev) {
        return getAverageHitRatio() + (kStdDev * getStdDevHitRatio());
    }

    //getting X% with most misses
    public double missThreshold(int kStdDev) {
        return getAverageMissRatio() + (kStdDev * getStdDevMissRatio());
    }

    //getting X% most expensive methods
    public double expensivenessThreshold(int kStdDev) {
        return getAverageExecutionTime() + (kStdDev * getStdDevExecutionTimeRatio());
    }

    public double shareabilityThreshold(int kStdDev) {
        return getAverageShareability() + (kStdDev * getStdDevShareability());
    }

    //getting X% most frenquent
    public double frequencyThreshold(int kStdDev) {
        return getAverageFrequency() + (kStdDev * getStdDevFrequency());
    }

    /**
     * General miss ratio from a signature
     *
     * @param signature
     * @return
     */
    public double getMissRatio(String signature) {
        long occurrences = 0;
        long methods = 0;
        for (MethodInfo mi : methodsInfoMap.keySet()) {
            if (mi.getSignature().equals(signature)) {
                occurrences += methodsInfoMap.get(mi).getNumberOfSameOccurrences();
                methods++;
            }
        }
        return occurrences / methods;
    }

    /**
     * General hit ratio from a signature
     *
     * @param signature
     * @return
     */
    public double getHitRatio(String signature) {
        long occurrences = 0;
        long methods = 0;
        for (MethodInfo mi : methodsInfoMap.keySet()) {
            if (mi.getSignature().equals(signature)) {
                occurrences += methodsInfoMap.get(mi).getNumberOfSameOccurrences();
                methods++;
            }
        }
        return occurrences / methods;
    }

    private Double AverageFrequency;

    public double getAverageFrequency() {
        if (frequencies.isEmpty()) {
            return 0;
        }
        if (AverageFrequency != null) {
            return AverageFrequency;
        }

        AverageFrequency = new BigDecimal(sumFrequency).divide(new BigDecimal(frequencies.size()), 5, RoundingMode.HALF_UP).doubleValue();
        return AverageFrequency;
    }
}
