/*
 * Copyright 2010-2011 Ning, Inc.
 *
 * Ning licenses this file to you under the Apache License, version 2.0
 * (the "License"); you may not use this file except in compliance with the
 * License.  You may obtain a copy of the License at:
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.  See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

package com.ning.billing.entitlement.api.billing;

import java.math.BigDecimal;
import java.util.Iterator;
import java.util.SortedSet;
import java.util.TreeSet;
import java.util.UUID;

import javax.annotation.Nullable;

import org.joda.time.DateTime;
import org.testng.Assert;
import org.testng.annotations.Test;

import com.ning.billing.catalog.DefaultPrice;
import com.ning.billing.catalog.MockInternationalPrice;
import com.ning.billing.catalog.MockPlan;
import com.ning.billing.catalog.MockPlanPhase;
import com.ning.billing.catalog.api.BillingPeriod;
import com.ning.billing.catalog.api.Currency;
import com.ning.billing.catalog.api.InternationalPrice;
import com.ning.billing.catalog.api.PhaseType;
import com.ning.billing.catalog.api.Plan;
import com.ning.billing.catalog.api.PlanPhase;
import com.ning.billing.entitlement.api.user.Subscription;
import com.ning.billing.entitlement.api.user.SubscriptionTransition.SubscriptionTransitionType;

public class TestDefaultBillingEvent {
	public static final UUID ID_ZERO = new UUID(0L,0L);
	public static final UUID ID_ONE = new UUID(0L,1L);
	public static final UUID ID_TWO = new UUID(0L,2L);

	@Test(groups={"fast"})
	public void testEventOrderingSubscription() {
	
		BillingEvent event0 = createEvent(subscription(ID_ZERO), new DateTime("2012-01-31T00:02:04.000Z"), SubscriptionTransitionType.CREATE);
		BillingEvent event1 = createEvent(subscription(ID_ONE), new DateTime("2012-01-31T00:02:04.000Z"), SubscriptionTransitionType.CREATE);
		BillingEvent event2 = createEvent(subscription(ID_TWO), new DateTime("2012-01-31T00:02:04.000Z"), SubscriptionTransitionType.CREATE);
		
		SortedSet<BillingEvent> set = new TreeSet<BillingEvent>();
		set.add(event2);
		set.add(event1);
		set.add(event0);
		
		Iterator<BillingEvent> it = set.iterator();
		
		Assert.assertEquals(event0, it.next());
		Assert.assertEquals(event1, it.next());
		Assert.assertEquals(event2, it.next());
	}
	
	@Test(groups={"fast"})
	public void testEventOrderingDate() {
	
		BillingEvent event0 = createEvent(subscription(ID_ZERO), new DateTime("2012-01-01T00:02:04.000Z"), SubscriptionTransitionType.CREATE);
		BillingEvent event1 = createEvent(subscription(ID_ZERO), new DateTime("2012-02-01T00:02:04.000Z"), SubscriptionTransitionType.CREATE);
		BillingEvent event2 = createEvent(subscription(ID_ZERO), new DateTime("2012-03-01T00:02:04.000Z"), SubscriptionTransitionType.CREATE);
		
		SortedSet<BillingEvent> set = new TreeSet<BillingEvent>();
		set.add(event2);
		set.add(event1);
		set.add(event0);
		
		Iterator<BillingEvent> it = set.iterator();
		
		Assert.assertEquals(event0, it.next());
		Assert.assertEquals(event1, it.next());
		Assert.assertEquals(event2, it.next());
	}
	
	@Test(groups={"fast"})
	public void testEventOrderingType() {
	
		BillingEvent event0 = createEvent(subscription(ID_ZERO), new DateTime("2012-01-01T00:02:04.000Z"), SubscriptionTransitionType.CREATE);
		BillingEvent event1 = createEvent(subscription(ID_ZERO), new DateTime("2012-01-01T00:02:04.000Z"), SubscriptionTransitionType.CHANGE);
		BillingEvent event2 = createEvent(subscription(ID_ZERO), new DateTime("2012-01-01T00:02:04.000Z"), SubscriptionTransitionType.CANCEL);
		
		SortedSet<BillingEvent> set = new TreeSet<BillingEvent>();
		set.add(event2);
		set.add(event1);
		set.add(event0);
		
		Iterator<BillingEvent> it = set.iterator();
		
		Assert.assertEquals(event0, it.next());
		Assert.assertEquals(event1, it.next());
		Assert.assertEquals(event2, it.next());
	}
	
	@Test(groups={"fast"})
	public void testEventOrderingMix() {
	
		BillingEvent event0 = createEvent(subscription(ID_ZERO), new DateTime("2012-01-01T00:02:04.000Z"), SubscriptionTransitionType.CREATE);
		BillingEvent event1 = createEvent(subscription(ID_ZERO), new DateTime("2012-01-02T00:02:04.000Z"), SubscriptionTransitionType.CHANGE);
		BillingEvent event2 = createEvent(subscription(ID_ONE), new DateTime("2012-01-01T00:02:04.000Z"), SubscriptionTransitionType.CANCEL);
		
		SortedSet<BillingEvent> set = new TreeSet<BillingEvent>();
		set.add(event2);
		set.add(event1);
		set.add(event0);
		
		Iterator<BillingEvent> it = set.iterator();
		
		Assert.assertEquals(event0, it.next());
		Assert.assertEquals(event1, it.next());
		Assert.assertEquals(event2, it.next());
	}

	
	private BillingEvent createEvent(Subscription sub, DateTime effectiveDate, SubscriptionTransitionType type) {
		InternationalPrice zeroPrice = new MockInternationalPrice(new DefaultPrice(BigDecimal.ZERO, Currency.USD));
		int billCycleDay = 1;

		Plan shotgun = new MockPlan();
		PlanPhase shotgunMonthly = createMockMonthlyPlanPhase(null, BigDecimal.ZERO, PhaseType.TRIAL);
		
		return new DefaultBillingEvent(sub , effectiveDate,
				shotgun, shotgunMonthly,
				zeroPrice, null, BillingPeriod.NO_BILLING_PERIOD, billCycleDay,
				BillingModeType.IN_ADVANCE, "Test Event 1", type);
	}

	private MockPlanPhase createMockMonthlyPlanPhase(@Nullable final BigDecimal recurringRate,
			final BigDecimal fixedRate, PhaseType phaseType) {
		return new MockPlanPhase(new MockInternationalPrice(new DefaultPrice(recurringRate, Currency.USD)),
				new MockInternationalPrice(new DefaultPrice(fixedRate, Currency.USD)),
				BillingPeriod.MONTHLY, phaseType);
	}
	
	private Subscription subscription(final UUID id) {
		return new BrainDeadSubscription() {
			public UUID getId() {
				return id;
			}
		};
	}

}
