/*
 * Copyright 2010-2013 Ning, Inc.
 *
 * Ning licenses this file to you under the Apache License, version 2.0
 * (the "License"); you may not use this file except in compliance with the
 * License.  You may obtain a copy of the License at:
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.  See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

package com.ning.billing.overdue.calculator;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.List;
import java.util.SortedSet;
import java.util.UUID;

import javax.annotation.Nullable;

import org.joda.time.DateTimeZone;
import org.joda.time.LocalDate;
import org.mockito.Mockito;
import org.testng.Assert;
import org.testng.annotations.Test;

import com.ning.billing.catalog.MockPlan;
import com.ning.billing.catalog.MockPriceList;
import com.ning.billing.catalog.api.Plan;
import com.ning.billing.catalog.api.PriceList;
import com.ning.billing.subscription.api.SubscriptionBase;
import com.ning.billing.subscription.api.user.SubscriptionBaseBundle;
import com.ning.billing.invoice.api.Invoice;
import com.ning.billing.invoice.api.InvoiceItem;
import com.ning.billing.overdue.config.api.BillingStateBundle;
import com.ning.billing.overdue.config.api.PaymentResponse;
import com.ning.billing.util.callcontext.InternalTenantContext;
import com.ning.billing.util.svcapi.subscription.SubscriptionBaseInternalApi;

import com.google.common.base.Predicate;
import com.google.common.collect.Collections2;
import com.google.common.collect.ImmutableList;

public class TestBillingStateCalculatorBundle extends TestBillingStateCalculator {

    private List<InvoiceItem> createInvoiceItems(final UUID[] bundleIds) {
        final List<InvoiceItem> result = new ArrayList<InvoiceItem>();
        for (final UUID id : bundleIds) {
            final InvoiceItem ii = Mockito.mock(InvoiceItem.class);
            Mockito.when(ii.getBundleId()).thenReturn(id);
            result.add(ii);
        }
        return result;
    }

    @Test(groups = "fast")
    public void testBillingStateAfterCancellation() throws Exception {
        Mockito.when(invoiceApi.getUnpaidInvoicesByAccountId(Mockito.<UUID>any(), Mockito.<LocalDate>any(), Mockito.<InternalTenantContext>any())).thenReturn(ImmutableList.<Invoice>of());

        final UUID bundleId = UUID.randomUUID();
        final SubscriptionBaseBundle bundle = Mockito.mock(SubscriptionBaseBundle.class);
        Mockito.when(bundle.getId()).thenReturn(bundleId);

        final SubscriptionBaseInternalApi subscriptionApi = Mockito.mock(SubscriptionBaseInternalApi.class);
        final SubscriptionBase subscription = Mockito.mock(SubscriptionBase.class);
        Mockito.when(subscriptionApi.getBaseSubscription(Mockito.eq(bundleId), Mockito.<InternalTenantContext>any())).thenReturn(subscription);

        final BillingStateCalculatorBundle calc = new BillingStateCalculatorBundle(subscriptionApi, invoiceApi, accountApi, clock);
        final BillingStateBundle billingStateBundle = calc.calculateBillingState(bundle, internalCallContext);
        Assert.assertNull(billingStateBundle.getBasePlanBillingPeriod());
        Assert.assertNull(billingStateBundle.getBasePlanPhaseType());
        Assert.assertNull(billingStateBundle.getBasePlanPriceList());
        Assert.assertNull(billingStateBundle.getBasePlanProduct());
    }

    @Test(groups = "fast")
    public void testUnpaidInvoiceForBundle() {
        final UUID thisBundleId = new UUID(0L, 0L);
        final UUID thatBundleId = new UUID(0L, 1L);

        now = new LocalDate();
        final List<Invoice> invoices = new ArrayList<Invoice>(5);
        invoices.add(createInvoice(now, BigDecimal.ZERO, createInvoiceItems(new UUID[]{thisBundleId, thatBundleId})));
        // Will not be seen below
        invoices.add(createInvoice(now.plusDays(1), BigDecimal.TEN, createInvoiceItems(new UUID[]{thatBundleId})));
        invoices.add(createInvoice(now.plusDays(2), new BigDecimal("100.00"), createInvoiceItems(new UUID[]{thatBundleId, thisBundleId, thatBundleId})));
        invoices.add(createInvoice(now.plusDays(3), new BigDecimal("1000.00"), createInvoiceItems(new UUID[]{thisBundleId})));
        invoices.add(createInvoice(now.plusDays(4), new BigDecimal("10000.00"), createInvoiceItems(new UUID[]{thatBundleId, thisBundleId})));

        Mockito.when(invoiceApi.getUnpaidInvoicesByAccountId(Mockito.<UUID>any(), Mockito.<LocalDate>any(), Mockito.<InternalTenantContext>any())).thenReturn(Collections2.filter(invoices, new Predicate<Invoice>() {
            @Override
            public boolean apply(@Nullable final Invoice invoice) {
                return invoice != null && BigDecimal.ZERO.compareTo(invoice.getBalance()) < 0;
            }
        }));

        final BillingStateCalculatorBundle calc = new BillingStateCalculatorBundle(subscriptionApi, invoiceApi, accountApi, clock);
        final SortedSet<Invoice> resultinvoices = calc.unpaidInvoicesForBundle(thisBundleId, new UUID(0L, 0L), DateTimeZone.UTC, internalCallContext);

        Assert.assertEquals(resultinvoices.size(), 3);
        Assert.assertEquals(new BigDecimal("100.0").compareTo(resultinvoices.first().getBalance()), 0);
        Assert.assertEquals(new BigDecimal("10000.0").compareTo(resultinvoices.last().getBalance()), 0);
    }

    @Test(groups = "fast")
    public void testcalculateBillingStateForBundle() throws Exception {
        final UUID thisBundleId = new UUID(0L, 0L);
        final UUID thatBundleId = new UUID(0L, 1L);

        now = new LocalDate();
        final List<Invoice> invoices = new ArrayList<Invoice>(5);
        invoices.add(createInvoice(now.minusDays(5), BigDecimal.ZERO, createInvoiceItems(new UUID[]{thisBundleId, thatBundleId})));
        invoices.add(createInvoice(now.minusDays(4), BigDecimal.TEN, createInvoiceItems(new UUID[]{thatBundleId})));
        invoices.add(createInvoice(now.minusDays(3), new BigDecimal("100.00"), createInvoiceItems(new UUID[]{thatBundleId, thisBundleId, thatBundleId})));
        invoices.add(createInvoice(now.minusDays(2), new BigDecimal("1000.00"), createInvoiceItems(new UUID[]{thisBundleId})));
        invoices.add(createInvoice(now.minusDays(1), new BigDecimal("10000.00"), createInvoiceItems(new UUID[]{thatBundleId, thisBundleId})));

        Mockito.when(invoiceApi.getUnpaidInvoicesByAccountId(Mockito.<UUID>any(), Mockito.<LocalDate>any(), Mockito.<InternalTenantContext>any())).thenReturn(invoices);

        final SubscriptionBaseBundle bundle = Mockito.mock(SubscriptionBaseBundle.class);
        Mockito.when(bundle.getId()).thenReturn(thisBundleId);
        Mockito.when(bundle.getAccountId()).thenReturn(UUID.randomUUID());

        final SubscriptionBase subscription = Mockito.mock(SubscriptionBase.class);
        Mockito.when(subscriptionApi.getBaseSubscription(Mockito.<UUID>any(), Mockito.<InternalTenantContext>any())).thenReturn(subscription);

        final Plan plan = MockPlan.createBicycleNoTrialEvergreen1USD();
        final PriceList pricelist = new MockPriceList();
        Mockito.when(subscription.getCurrentPlan()).thenReturn(plan);
        Mockito.when(subscription.getCurrentPriceList()).thenReturn(pricelist);
        Mockito.when(subscription.getCurrentPhase()).thenReturn(plan.getFinalPhase());

        final BillingStateBundle state = calculatorBundle.calculateBillingState(bundle, internalCallContext);

        Assert.assertEquals(state.getNumberOfUnpaidInvoices(), 4);
        Assert.assertEquals(state.getBalanceOfUnpaidInvoices().intValue(), 11100);
        Assert.assertEquals(state.getDateOfEarliestUnpaidInvoice().compareTo(now.minusDays(5)), 0);
        Assert.assertEquals(state.getResponseForLastFailedPayment(), PaymentResponse.INSUFFICIENT_FUNDS); //TODO needs more when implemented
        Assert.assertEquals(state.getTags().length, 0);//TODO needs more when implemented
        Assert.assertEquals(state.getBasePlanBillingPeriod(), plan.getBillingPeriod());
        Assert.assertEquals(state.getBasePlanPhaseType(), plan.getFinalPhase().getPhaseType());
        Assert.assertEquals(state.getBasePlanPriceList(), pricelist);
        Assert.assertEquals(state.getBasePlanProduct(), plan.getProduct());
    }

    @Test(groups = "fast")
    public void testcalculateBillingStateForBundleNoOverdueInvoices() throws Exception {
        final UUID thisBundleId = new UUID(0L, 0L);

        now = new LocalDate();
        final List<Invoice> invoices = new ArrayList<Invoice>(5);

        Mockito.when(invoiceApi.getUnpaidInvoicesByAccountId(Mockito.<UUID>any(), Mockito.<LocalDate>any(), Mockito.<InternalTenantContext>any())).thenReturn(invoices);

        final SubscriptionBaseBundle bundle = Mockito.mock(SubscriptionBaseBundle.class);
        Mockito.when(bundle.getId()).thenReturn(thisBundleId);
        Mockito.when(bundle.getAccountId()).thenReturn(UUID.randomUUID());

        final SubscriptionBase subscription = Mockito.mock(SubscriptionBase.class);
        Mockito.when(subscriptionApi.getBaseSubscription(Mockito.<UUID>any(), Mockito.<InternalTenantContext>any())).thenReturn(subscription);

        final Plan plan = MockPlan.createBicycleNoTrialEvergreen1USD();
        final PriceList pricelist = new MockPriceList();
        Mockito.when(subscription.getCurrentPlan()).thenReturn(plan);
        Mockito.when(subscription.getCurrentPriceList()).thenReturn(pricelist);
        Mockito.when(subscription.getCurrentPhase()).thenReturn(plan.getFinalPhase());

        final BillingStateBundle state = calculatorBundle.calculateBillingState(bundle, internalCallContext);

        Assert.assertEquals(state.getNumberOfUnpaidInvoices(), 0);
        Assert.assertEquals(state.getBalanceOfUnpaidInvoices().intValue(), 0);
        Assert.assertEquals(state.getDateOfEarliestUnpaidInvoice(), null);
        Assert.assertEquals(state.getResponseForLastFailedPayment(), PaymentResponse.INSUFFICIENT_FUNDS); //TODO needs more when implemented
        Assert.assertEquals(state.getTags().length, 0);//TODO needs more when implemented
        Assert.assertEquals(state.getBasePlanBillingPeriod(), plan.getBillingPeriod());
        Assert.assertEquals(state.getBasePlanPhaseType(), plan.getFinalPhase().getPhaseType());
        Assert.assertEquals(state.getBasePlanPriceList(), pricelist);
        Assert.assertEquals(state.getBasePlanProduct(), plan.getProduct());

    }

    public void testCorrectBehaviorForNoOverdueConfig() {
        //TODO with no overdue config the system should be fine - take no action but see no NPEs
    }
}
