require "spec_helper"

describe "bundle platform" do
  context "without flags" do
    it "returns all the output" do
      gemfile <<-G
        source "file://#{gem_repo1}"

        #{ruby_version_correct}

        gem "foo"
      G

      bundle "platform"
      out.should == <<-G.chomp
Your platform is: #{RUBY_PLATFORM}

Your app has gems that work on these platforms:
* ruby

Your Gemfile specifies a Ruby version requirement:
* ruby #{RUBY_VERSION}

Your current platform satisfies the Ruby version requirement.
G
    end

    it "doesn't print ruby version requirement if it isn't specified" do
      gemfile <<-G
        source "file://#{gem_repo1}"

        gem "foo"
      G

      bundle "platform"
      out.should == <<-G.chomp
Your platform is: #{RUBY_PLATFORM}

Your app has gems that work on these platforms:
* ruby

Your Gemfile does not specify a Ruby version requirement.
G
    end

    it "doesn't match the ruby version requirement" do
      gemfile <<-G
        source "file://#{gem_repo1}"

        #{ruby_version_incorrect}

        gem "foo"
      G

      bundle "platform"
      out.should == <<-G.chomp
Your platform is: #{RUBY_PLATFORM}

Your app has gems that work on these platforms:
* ruby

Your Gemfile specifies a Ruby version requirement:
* ruby #{not_local_ruby_version}

Your Ruby version is #{RUBY_VERSION}, but your Gemfile specified #{not_local_ruby_version}
G
    end
  end

  context "--ruby" do
    it "returns ruby version when explicit" do
      gemfile <<-G
        source "file://#{gem_repo1}"
        ruby "1.9.3", :engine => 'ruby', :engine_version => '1.9.3'

        gem "foo"
      G

      bundle "platform --ruby"

      out.should eq("ruby 1.9.3")
    end

    it "engine defaults to MRI" do
      gemfile <<-G
        source "file://#{gem_repo1}"
        ruby "1.9.3"

        gem "foo"
      G

      bundle "platform --ruby"

      out.should eq("ruby 1.9.3")
    end

    it "handles jruby" do
      gemfile <<-G
        source "file://#{gem_repo1}"
        ruby "1.8.7", :engine => 'jruby', :engine_version => '1.6.5'

        gem "foo"
      G

      bundle "platform --ruby"

      out.should eq("ruby 1.8.7 (jruby 1.6.5)")
    end

    it "handles rbx" do
      gemfile <<-G
        source "file://#{gem_repo1}"
        ruby "1.8.7", :engine => 'rbx', :engine_version => '1.2.4'

        gem "foo"
      G

      bundle "platform --ruby"

      out.should eq("ruby 1.8.7 (rbx 1.2.4)")
    end

    it "raises an error if engine is used but engine version is not" do
      gemfile <<-G
        source "file://#{gem_repo1}"
        ruby "1.8.7", :engine => 'rbx'

        gem "foo"
      G

      bundle "platform", :exitstatus => true

      exitstatus.should_not == 0
    end

    it "raises an error if engine_version is used but engine is not" do
      gemfile <<-G
        source "file://#{gem_repo1}"
        ruby "1.8.7", :engine_version => '1.2.4'

        gem "foo"
      G

      bundle "platform", :exitstatus => true

      exitstatus.should_not == 0
    end

    it "raises an error if engine version doesn't match ruby version for mri" do
      gemfile <<-G
        source "file://#{gem_repo1}"
        ruby "1.8.7", :engine => 'ruby', :engine_version => '1.2.4'

        gem "foo"
      G

      bundle "platform", :exitstatus => true

      exitstatus.should_not == 0
    end
  end

  let(:ruby_version_correct) { "ruby \"#{RUBY_VERSION}\", :engine => \"#{local_ruby_engine}\", :engine_version => \"#{local_engine_version}\"" }
  let(:ruby_version_correct_engineless) { "ruby \"#{RUBY_VERSION}\"" }
  let(:ruby_version_incorrect) { "ruby \"#{not_local_ruby_version}\", :engine => \"#{local_ruby_engine}\", :engine_version => \"#{not_local_ruby_version}\"" }
  let(:engine_incorrect) { "ruby \"#{RUBY_VERSION}\", :engine => \"#{not_local_tag}\", :engine_version => \"#{RUBY_VERSION}\"" }
  let(:engine_version_incorrect) { "ruby \"#{RUBY_VERSION}\", :engine => \"#{local_ruby_engine}\", :engine_version => \"#{not_local_engine_version}\"" }

  def should_be_ruby_version_incorrect(opts = {:exitstatus => true})
    exitstatus.should eq(18) if opts[:exitstatus]
    out.should be_include("Your Ruby version is #{RUBY_VERSION}, but your Gemfile specified #{not_local_ruby_version}")
  end

  def should_be_engine_incorrect(opts = {:exitstatus => true})
    exitstatus.should eq(18) if opts[:exitstatus]
    out.should be_include("Your Ruby engine is #{local_ruby_engine}, but your Gemfile specified #{not_local_tag}")
  end

  def should_be_engine_version_incorrect(opts = {:exitstatus => true})
    exitstatus.should eq(18) if opts[:exitstatus]
    out.should be_include("Your #{local_ruby_engine} version is #{local_engine_version}, but your Gemfile specified #{local_ruby_engine} #{not_local_engine_version}")
  end

  context "bundle install" do
    it "installs fine when the ruby version matches" do
      install_gemfile <<-G
        source "file://#{gem_repo1}"
        gem "rack"

        #{ruby_version_correct}
      G

      bundled_app('Gemfile.lock').should exist
    end

    it "installs fine with any engine" do
      simulate_ruby_engine "jruby" do
        install_gemfile <<-G
          source "file://#{gem_repo1}"
          gem "rack"

          #{ruby_version_correct_engineless}
        G

        bundled_app('Gemfile.lock').should exist
      end
    end

    it "doesn't install when the ruby version doesn't match" do
      install_gemfile <<-G, :exitstatus => true
        source "file://#{gem_repo1}"
        gem "rack"

        #{ruby_version_incorrect}
      G

      bundled_app('Gemfile.lock').should_not exist
      should_be_ruby_version_incorrect
    end

    it "doesn't install when engine doesn't match" do
      install_gemfile <<-G, :exitstatus => true
        source "file://#{gem_repo1}"
        gem "rack"

        #{engine_incorrect}
      G

      bundled_app('Gemfile.lock').should_not exist
      should_be_engine_incorrect
    end

    it "doesn't install when engine version doesn't match" do
      simulate_ruby_engine "jruby" do
        install_gemfile <<-G, :exitstatus => true
          source "file://#{gem_repo1}"
          gem "rack"

          #{engine_version_incorrect}
        G

        bundled_app('Gemfile.lock').should_not exist
        should_be_engine_version_incorrect
      end
    end
  end

  context "bundle check" do
    it "checks fine when the ruby version matches" do
      install_gemfile <<-G
        source "file://#{gem_repo1}"
        gem "rack"
      G

      gemfile <<-G
        source "file://#{gem_repo1}"
        gem "rack"

        #{ruby_version_correct}
      G

      bundle :check, :exitstatus => true
      exitstatus.should eq(0)
      out.should == "The Gemfile's dependencies are satisfied"
    end

    it "checks fine with any engine" do
      simulate_ruby_engine "jruby" do
        install_gemfile <<-G
          source "file://#{gem_repo1}"
          gem "rack"
        G

        gemfile <<-G
          source "file://#{gem_repo1}"
          gem "rack"

          #{ruby_version_correct_engineless}
        G

        bundle :check, :exitstatus => true
        exitstatus.should eq(0)
        out.should == "The Gemfile's dependencies are satisfied"
      end
    end

    it "fails when ruby version doesn't match" do
      install_gemfile <<-G
        source "file://#{gem_repo1}"
        gem "rack"
      G

      gemfile <<-G
        source "file://#{gem_repo1}"
        gem "rack"

        #{ruby_version_incorrect}
      G

      bundle :check, :exitstatus => true
      should_be_ruby_version_incorrect
    end

    it "fails when engine doesn't match" do
      install_gemfile <<-G
        source "file://#{gem_repo1}"
        gem "rack"
      G

      gemfile <<-G
        source "file://#{gem_repo1}"
        gem "rack"

        #{engine_incorrect}
      G

      bundle :check, :exitstatus => true
      should_be_engine_incorrect
    end

    it "fails when engine version doesn't match" do
      simulate_ruby_engine "ruby" do
        install_gemfile <<-G
          source "file://#{gem_repo1}"
          gem "rack"
        G

        gemfile <<-G
          source "file://#{gem_repo1}"
          gem "rack"

          #{engine_version_incorrect}
        G

        bundle :check, :exitstatus => true
        should_be_engine_version_incorrect
      end
    end
  end

  context "bundle update" do
    before do
      build_repo2

      install_gemfile <<-G
        source "file://#{gem_repo2}"
        gem "activesupport"
        gem "rack-obama"
      G
    end

    it "updates successfully when the ruby version matches" do
      gemfile <<-G
        source "file://#{gem_repo2}"
        gem "activesupport"
        gem "rack-obama"

        #{ruby_version_correct}
      G
      update_repo2 do
        build_gem "activesupport", "3.0"
      end

      bundle "update"
      should_be_installed "rack 1.2", "rack-obama 1.0", "activesupport 3.0"
    end

    it "updates fine with any engine" do
      simulate_ruby_engine "jruby" do
        gemfile <<-G
          source "file://#{gem_repo2}"
          gem "activesupport"
          gem "rack-obama"

          #{ruby_version_correct_engineless}
        G
        update_repo2 do
          build_gem "activesupport", "3.0"
        end

        bundle "update"
        should_be_installed "rack 1.2", "rack-obama 1.0", "activesupport 3.0"
      end
    end

    it "fails when ruby version doesn't match" do
      gemfile <<-G
        source "file://#{gem_repo2}"
        gem "activesupport"
        gem "rack-obama"

        #{ruby_version_incorrect}
      G
      update_repo2 do
        build_gem "activesupport", "3.0"
      end

      bundle :update, :exitstatus => true
      should_be_ruby_version_incorrect
    end

    it "fails when ruby engine doesn't match" do
      gemfile <<-G
        source "file://#{gem_repo2}"
        gem "activesupport"
        gem "rack-obama"

        #{engine_incorrect}
      G
      update_repo2 do
        build_gem "activesupport", "3.0"
      end

      bundle :update, :exitstatus => true
      should_be_engine_incorrect
    end

    it "fails when ruby engine version doesn't match" do
      simulate_ruby_engine "jruby" do
        gemfile <<-G
          source "file://#{gem_repo2}"
          gem "activesupport"
          gem "rack-obama"

          #{engine_version_incorrect}
        G
        update_repo2 do
          build_gem "activesupport", "3.0"
        end

        bundle :update, :exitstatus => true
        should_be_engine_version_incorrect
      end
    end
  end

  context "bundle show" do
    before do
      install_gemfile <<-G
        source "file://#{gem_repo1}"
        gem "rails"
      G
    end

    it "prints path if ruby version is correct" do
      gemfile <<-G
        source "file://#{gem_repo1}"
        gem "rails"

        #{ruby_version_correct}
      G

      bundle "show rails"
      out.should == default_bundle_path('gems', 'rails-2.3.2').to_s
    end

    it "prints path if ruby version is correct for any engine" do
      simulate_ruby_engine "jruby" do
        gemfile <<-G
          source "file://#{gem_repo1}"
          gem "rails"

          #{ruby_version_correct_engineless}
        G

        bundle "show rails"
        out.should == default_bundle_path('gems', 'rails-2.3.2').to_s
      end
    end

    it "fails if ruby version doesn't match" do
      gemfile <<-G
        source "file://#{gem_repo1}"
        gem "rails"

        #{ruby_version_incorrect}
      G

      bundle "show rails", :exitstatus => true
      should_be_ruby_version_incorrect
    end

    it "fails if engine doesn't match" do
      gemfile <<-G
        source "file://#{gem_repo1}"
        gem "rails"

        #{engine_incorrect}
      G

      bundle "show rails", :exitstatus => true
      should_be_engine_incorrect
    end

    it "fails if engine version doesn't match" do
      simulate_ruby_engine "jruby" do
        gemfile <<-G
          source "file://#{gem_repo1}"
          gem "rails"

          #{engine_version_incorrect}
        G

        bundle "show rails", :exitstatus => true
        should_be_engine_version_incorrect
      end
    end
  end

  context "bundle cache" do
    before do
      gemfile <<-G
        gem 'rack'
      G

      system_gems "rack-1.0.0"
    end

    it "copies the .gem file to vendor/cache when ruby version matches" do
      gemfile <<-G
        gem 'rack'

        #{ruby_version_correct}
      G

      bundle :cache
      bundled_app("vendor/cache/rack-1.0.0.gem").should exist
    end

    it "copies the .gem file to vendor/cache when ruby version matches for any engine" do
      simulate_ruby_engine "jruby" do
        gemfile <<-G
          gem 'rack'

          #{ruby_version_correct_engineless}
        G

        bundle :cache
        bundled_app("vendor/cache/rack-1.0.0.gem").should exist
      end
    end

    it "fails if the ruby version doesn't match" do
      gemfile <<-G
        gem 'rack'

        #{ruby_version_incorrect}
      G

      bundle :cache, :exitstatus => true
      should_be_ruby_version_incorrect
    end

    it "fails if the engine doesn't match" do
      gemfile <<-G
        gem 'rack'

        #{engine_incorrect}
      G

      bundle :cache, :exitstatus => true
      should_be_engine_incorrect
    end

    it "fails if the engine version doesn't match" do
      simulate_ruby_engine "jruby" do
        gemfile <<-G
        gem 'rack'

        #{engine_version_incorrect}
        G

        bundle :cache, :exitstatus => true
        should_be_engine_version_incorrect
      end
    end
  end

  context "bundle pack" do
    before do
      gemfile <<-G
        gem 'rack'
      G

      system_gems "rack-1.0.0"
    end

    it "copies the .gem file to vendor/cache when ruby version matches" do
      gemfile <<-G
        gem 'rack'

        #{ruby_version_correct}
      G

      bundle :pack
      bundled_app("vendor/cache/rack-1.0.0.gem").should exist
    end

    it "copies the .gem file to vendor/cache when ruby version matches any engine" do
      simulate_ruby_engine "jruby" do
        gemfile <<-G
          gem 'rack'

          #{ruby_version_correct_engineless}
        G

        bundle :pack
        bundled_app("vendor/cache/rack-1.0.0.gem").should exist
      end
    end

    it "fails if the ruby version doesn't match" do
      gemfile <<-G
        gem 'rack'

        #{ruby_version_incorrect}
      G

      bundle :pack, :exitstatus => true
      should_be_ruby_version_incorrect
    end

    it "fails if the engine doesn't match" do
      gemfile <<-G
        gem 'rack'

        #{engine_incorrect}
      G

      bundle :pack, :exitstatus => true
      should_be_engine_incorrect
    end

    it "fails if the engine version doesn't match" do
      simulate_ruby_engine "jruby" do
        gemfile <<-G
        gem 'rack'

        #{engine_version_incorrect}
        G

        bundle :pack, :exitstatus => true
        should_be_engine_version_incorrect
      end
    end
  end

  context "bundle exec" do
    before do
      system_gems "rack-1.0.0", "rack-0.9.1"
    end

    it "activates the correct gem when ruby version matches" do
      gemfile <<-G
        gem "rack", "0.9.1"

        #{ruby_version_correct}
      G

      bundle "exec rackup"
      out.should == "0.9.1"
    end

    it "activates the correct gem when ruby version matches any engine" do
      simulate_ruby_engine "jruby" do
        gemfile <<-G
          gem "rack", "0.9.1"

          #{ruby_version_correct_engineless}
        G

        bundle "exec rackup"
        out.should == "0.9.1"
      end
    end

    it "fails when the ruby version doesn't match" do
      gemfile <<-G
        gem "rack", "0.9.1"

        #{ruby_version_incorrect}
      G

      bundle "exec rackup", :exitstatus => true
      should_be_ruby_version_incorrect
    end

    it "fails when the engine doesn't match" do
      gemfile <<-G
        gem "rack", "0.9.1"

        #{engine_incorrect}
      G

      bundle "exec rackup", :exitstatus => true
      should_be_engine_incorrect
    end

    it "fails when the engine version doesn't match" do
      simulate_ruby_engine "jruby" do
        gemfile <<-G
          gem "rack", "0.9.1"

          #{engine_version_incorrect}
        G

        bundle "exec rackup", :exitstatus => true
        should_be_engine_version_incorrect
      end
    end
  end

  context "bundle console" do
    before do
      install_gemfile <<-G
        source "file://#{gem_repo1}"
        gem "rack"
        gem "activesupport", :group => :test
        gem "rack_middleware", :group => :development
      G
    end

    it "starts IRB with the default group loaded when ruby version matches" do
      gemfile <<-G
        source "file://#{gem_repo1}"
        gem "rack"
        gem "activesupport", :group => :test
        gem "rack_middleware", :group => :development

        #{ruby_version_correct}
      G

      bundle "console" do |input|
        input.puts("puts RACK")
        input.puts("exit")
      end
      out.should include("0.9.1")
    end

    it "starts IRB with the default group loaded when ruby version matches any engine" do
      simulate_ruby_engine "jruby" do
        gemfile <<-G
          source "file://#{gem_repo1}"
          gem "rack"
          gem "activesupport", :group => :test
          gem "rack_middleware", :group => :development

          #{ruby_version_correct_engineless}
        G

        bundle "console" do |input|
          input.puts("puts RACK")
          input.puts("exit")
        end
        out.should include("0.9.1")
      end
    end

    it "fails when ruby version doesn't match" do
      gemfile <<-G
        source "file://#{gem_repo1}"
        gem "rack"
        gem "activesupport", :group => :test
        gem "rack_middleware", :group => :development

        #{ruby_version_incorrect}
      G

      bundle "console", :exitstatus => true
      should_be_ruby_version_incorrect
    end

    it "fails when engine doesn't match" do
      gemfile <<-G
        source "file://#{gem_repo1}"
        gem "rack"
        gem "activesupport", :group => :test
        gem "rack_middleware", :group => :development

        #{engine_incorrect}
      G

      bundle "console", :exitstatus => true
      should_be_engine_incorrect
    end

    it "fails when engine version doesn't match" do
      simulate_ruby_engine "jruby" do
        gemfile <<-G
          source "file://#{gem_repo1}"
          gem "rack"
          gem "activesupport", :group => :test
          gem "rack_middleware", :group => :development

          #{engine_version_incorrect}
        G

        bundle "console", :exitstatus => true
        should_be_engine_version_incorrect
      end
    end
  end

  context "Bundler.setup" do
    before do
      install_gemfile <<-G
        source "file://#{gem_repo1}"
        gem "yard"
        gem "rack", :group => :test
      G
    end

    it "makes a Gemfile.lock if setup succeeds" do
      install_gemfile <<-G
        source "file://#{gem_repo1}"
        gem "yard"
        gem "rack"

        #{ruby_version_correct}
      G

      File.read(bundled_app("Gemfile.lock"))

      FileUtils.rm(bundled_app("Gemfile.lock"))

      run "1"
      bundled_app("Gemfile.lock").should exist
    end

    it "makes a Gemfile.lock if setup succeeds for any engine" do
      simulate_ruby_engine "jruby" do
        install_gemfile <<-G
          source "file://#{gem_repo1}"
          gem "yard"
          gem "rack"

          #{ruby_version_correct_engineless}
        G

        File.read(bundled_app("Gemfile.lock"))

        FileUtils.rm(bundled_app("Gemfile.lock"))

        run "1"
        bundled_app("Gemfile.lock").should exist
      end
    end

    it "fails when ruby version doesn't match" do
      install_gemfile <<-G
        source "file://#{gem_repo1}"
        gem "yard"
        gem "rack"

        #{ruby_version_incorrect}
      G

      File.read(bundled_app("Gemfile.lock"))

      FileUtils.rm(bundled_app("Gemfile.lock"))

      ruby <<-R
        require 'rubygems'
        require 'bundler'

        begin
          Bundler.setup
        rescue Bundler::RubyVersionMismatch => e
          puts e.message
        end
      R

      bundled_app("Gemfile.lock").should_not exist
      should_be_ruby_version_incorrect(:exitstatus => false)
    end

    it "fails when engine doesn't match" do
      install_gemfile <<-G
        source "file://#{gem_repo1}"
        gem "yard"
        gem "rack"

        #{engine_incorrect}
      G

      File.read(bundled_app("Gemfile.lock"))

      FileUtils.rm(bundled_app("Gemfile.lock"))

      ruby <<-R
        require 'rubygems'
        require 'bundler'

        begin
          Bundler.setup
        rescue Bundler::RubyVersionMismatch => e
          puts e.message
        end
      R

      bundled_app("Gemfile.lock").should_not exist
      should_be_engine_incorrect(:exitstatus => false)
    end

    it "fails when engine version doesn't match" do
      simulate_ruby_engine "jruby" do
        install_gemfile <<-G
          source "file://#{gem_repo1}"
          gem "yard"
          gem "rack"

          #{engine_version_incorrect}
        G

        File.read(bundled_app("Gemfile.lock"))

        FileUtils.rm(bundled_app("Gemfile.lock"))

        ruby <<-R
          require 'rubygems'
          require 'bundler'

          begin
            Bundler.setup
          rescue Bundler::RubyVersionMismatch => e
            puts e.message
          end
        R

        bundled_app("Gemfile.lock").should_not exist
        should_be_engine_version_incorrect(:exitstatus => false)
      end
    end
  end

  context "bundle outdated" do
    before do
      build_repo2 do
        build_git "foo", :path => lib_path("foo")
      end

      install_gemfile <<-G
        source "file://#{gem_repo2}"
        gem "activesupport", "2.3.5"
        gem "foo", :git => "#{lib_path('foo')}"
      G
    end

    it "returns list of outdated gems when the ruby version matches" do
      update_repo2 do
        build_gem "activesupport", "3.0"
        update_git "foo", :path => lib_path("foo")
      end

      gemfile <<-G
        source "file://#{gem_repo2}"
        gem "activesupport", "2.3.5"
        gem "foo", :git => "#{lib_path('foo')}"

        #{ruby_version_correct}
      G

      bundle "outdated"
      out.should include("activesupport (3.0 > 2.3.5)")
      out.should include("foo (1.0")
    end

    it "returns list of outdated gems when the ruby version matches for any engine" do
      simulate_ruby_engine "jruby" do
        update_repo2 do
          build_gem "activesupport", "3.0"
          update_git "foo", :path => lib_path("foo")
        end

        gemfile <<-G
          source "file://#{gem_repo2}"
          gem "activesupport", "2.3.5"
          gem "foo", :git => "#{lib_path('foo')}"

          #{ruby_version_correct_engineless}
        G

        bundle "outdated"
        out.should include("activesupport (3.0 > 2.3.5)")
        out.should include("foo (1.0")
      end
    end

    it "fails when the ruby version doesn't match" do
      update_repo2 do
        build_gem "activesupport", "3.0"
        update_git "foo", :path => lib_path("foo")
      end

      gemfile <<-G
        source "file://#{gem_repo2}"
        gem "activesupport", "2.3.5"
        gem "foo", :git => "#{lib_path('foo')}"

        #{ruby_version_incorrect}
      G

      bundle "outdated", :exitstatus => true
      should_be_ruby_version_incorrect
    end

    it "fails when the engine doesn't match" do
      update_repo2 do
        build_gem "activesupport", "3.0"
        update_git "foo", :path => lib_path("foo")
      end

      gemfile <<-G
        source "file://#{gem_repo2}"
        gem "activesupport", "2.3.5"
        gem "foo", :git => "#{lib_path('foo')}"

        #{engine_incorrect}
      G

      bundle "outdated", :exitstatus => true
      should_be_engine_incorrect
    end

    it "fails when the engine version doesn't match" do
      simulate_ruby_engine "jruby" do
        update_repo2 do
          build_gem "activesupport", "3.0"
          update_git "foo", :path => lib_path("foo")
        end

        gemfile <<-G
          source "file://#{gem_repo2}"
          gem "activesupport", "2.3.5"
          gem "foo", :git => "#{lib_path('foo')}"

          #{engine_version_incorrect}
        G

        bundle "outdated", :exitstatus => true
        should_be_engine_version_incorrect
      end
    end
  end
end
