library(ggplot2)
library(ggforce)

requests <- read.csv2("../applications/output/requests-handled.csv", header=TRUE, sep=",", dec=".")

print("median")

aggregate <- aggregate(formula = requests~application+version+users+execution, data = requests, FUN = length)
aggregate <- reshape(aggregate, idvar = c("application", "version", "users"), timevar = "execution", direction = "wide")

min_aggregate <- aggregate
min_aggregate[, "min_requests"] <- apply(min_aggregate[, 4:length(min_aggregate)], 1, FUN = which.min)
min_aggregate[, "reference"] <- paste(min_aggregate$application, min_aggregate$version, min_aggregate$users, min_aggregate$min_requests)

median_aggregate <- aggregate
median_aggregate[, "median_requests"] <- apply(median_aggregate[, 4:length(median_aggregate)], 1, FUN = function(x) which.min(abs(x - median(x))))
median_aggregate[, "reference"] <- paste(median_aggregate$application, median_aggregate$version, median_aggregate$users, median_aggregate$median_requests)

max_aggregate <- aggregate
max_aggregate[, "max_requests"] <- apply(max_aggregate[, 4:length(max_aggregate)], 1, FUN = which.max)
max_aggregate[, "reference"] <- paste(max_aggregate$application, max_aggregate$version, max_aggregate$users, max_aggregate$max_requests)

print("throughput")

slice <- requests
min_slice <- subset(slice, paste(slice$application, slice$version, slice$users, slice$execution) %in% min_aggregate$reference)
median_slice <- subset(slice, paste(slice$application, slice$version, slice$users, slice$execution) %in% median_aggregate$reference)
max_slice <- subset(slice, paste(slice$application, slice$version, slice$users, slice$execution) %in% max_aggregate$reference)

requests_slice <- aggregate(formula = requests~application+version+users, data = slice, FUN = max)
min_slice <- aggregate(formula = time~application+version+users+execution, data = min_slice, FUN = max)
median_slice <- aggregate(formula = time~application+version+users+execution, data = median_slice, FUN = max)
max_slice <- aggregate(formula = time~application+version+users+execution, data = max_slice, FUN = max)

min_slice[, "execution"] <- NULL
names(min_slice)[names(min_slice) == "time"] <- "min_time"

median_slice[, "execution"] <- NULL
names(median_slice)[names(median_slice) == "time"] <- "median_time"

max_slice[, "execution"] <- NULL
names(max_slice)[names(max_slice) == "time"] <- "max_time"

throughput <- requests_slice
throughput <- merge(throughput, min_slice)
throughput <- merge(throughput, median_slice)
throughput <- merge(throughput, max_slice)
throughput[, "min_throughput"] <- throughput$requests / throughput$min_time
throughput[, "median_throughput"] <- throughput$requests / throughput$median_time
throughput[, "max_throughput"] <- throughput$requests / throughput$max_time

pdf("throughput.pdf")

iter_applications = unique(slice$application)
for(i in 1:ceiling(length(iter_applications) / 3)) {
	plot <- ggplot(throughput, aes(x = version, y = median_throughput, fill = version)) +
		geom_bar(stat = "identity") +
		geom_errorbar(aes(ymin = min_throughput, ymax = max_throughput), width = .2, position = position_dodge(.9)) +
		geom_text(aes(label = sprintf("%.2f", median_throughput)), vjust = 2, size = 2.5, colour = "white") +
		scale_fill_grey(start = 0.2, end = 0.6) +
		scale_colour_grey(start = 0.2, end = 0.6) +
		facet_wrap_paginate(application ~ users, scales = "free", ncol = 3, nrow = 3, page = i) +
		theme(axis.text.x = element_text(angle = 20)) +
		theme(legend.position = "bottom")
	print(plot)
}
