/*
 * To change this license header, choose License Headers in Project Properties.
 * To change this template file, choose Tools | Templates
 * and open the template in the editor.
 */
package br.ufrgs.inf.prosoft.cache;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

/**
 *
 * @author romulo
 * @param <K>
 * @param <V>
 */
public class MultiCache<K, V> implements Cache<K, V> {

    private static final boolean CACHE_REGISTER_SIZE = System.getenv("CACHE_REGISTER_SIZE") != null && System.getenv("CACHE_REGISTER_SIZE").equals("true");
    private final HashMap<K, V> map;
    private final ConcurrentHashMap<Optional<K>, Long> keyHasTTL;
    private final CachePerformance cachePerformance;

    public MultiCache() {
        this(new CachePerformance());
    }

    public MultiCache(String name) {
        this(new CachePerformance(name));
    }

    public MultiCache(CachePerformance cachingPerformance) {
        this.cachePerformance = cachingPerformance;
        this.map = new HashMap<>();
        this.keyHasTTL = new ConcurrentHashMap<>();

        ScheduledExecutorService batchEntriesInvalidator = Executors.newScheduledThreadPool(1);
        batchEntriesInvalidator.scheduleAtFixedRate(() -> {
            this.keyHasTTL.forEach((key, value) -> {
                if (!isValid(key.orElse(null))) {
                    invalidate(key.orElse(null));
                }
            });
        }, 10, 10, TimeUnit.SECONDS);
    }

    public CachePerformance getCachePerformance() {
        return this.cachePerformance;
    }

    @Override
    public void put(K key, V value, long timeToLive) {
        put(key, value);
        this.keyHasTTL.put(Optional.ofNullable(key), System.currentTimeMillis() + timeToLive);
    }

    @Override
    public void put(K key, V value) {
        invalidate(key);
        this.map.put(key, value);
        String identifier = getIdentifier(value);
        if (CACHE_REGISTER_SIZE) {
            this.cachePerformance.registerEvent(EventType.ADDITION, identifier, CachePerformance.calculateObjectSize(value));
        } else {
            this.cachePerformance.registerEvent(EventType.ADDITION, identifier);
        }
        this.cachePerformance.registerSize(this.map.size());
    }

    @Override
    public V get(K key) throws KeyNotFoundException {
        if (!containsKey(key)) {
            this.cachePerformance.registerEvent(EventType.MISS);
            throw new KeyNotFoundException();
        }
        V get = this.map.get(key);
        String identifier = getIdentifier(get);
        if (CACHE_REGISTER_SIZE) {
            this.cachePerformance.registerEvent(EventType.HIT, identifier, CachePerformance.calculateObjectSize(get));
        } else {
            this.cachePerformance.registerEvent(EventType.HIT, identifier);
        }
        return get;
    }

    @Override
    public void invalidate(K key) {
        this.keyHasTTL.remove(Optional.ofNullable(key));
        if (this.map.containsKey(key)) {
            V remove = this.map.remove(key);
            String identifier = getIdentifier(remove);
            if (CACHE_REGISTER_SIZE) {
                this.cachePerformance.registerEvent(EventType.INVALIDATION, identifier, CachePerformance.calculateObjectSize(remove));
            } else {
                this.cachePerformance.registerEvent(EventType.INVALIDATION, identifier);
            }
        }
    }

    private boolean isValid(K key) {
        try {
            return System.currentTimeMillis() <= this.keyHasTTL.get(Optional.ofNullable(key));
        } catch (NullPointerException ex) {
            return true;
        }
    }

    public Set<Map.Entry<K, V>> entrySet() {
        return this.map.entrySet().stream()
                .filter(entry -> isValid(entry.getKey()))
                .collect(Collectors.
                        toMap(entry -> entry.getKey(), entry -> entry.getValue())
                ).entrySet();
    }

    public List<V> values() {
        return this.map.entrySet().stream()
                .filter(entry -> isValid(entry.getKey()))
                .map(entry -> entry.getValue())
                .collect(Collectors.toList());
    }

    public boolean containsKey(K key) {
        return this.map.containsKey(key) && isValid(key);
    }

    private String getIdentifier(V value) {
        return value != null ? String.valueOf(value.hashCode()) : "null";
    }
}
