
package org.springframework.samples.petclinic.web;

import java.util.Collection;

import javax.validation.Valid;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.samples.petclinic.Owner;
import org.springframework.samples.petclinic.repository.OwnerRepository;
import org.springframework.stereotype.Controller;
import org.springframework.ui.Model;
import org.springframework.validation.BindingResult;
import org.springframework.web.bind.WebDataBinder;
import org.springframework.web.bind.annotation.InitBinder;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.SessionAttributes;
import org.springframework.web.bind.support.SessionStatus;
import org.springframework.web.servlet.ModelAndView;

/**
 * JavaBean form controller that is used to handle <code>Owner</code>s .
 * 
 * @author Juergen Hoeller
 * @author Ken Krebs
 * @author Arjen Poutsma
 * @author Michael Isvy
 */
@Controller
@SessionAttributes(types = Owner.class)
public class OwnerController {

	private final OwnerRepository ownerRepository;


	@Autowired
	public OwnerController(OwnerRepository ownerRepository) {
		this.ownerRepository = ownerRepository;
	}

	@InitBinder
	public void setAllowedFields(WebDataBinder dataBinder) {
		dataBinder.setDisallowedFields("id");
	}

	@RequestMapping(value="/owners/new", method = RequestMethod.GET)
	public String initCreationForm(Model model) {
		Owner owner = new Owner();
		model.addAttribute(owner);
		return "owners/createOrUpdateOwnerForm";
	}

	@RequestMapping(value="/owners/new", method = RequestMethod.POST)
	public String processCreationForm(@Valid Owner owner, BindingResult result, SessionStatus status) {
		if (result.hasErrors()) {
			return "owners/createOrUpdateOwnerForm";
		}
		else {
			this.ownerRepository.save(owner);
			status.setComplete();
			return "redirect:/owners/" + owner.getId();
		}
	}
	
	@RequestMapping(value = "/owners/find", method = RequestMethod.GET)
	public String initFindForm(Model model) {
		throw new RuntimeException("aaaaaaa");
	}

	@RequestMapping(value = "/owners", method = RequestMethod.GET)
	public String processFindForm(Owner owner, BindingResult result, Model model) {

		// allow parameterless GET request for /owners to return all records
		if (owner.getLastName() == null) {
			owner.setLastName(""); // empty string signifies broadest possible search
		}

		// find owners by last name
		Collection<Owner> results = this.ownerRepository.findByLastName(owner.getLastName());
		if (results.size() < 1) {
			// no owners found
			result.rejectValue("lastName", "notFound", "not found");
			return "owners/findOwners";
		}
		if (results.size() > 1) {
			// multiple owners found
			model.addAttribute("selections", results);
			return "owners/ownersList";
		}
		else {
			// 1 owner found
			owner = results.iterator().next();
			return "redirect:/owners/" + owner.getId();
		}
	}
	
	@RequestMapping(value="/owners/{ownerId}/edit", method = RequestMethod.GET)
	public String initUpdateOwnerForm(@PathVariable("ownerId") int ownerId, Model model) {
		Owner owner = this.ownerRepository.findById(ownerId);
		model.addAttribute(owner);
		return "owners/createOrUpdateOwnerForm";
	}

	@RequestMapping(value="/owners/{ownerId}/edit", method = RequestMethod.PUT)
	public String processUpdateOwnerForm(@Valid Owner owner, BindingResult result, SessionStatus status) {
		if (result.hasErrors()) {
			return "owners/createOrUpdateOwnerForm";
		}
		else {
			this.ownerRepository.save(owner);
			status.setComplete();
			return "redirect:/owners/" + owner.getId();
		}
	}
	
	/**
	 * Custom handler for displaying an owner.
	 *
	 * @param ownerId the ID of the owner to display
	 * @return a ModelMap with the model attributes for the view
	 */
	@RequestMapping("/owners/{ownerId}")
	public ModelAndView showOwner(@PathVariable("ownerId") int ownerId) {
		ModelAndView mav = new ModelAndView("owners/ownerDetails");
		mav.addObject(this.ownerRepository.findById(ownerId));
		return mav;
	}

}
