package br.ufrgs.inf.prosoft.tfcache.configuration;

import com.google.gson.JsonObject;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.logging.Level;
import java.util.logging.Logger;

public class Configuration {

    private static final Logger LOGGER = Logger.getLogger(Configuration.class.getName());

    private static String input = null;
    private static String store = null;
    private static String level = "method";
    private static String output = null;
    private static String changeability = "allow";
    private static String staleness = "ignore";
    private static String kernel = "exhaustive";
    private static ArrayList<Double> preferences = new ArrayList<>(List.of(1D, 0D));
    private static boolean verbose = false;

    public static void process(String[] args) {
        process(Arguments.parse(args));
    }

    public static void process(Map<String, String> arguments) {
        setInput(arguments.get("trace"));
        setStore(arguments.get("store"));
        setLevel(arguments.get("level"));
        setOutput(arguments.get("output"));
        if (getLevel().equals("input") && getOutput() == null) {
            LOGGER.severe("Output is required for input-level recommendations");
            System.exit(1);
        }
        setChangeability(arguments.get("changeability"));
        setStaleness(arguments.get("staleness"));
        setKernel(arguments.get("kernel"));
        setVerbose(arguments.get("verbose"));
        setPreferences(arguments.get("preferences"));
    }

    public static String getInput() {
        return input;
    }

    public static void setInput(String input) {
        if (input == null || input.isBlank()) {
            LOGGER.severe("<TracePath> is required");
            System.exit(1);
        }
        Configuration.input = input;
    }

    public static String getStore() {
        return store;
    }

    public static void setStore(String store) {
        if (store == null) {
            LOGGER.info("store disabled");
            return;
        }
        if (store.isBlank()) {
            LOGGER.severe("storePath cannot be empty");
            System.exit(1);
        }
        Configuration.store = store;
    }

    public static String getLevel() {
        return level;
    }

    public static void setLevel(String level) {
        if (level == null) {
            LOGGER.log(Level.INFO, "Using default level: {0}", Configuration.level);
            return;
        }
        if (!level.equals("method") && !level.equals("input")) {
            LOGGER.severe("Unrecognised option for level");
            System.exit(1);
        }
        Configuration.level = level;
    }

    public static String getOutput() {
        return output;
    }

    public static void setOutput(String output) {
        if (output == null || output.isBlank()) {
            return;
        }
        Configuration.output = output;
    }

    public static String getChangeability() {
        return changeability;
    }

    public static void setChangeability(String changeability) {
        if (changeability == null) {
            LOGGER.log(Level.INFO, "Using default changeability: {0}", Configuration.changeability);
            return;
        }
        if (!changeability.equals("allow") && !changeability.equals("deny")) {
            LOGGER.severe("Unrecognised option for changeability");
            System.exit(1);
        }
        Configuration.changeability = changeability;
    }

    public static String getStaleness() {
        return staleness;
    }

    public static void setStaleness(String staleness) {
        if (staleness == null) {
            LOGGER.log(Level.INFO, "Using default staleness: {0}", Configuration.staleness);
            return;
        }
        if (!staleness.equals("ignore") && !staleness.equals("shrink")) {
            LOGGER.severe("Unrecognised option for staleness");
            System.exit(1);
        }
        Configuration.staleness = staleness;
    }

    public static String getKernel() {
        return kernel;
    }

    public static void setKernel(String kernel) {
        if (kernel == null) {
            LOGGER.log(Level.INFO, "Using default kernel: {0}", Configuration.kernel);
            return;
        }
        if (!kernel.equals("exhaustive") && !kernel.equals("optimised") && !kernel.equals("test")) {
            LOGGER.severe("Unrecognised option for kernel");
            System.exit(1);
        }
        Configuration.kernel = kernel;
    }

    public static boolean getVerbose() {
        return verbose;
    }

    public static void setVerbose(String verbose) {
        if (verbose == null) {
            return;
        }
        if (!verbose.equals("true") && !verbose.equals("false")) {
            LOGGER.severe("Unrecognised option for verbose");
            System.exit(1);
        }
        Configuration.verbose = verbose.equals("true");
    }

    public static List<Double> getPreferences() {
        return preferences;
    }

    public static void setPreferences(String preferences) {
        if (preferences == null) {
            LOGGER.info("using default preferences");
            return;
        }
        String[] split = preferences.split(",");
        if (split.length != 2) {
            LOGGER.severe("Wrong input for preferences");
            System.exit(1);
        }
        Configuration.preferences.clear();
        Configuration.preferences.add(Double.valueOf(split[0]));
        Configuration.preferences.add(Double.valueOf(split[1]));
    }

    public static String getUUID() {
        return "level:" + Configuration.getLevel()
                + ",staleness:" + Configuration.getStaleness()
                + ",kernel:" + Configuration.getKernel();
    }

    public static JsonObject toJSONObject() {
        JsonObject configuration = new JsonObject();
        configuration.addProperty("level", Configuration.getLevel());
        configuration.addProperty("staleness", Configuration.getStaleness());
        configuration.addProperty("kernel", Configuration.getKernel());
        return configuration;
    }

}
