package br.ufrgs.inf.prosoft.tfcache;

import br.ufrgs.inf.prosoft.tfcache.configuration.Configuration;
import br.ufrgs.inf.prosoft.tfcache.metadata.Occurrence;
import com.google.gson.*;

import java.io.*;
import java.util.ArrayList;
import java.util.List;
import java.util.function.Supplier;
import java.util.logging.Logger;

public class StorageManager {

    private static final Logger LOGGER = Logger.getLogger(StorageManager.class.getName());
    private static JsonObject STORAGE = null;
    private static String UUID = null;

    public static void put(String uuid, List<Occurrence> occurrences, Pareto pareto) {
        if (occurrences == null || occurrences.isEmpty() || occurrences.size() < 2) {
            LOGGER.severe("Wrong list of occurrences provided");
            System.exit(1);
        }
        if (Configuration.getStore() == null) {
            return;
        }
        if (STORAGE == null) {
            load();
        }
        String batchUUID = Metrics.getUUID(occurrences.stream());
        STORAGE.getAsJsonObject(uuid).getAsJsonObject("batches").add(batchUUID, new JsonArray());
        JsonArray jsonPareto = STORAGE.getAsJsonObject(uuid).getAsJsonObject("batches").getAsJsonArray(batchUUID);
        Gson gson = new Gson();
        pareto.values().stream().map(gson::toJsonTree).forEach(jsonPareto::add);
    }

    public static Pareto get(String uuid, List<Occurrence> occurrences) {
        if (occurrences == null || occurrences.isEmpty() || occurrences.size() < 2) {
            LOGGER.severe("Wrong list of occurrences provided");
            System.exit(1);
        }
        if (Configuration.getStore() == null) {
            return null;
        }
        if (STORAGE == null) {
            load();
        }
        if (STORAGE.get(uuid) == null) {
            return null;
        }
        String batchUUID = Metrics.getUUID(occurrences.stream());
        JsonArray jsonPareto = STORAGE.getAsJsonObject(uuid).getAsJsonObject("batches").getAsJsonArray(batchUUID);
        if (jsonPareto == null) {
            return null;
        }
        Gson gson = new Gson();
        List<Metrics> paretoMetrics = new ArrayList<>();
        jsonPareto.forEach(jsonElement -> {
                    Metrics metrics = gson.fromJson(jsonElement, Metrics.class);
                    paretoMetrics.add(metrics);
                }
        );
        return new Pareto(paretoMetrics);
    }

    public static Pareto computeIfAbsent(List<Occurrence> occurrences, Supplier<Pareto> supplier) {
        Pareto pareto = get(UUID, occurrences);
        if (pareto != null) {
            return pareto;
        }

        pareto = supplier.get();
        put(UUID, occurrences, pareto);
        return pareto;
    }

    private static JsonObject loadStorage() {
        FileReader fileReader = null;
        try {
            fileReader = new FileReader(Configuration.getStore());
        } catch (FileNotFoundException ex) {
            Gson gson = new GsonBuilder().setPrettyPrinting().create();
            try (Writer writer = new FileWriter(Configuration.getStore(), true)) {
                gson.toJson(new JsonObject(), writer);
            } catch (IOException ignored) {
            }
            try {
                fileReader = new FileReader(Configuration.getStore());
            } catch (FileNotFoundException ignored) {
            }
        }
        JsonParser jsonParser = new JsonParser();
        assert fileReader != null;
        return jsonParser.parse(fileReader).getAsJsonObject();
    }

    public static void load() {
        if (Configuration.getStore() == null) {
            return;
        }
        STORAGE = loadStorage();
        UUID = Configuration.getUUID();
        if (STORAGE.get(UUID) == null) {
            LOGGER.info("Configurations not stored yet");
            STORAGE.add(UUID, new JsonObject());
            STORAGE.get(UUID).getAsJsonObject().add("configuration", Configuration.toJSONObject());
            STORAGE.get(UUID).getAsJsonObject().add("batches", new JsonObject());
        }
    }

    public static void update() {
        if (Configuration.getStore() == null) {
            return;
        }
        JsonObject storage = loadStorage();
        if (storage.get(UUID) == null) {
            storage.add(UUID, STORAGE.get(UUID));
        } else {
            STORAGE.get(UUID).getAsJsonObject().get("batches").getAsJsonObject().entrySet().forEach(entry -> {
                String computedBatchUUID = entry.getKey();
                if (storage.get(UUID).getAsJsonObject().get("batches").getAsJsonObject().get(computedBatchUUID) == null) {
                    storage.get(UUID).getAsJsonObject().get("batches").getAsJsonObject().add(computedBatchUUID, entry.getValue());
                }
            });
        }
        try (Writer writer = new FileWriter(Configuration.getStore())) {
            Gson gson = new GsonBuilder().setPrettyPrinting().create();
            gson.toJson(storage, writer);
        } catch (IOException ignored) {
        }
    }

}
