/*
 * To change this license header, choose License Headers in Project Properties.
 * To change this template file, choose Tools | Templates
 * and open the template in the editor.
 */
package br.ufrgs.inf.prosoft.tfcache;

import java.math.BigDecimal;
import java.math.BigInteger;
import java.math.MathContext;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;
import java.util.stream.Stream;

/**
 *
 * @author romulo
 */
public class Metrics implements Comparable<Metrics> {

    private long ttl;
    private long savedTime;
    private BigDecimal hitsPerTimeInCache;

    public Metrics() {
        this.hitsPerTimeInCache = new BigDecimal(BigInteger.ZERO);
    }

    public Metrics(long ttl, long savedTime, BigDecimal hitsPerTimeInCache) {
        this.ttl = ttl;
        this.savedTime = savedTime;
        this.hitsPerTimeInCache = hitsPerTimeInCache;
    }

    public long getTtl() {
        return this.ttl;
    }

    public long getSavedTime() {
        return this.savedTime;
    }

    public BigDecimal getHitsPerTimeInCache() {
        return this.hitsPerTimeInCache;
    }

    public synchronized void keepBestMetrics(long ttl, long hits, long timeInCache, long savedTime) {
        if (ttl < 0 || hits < 0 || timeInCache < 0 || savedTime < 0) {
            throw new RuntimeException("Metrics cannot be under zero. TTL: " + ttl + " hits: " + hits + " timeInCache: " + timeInCache + " savedTime: " + savedTime);
        }
        if (ttl == 0 || hits == 0) {
            return;
        }
        if (timeInCache == 0 || savedTime == 0) {
            throw new RuntimeException("timeInCache and savedTime should not be zero");
        }
        BigDecimal hitsPerTimeInCache = new BigDecimal(hits).divide(new BigDecimal(timeInCache), MathContext.DECIMAL128);
        if (this.ttl == 0
                || this.hitsPerTimeInCache.compareTo(hitsPerTimeInCache) == -1
                || (this.hitsPerTimeInCache == hitsPerTimeInCache && this.savedTime < savedTime)
                || (this.hitsPerTimeInCache == hitsPerTimeInCache && this.savedTime == savedTime && this.ttl > ttl)) {
            this.ttl = ttl;
            this.savedTime = savedTime;
            this.hitsPerTimeInCache = hitsPerTimeInCache;
        }
    }

    public static void removeDominatedMetrics(Collection<Metrics> metrics) {
        Map<Long, List<Metrics>> groupBySavedTime = metrics.stream().collect(Collectors.groupingBy(Metrics::getSavedTime));
        groupBySavedTime.forEach((savedTime, value) -> {
            if (savedTime == 0) {
                value.clear();
                return;
            }
            Metrics max = value.stream().max(Metrics::compareTo).get();
            value.removeIf(metric -> metric.getHitsPerTimeInCache().compareTo(max.getHitsPerTimeInCache()) == -1);
        });

        List<Metrics> localMaxima = groupBySavedTime.entrySet().stream()
                .map(entry -> entry.getValue().stream())
                .reduce(Stream::concat)
                .orElse(Stream.empty())
                .collect(Collectors.toList());

        metrics.removeIf(metric -> !localMaxima.contains(metric));
    }

    @Override
    public int compareTo(Metrics other) {
        if (this.hitsPerTimeInCache.compareTo(other.hitsPerTimeInCache) == 1) {
            return 1;
        }
        if (this.hitsPerTimeInCache.compareTo(other.hitsPerTimeInCache) == -1) {
            return -1;
        }
        if (this.savedTime > other.savedTime) {
            return 1;
        }
        if (this.savedTime < other.savedTime) {
            return -1;
        }
        if (this.ttl < other.ttl) {
            return 1;
        }
        if (this.ttl > other.ttl) {
            return -1;
        }
        return 0;
    }

    @Override
    public int hashCode() {
        int hash = 7;
        hash = 97 * hash + (int) (this.ttl ^ (this.ttl >>> 32));
        return hash;
    }

    @Override
    public boolean equals(Object obj) {
        if (!(obj instanceof Metrics)) {
            return false;
        }
        return compareTo((Metrics) obj) == 0;
    }

    @Override
    public Metrics clone() {
        return new Metrics(this.ttl, this.savedTime, this.hitsPerTimeInCache);
    }

    @Override
    public String toString() {
        return "TTL: " + this.ttl + " HpTiC: " + this.hitsPerTimeInCache + " SavedTime: " + this.savedTime;
    }

}
