/*
 * To change this license header, choose License Headers in Project Properties.
 * To change this template file, choose Tools | Templates
 * and open the template in the editor.
 */
package br.ufrgs.inf.prosoft.tfcache;

import br.ufrgs.inf.prosoft.tfcache.metadata.Method;
import com.google.gson.Gson;
import com.google.gson.GsonBuilder;
import com.google.gson.JsonObject;
import java.io.FileWriter;
import java.io.IOException;
import java.util.List;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.stream.Stream;

/**
 *
 * @author romulo
 */
public class TFCache {

    private static final Logger LOGGER = Logger.getLogger(TFCache.class.getName());
    private final List<Method> methods;

    public TFCache(List<Method> methods) {
        this.methods = methods;
    }

    public Stream<Method> methods() {
        return this.methods.stream();
    }

    private void recommendCacheableMethods(boolean changeable) {
        if (!changeable) {
            LOGGER.log(Level.INFO, "Removing changeable from {0} methods", this.methods.size());
            this.methods.removeIf(Method::isChangeable);
        }
        LOGGER.log(Level.INFO, "Removing not reusable from {0} methods", this.methods.size());
        this.methods.removeIf(Method::isNotReusable);
        LOGGER.log(Level.INFO, "Recommending TTL to {0} methods", this.methods.size());
        this.methods.forEach(Method::recommendTTL);
        LOGGER.log(Level.INFO, "Ranking {0} methods according saved time", this.methods.size());
        this.methods.sort((method1, method2) -> Long.compare(method2.getSavedTime(), method1.getSavedTime()));
        LOGGER.log(Level.INFO, "Printing recommendations for {0} methods", this.methods.size());
        this.methods.forEach(method -> {
            System.out.println(method.getName() + " Occurrences " + method.getOccurrencesSize() + " Inputs " + method.groupsOfOccurrences().count() + " TTL " + method.getTtl() + " HpTiC " + method.getHitsPerTimeInCache() + " Saves " + method.getSavedTime());
        });
    }

    private void recommendCacheableInputs(boolean changeable, String outputPath) {
        if (!changeable) {
            LOGGER.log(Level.INFO, "Removing changeable inputs from {0} methods", this.methods.size());
            this.methods.forEach(Method::removeChangeableInputs);
        }
        LOGGER.log(Level.INFO, "Removing not reusable inputs from {0} methods", this.methods.size());
        this.methods.forEach(Method::removeSingleOccurrences);
        LOGGER.log(Level.INFO, "Removing not reusable methods from {0} methods", this.methods.size());
        this.methods.removeIf(method -> method.groupsOfOccurrences().count() < 1);
        LOGGER.log(Level.INFO, "Recommending TTL to {0} methods", this.methods.size());
        this.methods.forEach(Method::recommendTTLPerInput);
        LOGGER.log(Level.INFO, "Ranking {0} methods and inputs according saved time", this.methods.size());
        this.methods.forEach(Method::rankRecommendations);
        this.methods.sort((method1, method2) -> Long.compare(method2.getSavedTime(), method1.getSavedTime()));
        LOGGER.log(Level.INFO, "Printing recommendations for {0} methods", this.methods.size());
        this.methods.forEach(method -> {
            System.out.println(method.getName() + " Occurrences " + method.getOccurrencesSize() + " Inputs " + method.groupsOfOccurrences().count() + " TTL " + method.getTtl() + " HpTiC " + method.getHitsPerTimeInCache() + " Saves " + method.getSavedTime());
            method.groupsOfOccurrences().forEach(group -> {
                System.out.println("\t" + group.getParameters().hashCode() + " Occurrences " + group.getOccurrencesSize() + " -> TTL " + group.getTtl() + " HpTiC " + group.getHitsPerTimeInCache() + " Saves " + group.getSavedTime());
            });
        });
        try (FileWriter fileWriter = new FileWriter(outputPath)) {
            JsonObject jsonCacheableParameters = new JsonObject();
            this.methods.forEach(method -> {
                JsonObject cacheableParameters = new JsonObject();
                method.groupsOfOccurrences().forEach(group -> {
                    cacheableParameters.addProperty(group.getParameters(), group.getTtl());
                });
                jsonCacheableParameters.add(method.getName(), cacheableParameters);
            });
            Gson gson = new GsonBuilder().setPrettyPrinting().create();
            gson.toJson(jsonCacheableParameters, fileWriter);
        } catch (IOException ex) {
            LOGGER.log(Level.SEVERE, "invalid <outputPath>");
        }
    }

    public void recommend() {
        recommend(false);
    }

    public void recommend(String outputPath) {
        recommend(false, outputPath);
    }

    public void recommend(boolean changeable) {
        recommendCacheableMethods(changeable);
    }

    public void recommend(boolean changeable, String outputPath) {
        if (outputPath == null) {
            recommend(changeable);
        } else {
            recommendCacheableInputs(changeable, outputPath);
        }
    }

}
