/*
 * To change this license header, choose License Headers in Project Properties.
 * To change this template file, choose Tools | Templates
 * and open the template in the editor.
 */
package br.ufrgs.inf.prosoft.tfcache;

import br.ufrgs.inf.prosoft.tfcache.configuration.Configuration;
import br.ufrgs.inf.prosoft.tfcache.metadata.Method;
import br.ufrgs.inf.prosoft.tfcache.metadata.Occurrence;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Collectors;
import java.util.stream.LongStream;
import java.util.stream.Stream;

/**
 *
 * @author romulo
 */
public class Simulator {

    private final Metrics metrics;

    public Simulator() {
        this.metrics = new Metrics();
    }

    public Metrics getMetrics() {
        return metrics;
    }

    public void simulate(List<Occurrence> occurrences) {
        simulate(occurrences, this.metrics);
    }

    public void simulate(Stream<Occurrence> occurrences, long ttl) {
        simulate(occurrences, ttl, this.metrics);
    }

    public static void simulate(List<Occurrence> occurrences, Metrics metrics) {
        switch (Configuration.getKernel()) {
            case "exhaustive":
                simulate(occurrences, generateAllTTLs(occurrences), metrics);
                break;
            case "optimised":
                simulate(occurrences, generateTTLsOfInterest(occurrences), metrics);
                break;
            default:
                testKernels(occurrences);
                break;
        }
    }

    public static void simulate(List<Occurrence> occurrences, Stream<Long> ttls, Metrics metrics) {
        ttls.forEach(actualTTL -> {
            simulate(occurrences.stream(), actualTTL, metrics);
        });
    }

    public static void simulate(Stream<Occurrence> occurrences, long ttl, Metrics metrics) {
        Map<String, Long> inputHasCachedTime = new HashMap<>();
        Map<String, Object> inputHasOutput = new HashMap<>();

        long simulationSavedTime = 0;
        long realSavedTime = 0;
        long hits = 0;
        long timeInCache = 0;
        long stales = 0;

        Iterator<Occurrence> iterator = occurrences.iterator();
        while (iterator.hasNext()) {
            Occurrence occurrence = iterator.next();
            long adjustedStartTime = occurrence.getStartTime() - simulationSavedTime;
            long adjustedEndTime = occurrence.getEndTime() - simulationSavedTime;
            if (inputHasCachedTime.containsKey(occurrence.getParametersSerialised())) {
                if (adjustedStartTime - inputHasCachedTime.get(occurrence.getParametersSerialised()) > ttl) {
                    inputHasCachedTime.remove(occurrence.getParametersSerialised());
                }
                if (Configuration.getStaleness().equals("shrink")) {
                    if (Objects.deepEquals(inputHasOutput.get(occurrence.getParametersSerialised()), occurrence.getReturnValue())) {
                        realSavedTime += occurrence.getExecutionTime();
                    } else {
                        stales++;
                    }
                }
                simulationSavedTime += occurrence.getExecutionTime();
                hits++;
            } else {
                inputHasCachedTime.put(occurrence.getParametersSerialised(), adjustedEndTime);
                if (Configuration.getStaleness().equals("shrink")) {
                    inputHasOutput.put(occurrence.getParametersSerialised(), occurrence.getReturnValue());
                }
                timeInCache += ttl;
            }
        }
        if (Configuration.getStaleness().equals("shrink")) {
            metrics.keepBestMetrics(ttl, hits, timeInCache, stales, realSavedTime);
        } else {
            metrics.keepBestMetrics(ttl, hits, timeInCache, stales, simulationSavedTime);
        }
    }

    public static Stream<Long> generateTTLsOfInterest(Stream<Occurrence> occurrences) {
        return generateTTLsOfInterest(occurrences.collect(Collectors.toList()));
    }

    public static Stream<Long> generateTTLsOfInterest(List<Occurrence> occurrences) {
        List<Long> windows = new ArrayList<>();
        for (int i = 1; i < occurrences.size(); i++) {
            long window = occurrences.get(i).getStartTime() - occurrences.get(i - 1).getEndTime();
            if (window > 0) {
                windows.add(window);
            }
        }

        Set<Long> ttlsOfInterest = new HashSet<>(windows);
        for (int hits = 2; hits <= windows.size(); hits++) {
            for (int shift = 0; shift <= windows.size() - hits; shift++) {
                long ttl = 0;
                for (int k = shift; k < shift + hits; k++) {
                    ttl += windows.get(k);
                }
                ttlsOfInterest.add(ttl);
            }
        }
        return ttlsOfInterest.stream().parallel();
    }

    public static Stream<Long> generateAllTTLs(List<Occurrence> occurrences) {
        long maxTTL = occurrences.get(occurrences.size() - 1).getStartTime() - occurrences.get(0).getEndTime();
        long minTTL = Long.MAX_VALUE;
        for (int i = 0; i < occurrences.size() - 1; i++) {
            long ttl = occurrences.get(i + 1).getStartTime() - occurrences.get(i).getEndTime();
            if (ttl > 0 && ttl < minTTL) {
                minTTL = ttl;
            }
        }
        return LongStream.rangeClosed(minTTL, maxTTL).boxed().parallel();
    }

    private static void testKernels(List<Occurrence> occurrences) {
        Metrics optimisedMetrics = new Metrics();

        Map<String, List<Occurrence>> inputHasOccurrences = Method.groupByInput(occurrences);
        Set<Long> ttlsOfInterest = inputHasOccurrences.values().stream()
                .map(Simulator::generateTTLsOfInterest)
                .reduce(Stream::concat)
                .get()
                .collect(Collectors.toSet());
        simulate(occurrences, ttlsOfInterest.stream(), optimisedMetrics);

        List<Metrics> exhaustiveMetrics = Collections.synchronizedList(new ArrayList<>());
        generateAllTTLs(occurrences).forEach(actualTTL -> {
            Metrics partialMetrics = new Metrics();
            exhaustiveMetrics.add(partialMetrics);
            simulate(occurrences.stream(), actualTTL, partialMetrics);
        });

        Metrics.removeDominatedMetrics(exhaustiveMetrics);
        List<Long> missingTTLs = exhaustiveMetrics.stream().map(metrics -> metrics.getTtl())
                .filter(ttl -> !ttlsOfInterest.contains(ttl))
                .sorted()
                .collect(Collectors.toList());
        if (!missingTTLs.isEmpty()) {
            System.out.println("\tMissing ttls: " + missingTTLs);
        }

        Metrics maxExhaustiveMetrics = exhaustiveMetrics.stream().max(Metrics::compareTo).orElse(new Metrics());
        if (maxExhaustiveMetrics.getTtl() != optimisedMetrics.getTtl()) {
            System.out.println("\tDIFFERENT BEST METRICS");
            System.out.println("\tOptimised: " + optimisedMetrics);
            System.out.println("\tExhaustive: " + maxExhaustiveMetrics);
            switch (maxExhaustiveMetrics.compareTo(optimisedMetrics)) {
                case -1:
                    System.out.println("\tOptimised won");
                    break;
                case 1:
                    System.out.println("\tExhaustive won");
                    break;
                default:
                    System.out.println("\tEquivalent recommendation");
                    break;
            }
        }
    }

}
