/*
 * To change this license header, choose License Headers in Project Properties.
 * To change this template file, choose Tools | Templates
 * and open the template in the editor.
 */
package br.ufrgs.inf.prosoft.tfcache.metadata;

import br.ufrgs.inf.prosoft.tfcache.Metrics;
import br.ufrgs.inf.prosoft.tfcache.Simulator;
import br.ufrgs.inf.prosoft.tfcache.StorageManager;
import br.ufrgs.inf.prosoft.tfcache.configuration.Configuration;
import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.ConcurrentHashMap;
import java.util.function.Function;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.stream.Collectors;
import java.util.stream.Stream;

/**
 *
 * @author romulo
 */
public class Method {

    private static final Logger LOGGER = Logger.getLogger(Method.class.getName());

    private final String name;
    private List<Occurrence> occurrences;
    private List<GroupOfOccurrences> groupsOfOccurrences;

    private Integer countChangeableGroups;

    private Metrics bestMetrics;

    public Method(String name, List<Occurrence> occurrences) {
        this.name = name;
        if (occurrences == null) {
            throw new RuntimeException("Occurrences is null");
        }
        if (occurrences.isEmpty()) {
            throw new RuntimeException("Occurrences is empty");
        }
        this.occurrences = occurrences;
    }

    public String getName() {
        return name;
    }

    public Metrics getBestMetrics() {
        return this.bestMetrics;
    }

    public BigDecimal getEstimatedSavedTimePerTimeInCache() {
        if (getBestMetrics() != null) {
            return getBestMetrics().getSavedTimePerTimeInCache();
        }
        GroupOfOccurrences max = groupsOfOccurrences().max((group1, group2) -> group1.getSavedTimePerTimeInCache().compareTo(group2.getSavedTimePerTimeInCache())).get();
        return max.getSavedTimePerTimeInCache();
    }

    public Stream<Occurrence> occurrences() {
        if (this.occurrences == null) {
            throw new RuntimeException("Occurrences already consumed");
        }
        return this.occurrences.stream();
    }

    public int getOccurrencesSize() {
        if (this.occurrences == null) {
            if (this.groupsOfOccurrences == null) {
                throw new RuntimeException("groupsOfOccurrences is null");
            }
            return this.groupsOfOccurrences.stream().map(group -> group.getOccurrencesSize()).reduce(Integer::sum).orElse(0);
        }
        return this.occurrences.size();
    }

    public Stream<GroupOfOccurrences> groupsOfOccurrences() {
        if (this.groupsOfOccurrences == null) {
            groupByInput();
        }
        return this.groupsOfOccurrences.stream();
    }

    private void groupByInput() {
        if (this.occurrences == null) {
            throw new RuntimeException("Occurrences already consumed");
        }
        this.groupsOfOccurrences = groupByInput(this.occurrences).entrySet().stream()
                .map(entry -> new GroupOfOccurrences(entry.getKey(), entry.getValue()))
                .collect(Collectors.toList());
    }

    public static Map<String, List<Occurrence>> groupByInput(List<Occurrence> occurrences) {
        if (occurrences == null) {
            throw new RuntimeException("Occurrences cannot be null");
        }
        Map<String, List<Occurrence>> inputHasOccurrences = new ConcurrentHashMap<>();
        occurrences.stream().forEach(occurrence -> {
            String parameters = occurrence.getParametersSerialised();

            inputHasOccurrences.compute(parameters, (key, oldValue) -> {
                if (oldValue == null) {
                    oldValue = new ArrayList<>();
                }
                oldValue.add(occurrence);
                return oldValue;
            });
        });
        return inputHasOccurrences;
    }

    public void removeChangeableInputs() {
        if (this.countChangeableGroups != null) {
            throw new RuntimeException("Changeable already filtered");
        }
        if (this.groupsOfOccurrences == null) {
            groupByInput();
        }
        int initialCountOfInputs = this.groupsOfOccurrences.size();
        this.groupsOfOccurrences.removeIf(GroupOfOccurrences::isChangeable);
        this.countChangeableGroups = initialCountOfInputs - this.groupsOfOccurrences.size();
        if (this.countChangeableGroups > 0) {
            LOGGER.log(Level.INFO, "\tRemoved {0} of {1} changeable inputs from method {2}", new Object[]{this.countChangeableGroups, initialCountOfInputs, this.name});
        }
    }

    public boolean isChangeable() {
        if (this.countChangeableGroups != null) {
            return this.countChangeableGroups > 0;
        }
        if (this.groupsOfOccurrences != null) {
            return this.groupsOfOccurrences.stream().anyMatch(GroupOfOccurrences::isChangeable);
        }
        Map<String, Object> inputHasOutput = new HashMap<>();
        for (Occurrence occurrence : this.occurrences) {
            String parameters = occurrence.getParametersSerialised();
            if (!inputHasOutput.containsKey(parameters)) {
                inputHasOutput.put(parameters, occurrence.getReturnValue());
            } else if (!Objects.deepEquals(inputHasOutput.get(parameters), occurrence.getReturnValue())) {
                return true;
            }
        }
        return false;
    }

    public void removeSingleOccurrences() {
        if (this.groupsOfOccurrences == null) {
            groupByInput();
        }
        int initialCountofOccurrences = this.groupsOfOccurrences.size();
        this.groupsOfOccurrences.removeIf(groupOfOccurrences -> groupOfOccurrences.getOccurrencesSize() < 2);
        int removedOccurrences = initialCountofOccurrences - this.groupsOfOccurrences.size();
        if (removedOccurrences > 0) {
            LOGGER.log(Level.INFO, "\tRemoved {0} of {1} inputs from method {2}", new Object[]{removedOccurrences, initialCountofOccurrences, this.name});
        }
    }

    public boolean isReusable() {
        if (this.groupsOfOccurrences != null) {
            return this.groupsOfOccurrences.stream().anyMatch(groupOfOccurrences -> groupOfOccurrences.getOccurrencesSize() > 1);
        }
        Map<String, Long> inputHasFrequency = occurrences()
                .map(Occurrence::getParametersSerialised)
                .collect(Collectors.groupingBy(Function.identity(), Collectors.counting()));
        return inputHasFrequency.values().stream().anyMatch(frequency -> frequency > 1);
    }

    public boolean isNotReusable() {
        return !isReusable();
    }

    public void recommendTTL() {
        if (this.bestMetrics != null) {
            LOGGER.log(Level.WARNING, "SavedTimePerTimeInCache already calculated");
        }
        if (this.occurrences == null) {
            throw new RuntimeException("groupByInputs called. Cannot proceed");
        }
        this.bestMetrics = new Metrics();
        if (this.occurrences.size() < 2) {
            return;
        }
        this.occurrences.sort((occurrence1, occurrence2) -> Long.compare(occurrence1.getStartTime(), occurrence2.getStartTime()));
        Simulator.simulate(this.occurrences, this.bestMetrics);
    }

    public void recommendTTLPerInput() {
        if (this.groupsOfOccurrences == null) {
            groupByInput();
            removeSingleOccurrences();
        }
        if (Configuration.getVerbose()) {
            System.out.println("=== " + getName() + " ===");
        }
        groupsOfOccurrences().parallel().forEach(GroupOfOccurrences::calculateHitsPerTimeInCache);
        String uuid = Configuration.getUUID().replace("level:input", "level:method");
        Metrics metrics = StorageManager.get(uuid, this.occurrences);
        if (this.bestMetrics == null && metrics != null) {
            this.bestMetrics = metrics;
        }
    }

    public void removeNotRecommededInputs() {
        if (this.groupsOfOccurrences == null) {
            throw new RuntimeException("Recommendations not called");
        }
        int initialCountofOccurrences = this.groupsOfOccurrences.size();
        this.groupsOfOccurrences.removeIf(groupOfOccurrences -> groupOfOccurrences.getSavedTime() == 0);
        int removedOccurrences = initialCountofOccurrences - this.groupsOfOccurrences.size();
        if (removedOccurrences > 0) {
            LOGGER.log(Level.INFO, "\tRemoved {0} of {1} inputs from method {2}", new Object[]{removedOccurrences, initialCountofOccurrences, this.name});
        }
    }

    public void rankRecommendations() {
        if (this.groupsOfOccurrences == null) {
            throw new RuntimeException("groupsOfOccurrences is null");
        }
        this.groupsOfOccurrences.sort((group1, group2) -> group2.getSavedTimePerTimeInCache().compareTo(group1.getSavedTimePerTimeInCache()));
    }

    @Override
    public String toString() {
        return this.name;
    }

}
