package br.ufrgs.inf.prosoft.adaptivecaching.sampling.metrics;

import org.apache.commons.math3.stat.descriptive.SummaryStatistics;
import org.ehcache.sizeof.SizeOf;

import java.io.IOException;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

public class LightweightMetrics {

    private long occurrences;
    private SummaryStatistics timeStatistics;
    private SummaryStatistics memoryStatistics;
    private SummaryStatistics threadStatistics;
    private SummaryStatistics intervalStatistics;
    private long thrownCounter;
    private List<Long> startTimes;
    private SummaryStatistics returnSizeStatistics;
    private static SizeOf sizeOf = SizeOf.newInstance();
    private Set<String> users;
    private String name;
    private String longName;
    private StaticMetrics staticMetrics;

    public LightweightMetrics(String name, String longName) throws IOException {
        this.name = name;
        this.longName = longName;
        startTimes = new ArrayList<>();
        timeStatistics = new SummaryStatistics();
        memoryStatistics = new SummaryStatistics();
        threadStatistics = new SummaryStatistics();
        returnSizeStatistics = new SummaryStatistics();
        intervalStatistics = new SummaryStatistics();
        intervalStatistics.addValue(0);
        users = new HashSet<>();
        staticMetrics = new StaticMetrics();
    }

    public void addTime(long startTime, long time){
        startTimes.add(startTime);
        timeStatistics.addValue(time);
        if(startTimes.size() == 1)
            intervalStatistics.addValue(0);
        else
            intervalStatistics.addValue(startTime - startTimes.get(startTimes.size() - 2));
    }

    public void addThreadNumber(long threads){
        threadStatistics.addValue(threads);
    }

    public void addMemoryConsumption(long memory){
        memoryStatistics.addValue(memory);
    }

    public void addReturnSize(Object returnObject){
        if(returnObject != null)
            returnSizeStatistics.addValue(sizeOf.sizeOf(returnObject));
        else
            returnSizeStatistics.addValue(0);
    }

    public void incThrown() {
        thrownCounter++;
    }

    public void incOccurrence() {
        occurrences++;
    }

    public void addUser(String currentUser) {
        users.add(currentUser);
    }

    public List<Long> interIntervals(){
        List<Long> intervalTimes = new ArrayList<>();
        long lastTime = startTimes.get(0);
        for(Long l : startTimes){
            intervalTimes.add(l - lastTime);
            lastTime = l;
        }
        return intervalTimes;
    }

    /**
     *
     * @return number of occurrences
     */
    public long getFrequency() {
        return occurrences;
    }

    /**
     *
     * @return number of different users
     */
    public long getUserBehavior(){
        return users.size();
    }

    /**
     *
     * @return throughput
     */
    public double getLatency(){
        return (startTimes.get(startTimes.size()-1) - startTimes.get(0)) / startTimes.size();
    }

    /**
     *
     * @return inter interval times
     */
    public double getConcurrency(){
        return intervalStatistics.getMean();
    }

    /**
     *
     * @return processing times
     */
    public double getExpensiveness(){
        return timeStatistics.getMean();
    }

    /**
     *
     * @return class size
     */
    public double getChangeability(){
        return  returnSizeStatistics.getMean();
    }

    /**
     *
     * @return number of exceptions raised
     */
    public long getDynamicErrorprone(){
        return thrownCounter;
    }

    public long getStaticErrorprone(){
        return staticMetrics.getMetrics(name).maxNesting;
    }

    public long getMaintainability(){
        return staticMetrics.getMetrics(name).cyclomatic;
    }

    public long getGlobalImpact(){
        return staticMetrics.getMetrics(name).countOutput;
    }

    public String getName() {
        return name;
    }

    public String getLongName() {
        return longName;
    }

    @Override
    public String toString() {
        return "LightweightMetrics{" +
                "name='" + name + '\'' +
                ", frequency=" + getFrequency() +
                ", expensiveness=" + getExpensiveness() +
                '}';
    }
}
