package br.ufrgs.inf.prosoft.adaptivecaching.monitoring.application.aspects;

import br.ufrgs.inf.prosoft.adaptivecaching.analysis.Analyzer;
import br.ufrgs.inf.prosoft.adaptivecaching.analysis.decision.flowchart.model.MethodEntry;
import br.ufrgs.inf.prosoft.adaptivecaching.cachemanager.cacher.AdaptiveMethodCacher;
import br.ufrgs.inf.prosoft.adaptivecaching.cachemanager.cacher.key.Key;
import br.ufrgs.inf.prosoft.adaptivecaching.cachemanager.model.ValueWrapper;
import br.ufrgs.inf.prosoft.adaptivecaching.cachemanager.util.threads.NamedThreads;
import br.ufrgs.inf.prosoft.adaptivecaching.configuration.annotation.AdaptiveCaching;
import br.ufrgs.inf.prosoft.adaptivecaching.configuration.annotation.ComponentScan;
import br.ufrgs.inf.prosoft.adaptivecaching.configuration.annotation.types.RepositoryType;
import br.ufrgs.inf.prosoft.adaptivecaching.exceptions.CacheProviderException;
import br.ufrgs.inf.prosoft.adaptivecaching.exceptions.ConfigurationException;
import br.ufrgs.inf.prosoft.adaptivecaching.exceptions.StorageException;
import br.ufrgs.inf.prosoft.adaptivecaching.monitoring.application.metadata.LogTrace;
import br.ufrgs.inf.prosoft.adaptivecaching.monitoring.application.metadata.MethodInfo;
import br.ufrgs.inf.prosoft.adaptivecaching.monitoring.application.usersession.UserGetter;
import br.ufrgs.inf.prosoft.adaptivecaching.monitoring.application.usersession.UserGetterFactory;
import br.ufrgs.inf.prosoft.adaptivecaching.monitoring.cache.CacheMonitor;
import br.ufrgs.inf.prosoft.adaptivecaching.monitoring.cache.CacheMonitorFactory;
import br.ufrgs.inf.prosoft.adaptivecaching.monitoring.storage.Repository;
import br.ufrgs.inf.prosoft.adaptivecaching.monitoring.storage.RepositoryFactory;
import br.ufrgs.inf.prosoft.adaptivecaching.sampling.metrics.LightweightAnalyzer;
import br.ufrgs.inf.prosoft.adaptivecaching.sampling.metrics.LightweightMetricAspect;
import br.ufrgs.inf.prosoft.adaptivecaching.sampling.metrics.LightweightMetrics;
import br.ufrgs.inf.prosoft.adaptivecaching.sampling.metrics.StaticMetrics;
import org.apache.commons.math3.stat.descriptive.SummaryStatistics;
import org.aspectj.lang.ProceedingJoinPoint;
import org.aspectj.lang.annotation.Around;
import org.aspectj.lang.annotation.Aspect;
import org.aspectj.lang.annotation.Pointcut;
import org.reflections.Reflections;
import org.reflections.scanners.ResourcesScanner;
import org.reflections.scanners.SubTypesScanner;
import org.reflections.scanners.TypeAnnotationsScanner;
import org.reflections.util.ClasspathHelper;
import org.reflections.util.ConfigurationBuilder;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.IOException;
import java.util.*;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentSkipListSet;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;

import static java.lang.System.currentTimeMillis;

/**
 * Aspect that traces methods and creates the logs to be analyzed
 *
 * @ComponentScan and @Ignore annotations are used to customize the locations
 */
@Aspect
public class TracerAspect {

    //used in case the method info is required
    public static Set<MethodEntry> cacheableMethods = new ConcurrentSkipListSet<>();
    //used in case the method info is not required
    public static Set<Object> cacheableMethodKeys = new ConcurrentSkipListSet<>();
    //used to list methods that raise exceptions due to tracing, they deal with internal and low level classes of spring and others, shoud be avoided
    public static List<String> methodBlackList = new ArrayList<>();
    /**
     * Enable and disable tracer
     */
    public static boolean enabled = true;
    public static boolean tracerEnabled = true;
    public static boolean samplingEnabled = true;
    public static int samplingPercentage = 50;
    public static boolean analyzerEnabled = true;
    public static boolean lightweightTracerEnabled = true;
    public static boolean lightweightAnayzerEnabled = true;

//    public static String staticFile = "/home/jhonnymertz/workspace/adaptive-caching-framework/understand/petclinic.csv";
//    public static String staticFile = "/home/jhonnymertz/workspace/adaptive-caching-framework/understand/cloudstore.csv";
    public static String staticFile = "/home/jhonnymertz/workspace/adaptive-caching-framework/understand/shopizer.csv";


//    public static SummaryStatistics timeToLightweightTrace = new SummaryStatistics();
//    public static SummaryStatistics timeToStatefulTrace = new SummaryStatistics();
//    public static SummaryStatistics timeToCheckAllowedTrace = new SummaryStatistics();

    public static Map<String, Boolean> traceMethods = null;

    private final ScheduledExecutorService analyzerExecutor = Executors.newSingleThreadScheduledExecutor(new NamedThreads(
            "adaptivecaching-analyzer",
            "identifying cacheable methods"
    ));

    private final ScheduledExecutorService lightweightAnalyzerExecutor = Executors.newSingleThreadScheduledExecutor(new NamedThreads(
            "lightweight-analyzer",
            "computing lightweight metrics and setting allowedmethods"
    ));
    Logger logger = LoggerFactory.getLogger(TracerAspect.class);

    //traceable configuration
    private String[] allowed;
    private String[] notAllowed;

    private AdaptiveMethodCacher cacher;
    private CacheMonitor cacheMonitor;

    //adaptive caching configurations
    private AdaptiveCaching cachingConfig;
    private Properties properties;
    private UserGetter userGetter;
    private Repository repository;
    private Repository repositoryMongo;

    public static Map<String, LightweightMetrics> metrics;

    @Pointcut(
            //any execution except the own framework
            "(execution(!void *(..)) && !within(br.ufrgs.inf.prosoft.adaptivecaching..*) " +
                    //avoid calls from repository while serializing objects, it is necessary if a hash could not be used
                    "&& !cflow(call(* br.ufrgs.inf.prosoft.adaptivecaching.monitoring.storage..*(..)))" +
                    //TODO remove serialization for web pages: jsp, json, el
//                    "&& !cflow(call(* org.apache.jsp..*(..)))" +
//                    "&& !within(org.apache.jsp..*)" +
//                    "&& !cflowbelow(call(* javax.el.BeanELResolver.getValue(..)))" +
                    //conditional to enable and disable at runtime
                    "&& if())"
    )
    public static boolean anyCall() {
        return enabled || lightweightTracerEnabled;
    }

    //TODO Decouple repository from this
    public TracerAspect() throws IOException {
        initialize();
    }

    private void initialize() throws IOException {
        try {
            Class<?> configClass = getAvailableConfigurationClass();
            cachingConfig = configClass.getAnnotation(AdaptiveCaching.class);
            logger.info("AdaptiveCaching found, loading options...");

            if (!cachingConfig.enabled()) {
                turnoff();
                logger.info("Adaptive caching disabled manually on @AdaptiveCaching.");
                return;
            }

            //getting allowed packages from @ComponentScan
            ComponentScan componentScanConfig = configClass.getAnnotation(ComponentScan.class);
            if (componentScanConfig == null) {
                //if not specified, it assumes the same package where @AdaptiveCaching were declared
                allowed = new String[]{configClass.getPackage().getName()};
                //logger.error("ComponenScan for AdaptiveCaching not found, adaptive caching disabled.");
                //enabled = false;
                //return;
            }
            allowed = componentScanConfig.allowed();
            notAllowed = componentScanConfig.denied();
            logger.info("@AdaptiveCaching will trace and cache methods into {} packages...", allowed);

            //todo load uncacheable annotation and methods and add to the blacklist
            //todo load cacheable annotation and methods and add to the must-cache list

            if (cachingConfig.tracerEnabled()) {

                //setting up the repository
                properties = getPropertiesFile();
                if (properties == null || properties.isEmpty()) {
                    logger.error("adaptivecaching.properties is not defined, adaptive caching disabled.");
                    enabled = false;
                } else
                    logger.info("adaptivecaching.properties found, loading properties...");


                repository = RepositoryFactory.getRepository(properties, cachingConfig.logRepository());
                repositoryMongo = RepositoryFactory.getRepository(properties, RepositoryType.MONGODB);

                if (cachingConfig.clearMonitoringDataOnStart()) {
                    repository.removeAll();
                    logger.debug("Repository starting cleaned.");
                }

                //TODO switch and build: pass a parameter and build
                userGetter = UserGetterFactory.getInstance();

                this.cacher = new AdaptiveMethodCacher(cachingConfig.cacheProvider(), cachingConfig.expiryInterval());
                this.cacheMonitor = CacheMonitorFactory.getCacheMonitor(this.cacher.getCache(), cachingConfig.cacheProvider());

                //TODO load another options from @AdaptiveCaching
                configureAnalyzer();

                //TODO trigger by time
                //TODO in some cases (Ehcache) it is better to set a timetolive directly on cache provider
//            this.expirationExecutor.scheduleWithFixedDelay(
//                    new VerboseRunnable(() -> TracerAspect.this.clean()),
//                    cachingConfig.firstExpiry(), cachingConfig.expiryInterval(), TimeUnit.MILLISECONDS
//            );

//        if (adaptiveConfig.analyzerEnabled()) {
//            this.loaderExecutor.scheduleWithFixedDelay(
//                    new VerboseRunnable(() -> TracerAspect.this.cacheableMethodsloader()),
//                    1, 10000, TimeUnit.SECONDS
//            );
//        }
                methodBlackList = new ArrayList<>();
                traceMethods = new ConcurrentHashMap<>();

            } else tracerEnabled = false;


            if(lightweightTracerEnabled) {
                metrics = new ConcurrentHashMap<>();
                if (lightweightTracerEnabled)
                    lightweightAnalyzerExecutor.scheduleWithFixedDelay(
                            new LightweightAnalyzer(),
                            120000, 450000, TimeUnit.MILLISECONDS);
            }

        } catch (ConfigurationException e) {
            turnoff();
            logger.error("An error was found while trying to get the AdaptiveCaching annotation.", e);
        } catch (StorageException e) {
            turnoff();
            logger.error("Cannot connect with the specified repository, adaptive caching disabled.", e);
        } catch (CacheProviderException e) {
            turnoff();
            logger.error("Cannot connect with the specified cache, adaptive caching disabled.", e);
        }
    }

    private void configureAnalyzer() {
        //TODO get time from properties or see the external process
        if (cachingConfig.analyzerEnabled()) {
            //TODO trigger by time
            analyzerExecutor.scheduleWithFixedDelay(
                    new Analyzer(repository, cacheMonitor.getCacheInfo(), cachingConfig),
                    cachingConfig.firstAnalysis(), cachingConfig.analysisInterval(), TimeUnit.MILLISECONDS);
        } else TracerAspect.analyzerEnabled = false;
    }

    @Around("anyCall()")
    public Object aroundMethods(ProceedingJoinPoint joinPoint) throws Throwable {

        //DEBUG: see if a method is being caught
        //DEBUG: traceSpecificMethod(joinPoint);

//        long timeToCheckAllowed = currentTimeMillis();

        if (!isAllowed(joinPoint))
            return joinPoint.proceed();

        boolean detailedTrace = (!lightweightTracerEnabled ||
                (lightweightTracerEnabled && LightweightAnalyzer.allowedFineGrained
                        .contains(joinPoint.getSignature().toLongString())));
//        boolean detailedTrace = (LightweightAnalyzer.allowedFineGrained
//                        .contains(joinPoint.getSignature().toLongString()));

//        timeToCheckAllowedTrace.addValue(currentTimeMillis() - timeToCheckAllowed);

        boolean cacheable = false;
        Key key = null;
        if(detailedTrace) {
            //generate a hash of the method that will be used as: key to cache and compare if the method is allowed or not
            key = new Key(joinPoint);

            cacheable = cacheableMethodKeys.contains(key);

            if (cacheable) {
                Object cachedResult = this.cacher.getFromCache(key);
                if (cachedResult != null)
                    return cachedResult;
            }
        }

        //when method is already cached and obtained from it, no trace will be generated
        Object result = trace(key, joinPoint, detailedTrace);

        if (detailedTrace && cacheable && result != null) {
            this.cacher.putInCache(key, result);
        }

        return result;
    }

//    private int samplingIndex = 0;
//    private boolean shouldBeSampled() {
//        if(!samplingEnabled)
//            return true;
//
////        if((rand.nextInt(10) + 1) > samplingChance)
////            return true;
////        synchronized (this){
//            samplingIndex++;
//            if ((samplingIndex % samplingEach) == 0)
//                return true;
//            else return false;
////        }
//    }

    private Random rand = new Random();
    private boolean shouldBeSampled() {
        return !samplingEnabled || ((rand.nextInt(100) + 1) > (100 - samplingPercentage));
    }

    private boolean isAllowed(ProceedingJoinPoint joinPoint) throws Throwable {
        String signature = joinPoint.getSignature().toString();
        Boolean status = traceMethods.get(signature);
        if(status != null)
            return status;

        if (signature.contains("br.ufrgs.inf.prosoft.adaptivecaching")) {
            traceMethods.put(signature, false);
            return false;
        }

        for (String p : notAllowed) {
            //toString to avoid get the return class as a false positive
            if (signature.contains(p)) {
                traceMethods.put(signature, false);
                return false;
            }
        }

        //trace only allowed packages
        for (String p : allowed) {
            if (joinPoint.getSignature().toLongString().contains(p)
                    && !methodBlackList.contains(joinPoint.getSignature().toLongString())) {
                traceMethods.put(signature, true);
                return true;
            }
        }

        traceMethods.put(signature, false);
        return false;
    }

    private Object trace(Key key, ProceedingJoinPoint joinPoint, boolean detailedTrace) throws Throwable {

        Object[] joinPointArgs = null;
        //method calls can change the args, so it is better to get it at the beginning
        if(detailedTrace)
            joinPointArgs = joinPoint.getArgs();

        String signature = joinPoint.getSignature().toString();

        long startTime = currentTimeMillis();
        Object result = joinPoint.proceed();
        long endTime = currentTimeMillis();

        String user = userGetter.getCurrentUser();

        if(lightweightTracerEnabled) {
//            long timeToLightTrace = currentTimeMillis();
            LightweightMetrics metric = metrics.get(signature);
            if (metric != null)
                metric.incOccurrence();
            else {
                String name = joinPoint.getSignature().toString().split(" ")[1];
                name = name.substring(0, name.indexOf("("));
                metric = new LightweightMetrics(name, joinPoint.getSignature().toLongString());
                metric.incOccurrence();
                if (TracerAspect.analyzerEnabled)
                    metrics.put(signature, metric);
            }
            metric.addTime(startTime, endTime - startTime);
            metric.addReturnSize(result);
            metric.addUser(user);
//            timeToLightweightTrace.addValue(currentTimeMillis() - timeToLightTrace);
        }

        //trace only allowed by lightweight metrics
        if(tracerEnabled
            && detailedTrace
            && shouldBeSampled()) {
//            System.out.println("New trace: " + signature + LightweightAnalyzer.allowedFineGrained
//                    .contains(joinPoint.getSignature().toLongString()));
//            long timeToTrace = currentTimeMillis();

            //we do not cache null returns, but we trace them
            //maybe the method can sometimes return null... so there is not verification here
            LogTrace logTrace = new LogTrace();
            logTrace.setStartTime(startTime);
            logTrace.setEndTime(endTime);

            //TODO declare on properties the class name which implements UserGetter, parse and initialize
            logTrace.setUserId(user);

            MethodInfo methodInfo = new MethodInfo(joinPoint.getSignature().toLongString(), joinPointArgs, result, key);
            logTrace.setMethodInfo(methodInfo);

            try {
                //could not trace async or batch due to database sessions, when saving the session could be closed already
                if (analyzerEnabled)
                    repository.save(logTrace);
                else repositoryMongo.save(logTrace);
                logger.debug("New trace entry: " + logTrace);// " serialization and save time: " + (System.currentTimeMillis() - la));
            } catch (Exception e) {
                logger.debug("Couldn't trace " + logTrace.getMethodInfo().getSignature() + " due to: " + e.getMessage());// + " process time: " + (System.currentTimeMillis() - la), e);
                logger.debug("Adding " + logTrace.getMethodInfo().getSignature() + " to blacklist");
                TracerAspect.methodBlackList.add(logTrace.getMethodInfo().getSignature());
            }
//            timeToStatefulTrace.addValue(currentTimeMillis() - timeToTrace);
        }

        return result;
    }

    private Class<?> getAvailableConfigurationClass() {
        //TODO Decouple??
        Reflections reflections = new Reflections(
                new ConfigurationBuilder()
                        .setUrls(ClasspathHelper.forClassLoader())
                        .setScanners(new SubTypesScanner(false), new ResourcesScanner(), new TypeAnnotationsScanner()));

        //loading @AdaptiveCaching
        Set<Class<?>> configurations =
                reflections.getTypesAnnotatedWith(AdaptiveCaching.class);
        if (configurations.isEmpty())
            throw new ConfigurationException("@AdaptiveCaching not found, adaptive caching disabled.");
        if (configurations.size() > 1)
            throw new ConfigurationException("@AdaptiveCaching has too many definitions, adaptive caching disabled.");
        return configurations.iterator().next();
    }

    private void turnoff() {
        enabled = false;
        tracerEnabled = false;
        analyzerEnabled = false;
    }

    private Properties getPropertiesFile() throws IOException {
        Properties properties = new Properties();
        //file location:
        //System.out.println(getClass().getClassLoader().getResource("adaptivecaching.properties").getPath());
        properties.load(getClass().getClassLoader().getResourceAsStream("adaptivecaching.properties"));
        return properties;
    }

    /**
     * Load cacheable methods
     */
//    private void cacheableMethodsloader() {
//        try {
//            //TODO get db info from properties
//            MongoClient mongo = new MongoClient("localhost", 27017);
//            MongoDatabase database = mongo.getDatabase("cachemonitoring");
//            Repository cacheableRepository = new MongoRepository<MethodEntry>(database.getCollection("petclinicCacheable"), MethodEntry.class);
//
//            TracerAspect.cacheableMethods = (Set<MethodEntry>) cacheableRepository.findAll();
//            if (!TracerAspect.cacheableMethods.isEmpty()) {
//                logger.info(TracerAspect.cacheableMethods.size() + " cacheable methods loaded. Starting to work with them...");
//                if (cachingConfig.disableMonitoringAfterAnalysis()) {
//                    TracerAspect.enabled = false;
//                    logger.info("Tracer disabled after analysis...");
//                }
//            }
//        } catch (MongoTimeoutException e) {
//            logger.error("Cannot connect with MongoDB to get the cacheable methods.", e);
//        }
//    }
//
//    public void traceBatch() {
//        if (tempTraces.size() > 50000) {
//            logger.debug("Maximum number of traces in memory achieved. Analyzing them...");
//            Thread.currentThread().setPriority(Thread.MAX_PRIORITY);
//
//            TracerAspect.tracerEnabled = false;
//            logger.debug("Disabling traces...");
//
//            Analyzer analyzer = new Analyzer(repository, repository, cacheMonitor.getCacheInfo(), cachingConfig);
//
////            List<LogTrace> toSave = new ArrayList<LogTrace>(traces.size());
//            synchronized (tempTraces) {
//                Set<MethodEntry> cacheable = analyzer.analyzeAndReturn(tempTraces);
//                for (MethodEntry ma : cacheable)
//                    TracerAspect.cacheableMethods = cacheable;
//
////                logger.info("Trying to save the traces: " + traces.size());
////                repository.saveAll(traces);
//                tempTraces.clear();
//                logger.debug("Traces list clear: " + tempTraces.size());
//            }
//            //TODO get the future and reenables the trace?
//        }
//        TracerAspect.tracerEnabled = true;
//    }
//
//    private void traceSpecificMethod(ProceedingJoinPoint joinPoint) {
//        if (joinPoint.getSignature().toLongString().contains("findOwnerById")) {
//            System.out.println("pointcut: " + joinPoint);
//            for (StackTraceElement st : new Throwable().getStackTrace()) {
//                System.out.println(st.getClassName() + ":" + st.getMethodName());
//            }
//        }
//    }


//    hashing example:
//    methodInfo.setArguments(HashCodeBuilder.reflectionHashCode(joinPoint.getArgs()));
//    methodInfo.setReturnedValue(HashCodeBuilder.reflectionHashCode(result));
}