package br.ufrgs.inf.prosoft.adaptivecaching.cachemanager.extensions.guava;

import br.ufrgs.inf.prosoft.adaptivecaching.cachemanager.model.Cache;
import br.ufrgs.inf.prosoft.adaptivecaching.cachemanager.model.CacheManager;
import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheBuilderSpec;
import com.google.common.cache.CacheLoader;
import org.springframework.util.ObjectUtils;

import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;

/**
 * {@link CacheManager} implementation that lazily builds {@link GuavaCache}
 * instances for each {@link #getCache} request. Also supports a 'static' mode
 * where the set of cachemanager names is pre-defined through {@link #setCacheNames},
 * with no dynamic creation of further cachemanager regions at runtime.
 * <p>
 * <p>The configuration of the underlying cachemanager can be fine-tuned through a
 * Guava {@link CacheBuilder} or {@link CacheBuilderSpec}, passed into this
 * CacheManager through {@link #setCacheBuilder}/{@link #setCacheBuilderSpec}.
 * A {@link CacheBuilderSpec}-compliant expression value can also be applied
 * via the {@link #setCacheSpecification "cacheSpecification"} bean property.
 * <p>
 * <p>Requires Google Guava 12.0 or higher.
 *
 */
public class GuavaCacheManager implements CacheManager {

    private final ConcurrentMap<String, Cache> cacheMap = new ConcurrentHashMap<String, Cache>(16);

    private boolean dynamic = true;

    private CacheBuilder<Object, Object> cacheBuilder = CacheBuilder.newBuilder();

    private CacheLoader<Object, Object> cacheLoader;

    private boolean allowNullValues = true;


    /**
     * Construct a dynamic GuavaCacheManager,
     * lazily creating cachemanager instances as they are being requested.
     */
    public GuavaCacheManager() {
    }

    /**
     * Construct a static GuavaCacheManager,
     * managing caches for the specified cachemanager names only.
     */
    public GuavaCacheManager(String... cacheNames) {
        setCacheNames(Arrays.asList(cacheNames));
    }

    /**
     * Set the Guava CacheBuilder to use for building each individual
     * {@link GuavaCache} instance.
     *
     * @see #createNativeGuavaCache
     * @see CacheBuilder#build()
     */
    public void setCacheBuilder(CacheBuilder<Object, Object> cacheBuilder) {
//		Assert.notNull(cacheBuilder, "CacheBuilder must not be null");
        doSetCacheBuilder(cacheBuilder);
    }

    /**
     * Set the Guava CacheBuilderSpec to use for building each individual
     * {@link GuavaCache} instance.
     *
     * @see #createNativeGuavaCache
     * @see CacheBuilder#from(CacheBuilderSpec)
     */
    public void setCacheBuilderSpec(CacheBuilderSpec cacheBuilderSpec) {
        doSetCacheBuilder(CacheBuilder.from(cacheBuilderSpec));
    }

    /**
     * Set the Guava cachemanager specification String to use for building each
     * individual {@link GuavaCache} instance. The given value needs to
     * comply with Guava's {@link CacheBuilderSpec} (see its javadoc).
     *
     * @see #createNativeGuavaCache
     * @see CacheBuilder#from(String)
     */
    public void setCacheSpecification(String cacheSpecification) {
        doSetCacheBuilder(CacheBuilder.from(cacheSpecification));
    }

    /**
     * Set the Guava CacheLoader to use for building each individual
     * {@link GuavaCache} instance, turning it into a LoadingCache.
     *
     * @see #createNativeGuavaCache
     * @see CacheBuilder#build(CacheLoader)
     * @see com.google.common.cache.LoadingCache
     */
    public void setCacheLoader(CacheLoader<Object, Object> cacheLoader) {
        if (!ObjectUtils.nullSafeEquals(this.cacheLoader, cacheLoader)) {
            this.cacheLoader = cacheLoader;
            refreshKnownCaches();
        }
    }

    /**
     * Return whether this cachemanager manager accepts and converts {@code null} values
     * for all of its caches.
     */
    public boolean isAllowNullValues() {
        return this.allowNullValues;
    }

    /**
     * Specify whether to accept and convert {@code null} values for all caches
     * in this cachemanager manager.
     * <p>Default is "true", despite Guava itself not supporting {@code null} values.
     * An internal holder object will be used to store user-level {@code null}s.
     */
    public void setAllowNullValues(boolean allowNullValues) {
        if (this.allowNullValues != allowNullValues) {
            this.allowNullValues = allowNullValues;
            refreshKnownCaches();
        }
    }

    @Override
    public Collection<String> getCacheNames() {
        return Collections.unmodifiableSet(this.cacheMap.keySet());
    }

    /**
     * Specify the set of cachemanager names for this CacheManager's 'static' mode.
     * <p>The number of caches and their names will be fixed after a call to this method,
     * with no creation of further cachemanager regions at runtime.
     * <p>Calling this with a {@code null} collection argument resets the
     * mode to 'dynamic', allowing for further creation of caches again.
     */
    public void setCacheNames(Collection<String> cacheNames) {
        if (cacheNames != null) {
            for (String name : cacheNames) {
                this.cacheMap.put(name, createGuavaCache(name));
            }
            this.dynamic = false;
        } else {
            this.dynamic = true;
        }
    }

    @Override
    public Cache getCache(String name) {
        Cache cache = this.cacheMap.get(name);
        if (cache == null && this.dynamic) {
            synchronized (this.cacheMap) {
                cache = this.cacheMap.get(name);
                if (cache == null) {
                    cache = createGuavaCache(name);
                    this.cacheMap.put(name, cache);
                }
            }
        }
        return cache;
    }

    /**
     * Create a new GuavaCache instance for the specified cachemanager name.
     *
     * @param name the name of the cachemanager
     * @return the Spring GuavaCache adapter (or a decorator thereof)
     */
    protected Cache createGuavaCache(String name) {
        return new GuavaCache(name, createNativeGuavaCache(name), isAllowNullValues());
    }

    /**
     * Create a native Guava Cache instance for the specified cachemanager name.
     *
     * @param name the name of the cachemanager
     * @return the native Guava Cache instance
     */
    protected com.google.common.cache.Cache<Object, Object> createNativeGuavaCache(String name) {
        if (this.cacheLoader != null) {
            return this.cacheBuilder.recordStats().build(this.cacheLoader);
        } else {
            return this.cacheBuilder.recordStats().build();
        }
    }

    private void doSetCacheBuilder(CacheBuilder<Object, Object> cacheBuilder) {
        if (!ObjectUtils.nullSafeEquals(this.cacheBuilder, cacheBuilder)) {
            this.cacheBuilder = cacheBuilder;
            refreshKnownCaches();
        }
    }

    /**
     * Create the known caches again with the current state of this manager.
     */
    private void refreshKnownCaches() {
        for (Map.Entry<String, Cache> entry : this.cacheMap.entrySet()) {
            entry.setValue(createGuavaCache(entry.getKey()));
        }
    }

}
