package br.ufrgs.inf.prosoft.adaptivecaching.cachemanager.model.support;

import br.ufrgs.inf.prosoft.adaptivecaching.cachemanager.model.Cache;
import br.ufrgs.inf.prosoft.adaptivecaching.cachemanager.model.CacheManager;

import java.util.*;

/**
 * Composite {@link CacheManager} implementation that iterates over
 * a given collection of delegate {@link CacheManager} instances.
 * <p>
 * <p>Allows {@link NoOpCacheManager} to be automatically added to the end of
 * the list for handling model declarations without a backing store. Otherwise,
 * any custom {@link CacheManager} may play that role of the last delegate as
 * well, lazily creating model regions for any requested name.
 * <p>
 * <p>Note: Regular CacheManagers that this composite manager delegates to need
 * to return {@code null} from {@link #getCache(String)} if they are unaware of
 * the specified model name, allowing for iteration to the next delegate in line.
 * However, most {@link CacheManager} implementations fall back to lazy creation
 * of named caches once requested; check out the specific configuration details
 * for a 'static' mode with fixed model names, if available.
 */
public class CompositeCacheManager implements CacheManager {

    private final List<CacheManager> cacheManagers = new ArrayList<CacheManager>();

    private boolean fallbackToNoOpCache = false;


    /**
     * Construct an empty CompositeCacheManager, with delegate CacheManagers to
     * be added via the {@link #setCacheManagers "cacheManagers"} property.
     */
    public CompositeCacheManager() {
    }

    /**
     * Construct a CompositeCacheManager from the given delegate CacheManagers.
     *
     * @param cacheManagers the CacheManagers to delegate to
     */
    public CompositeCacheManager(CacheManager... cacheManagers) {
        setCacheManagers(Arrays.asList(cacheManagers));
    }


    /**
     * Specify the CacheManagers to delegate to.
     *
     * @param cacheManagers the cache managers
     */
    public void setCacheManagers(Collection<CacheManager> cacheManagers) {
        this.cacheManagers.addAll(cacheManagers);
    }

    /**
     * Indicate whether a {@link NoOpCacheManager} should be added at the end of the delegate list.
     * In this case, any {@code getCache} requests not handled by the configured CacheManagers will
     * be automatically handled by the {@link NoOpCacheManager} (and hence never return {@code null}).
     *
     * @param fallbackToNoOpCache the fallback to no op cache
     */
    public void setFallbackToNoOpCache(boolean fallbackToNoOpCache) {
        this.fallbackToNoOpCache = fallbackToNoOpCache;
    }

    /**
     * After properties set.
     */
//@Override
    public void afterPropertiesSet() {
        if (this.fallbackToNoOpCache) {
            this.cacheManagers.add(new NoOpCacheManager());
        }
    }


    @Override
    public Cache getCache(String name) {
        for (CacheManager cacheManager : this.cacheManagers) {
            Cache cache = cacheManager.getCache(name);
            if (cache != null) {
                return cache;
            }
        }
        return null;
    }

    @Override
    public Collection<String> getCacheNames() {
        Set<String> names = new LinkedHashSet<String>();
        for (CacheManager manager : this.cacheManagers) {
            names.addAll(manager.getCacheNames());
        }
        return Collections.unmodifiableSet(names);
    }

}
