package br.ufrgs.inf.prosoft.adaptivecaching.monitoring.application.aspects;

import br.ufrgs.inf.prosoft.adaptivecaching.analysis.Analyzer;
import br.ufrgs.inf.prosoft.adaptivecaching.analysis.decision.flowchart.model.MethodEntry;
import br.ufrgs.inf.prosoft.adaptivecaching.cachemanager.cacher.key.Key;
import br.ufrgs.inf.prosoft.adaptivecaching.cachemanager.extensions.ehcache.EhCacheCache;
import br.ufrgs.inf.prosoft.adaptivecaching.cachemanager.extensions.ehcache.EhCacheCacheManager;
import br.ufrgs.inf.prosoft.adaptivecaching.cachemanager.extensions.guava.GuavaCache;
import br.ufrgs.inf.prosoft.adaptivecaching.cachemanager.extensions.guava.GuavaCacheManager;
import br.ufrgs.inf.prosoft.adaptivecaching.cachemanager.model.Cache;
import br.ufrgs.inf.prosoft.adaptivecaching.cachemanager.model.CacheManager;
import br.ufrgs.inf.prosoft.adaptivecaching.cachemanager.util.support.ValueWrapper;
import br.ufrgs.inf.prosoft.adaptivecaching.cachemanager.util.threads.NamedThreads;
import br.ufrgs.inf.prosoft.adaptivecaching.cachemanager.util.threads.VerboseRunnable;
import br.ufrgs.inf.prosoft.adaptivecaching.configuration.annotation.AdaptiveCaching;
import br.ufrgs.inf.prosoft.adaptivecaching.configuration.annotation.ComponentScan;
import br.ufrgs.inf.prosoft.adaptivecaching.monitoring.application.customization.AnonymousUserGetter;
import br.ufrgs.inf.prosoft.adaptivecaching.monitoring.application.customization.SpringUserGetter;
import br.ufrgs.inf.prosoft.adaptivecaching.monitoring.application.customization.UserGetter;
import br.ufrgs.inf.prosoft.adaptivecaching.monitoring.application.metadata.LogTrace;
import br.ufrgs.inf.prosoft.adaptivecaching.monitoring.application.metadata.MethodInfo;
import br.ufrgs.inf.prosoft.adaptivecaching.monitoring.cache.CacheMonitor;
import br.ufrgs.inf.prosoft.adaptivecaching.monitoring.cache.vendors.ehcache.EhCacheMonitor;
import br.ufrgs.inf.prosoft.adaptivecaching.monitoring.cache.vendors.guava.GuavaMonitor;
import br.ufrgs.inf.prosoft.adaptivecaching.monitoring.storage.*;
import com.mongodb.MongoClient;
import com.mongodb.MongoTimeoutException;
import com.mongodb.client.MongoDatabase;
import net.sf.ehcache.config.CacheConfiguration;
import net.sf.ehcache.config.PersistenceConfiguration;
import net.sf.ehcache.management.ManagementService;
import net.sf.ehcache.store.MemoryStoreEvictionPolicy;
import org.aspectj.lang.ProceedingJoinPoint;
import org.aspectj.lang.annotation.Around;
import org.aspectj.lang.annotation.Aspect;
import org.aspectj.lang.annotation.Pointcut;
import org.reflections.Reflections;
import org.reflections.scanners.ResourcesScanner;
import org.reflections.scanners.SubTypesScanner;
import org.reflections.scanners.TypeAnnotationsScanner;
import org.reflections.util.ClasspathHelper;
import org.reflections.util.ConfigurationBuilder;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.management.MBeanServer;
import java.io.File;
import java.io.IOException;
import java.lang.management.ManagementFactory;
import java.util.*;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;

import static java.lang.System.currentTimeMillis;

/**
 * Aspect that traces methods and creates the logs to be analyzed
 *
 * @ComponentScan and @Ignore annotations are used to customize the locations
 */
@Aspect
public class TracerAspect {

    public static Set<MethodEntry> cacheableMethods;
    public static List<String> methodBlackList;
    /**
     * Enable and disable tracer
     */
    public static boolean enabled = true;
    public static boolean tracerEnabled = true;
    public static boolean analyzerEnabled = true;

    private final ScheduledExecutorService analyzerExecutor = Executors.newSingleThreadScheduledExecutor(new NamedThreads(
            "adaptivecaching-analyzer",
            "identifying cacheable methods"
    ));
    private final ScheduledExecutorService expirationExecutor = Executors.newSingleThreadScheduledExecutor(new NamedThreads(
            "adaptivecaching-expiration",
            "expiring old cacheable methods"
    ));
    private final ScheduledExecutorService statsExecutor = Executors.newSingleThreadScheduledExecutor(new NamedThreads(
            "adaptivecaching-stats",
            "showing stats of the framework"
    ));
    private final ScheduledExecutorService loaderExecutor = Executors.newSingleThreadScheduledExecutor(new NamedThreads(
            "adaptivecaching-loading",
            "loading cacheable methods from storage (offline analyzed)"
    ));
    private final ExecutorService tracerExecutor = Executors.newSingleThreadScheduledExecutor(
            new NamedThreads(
                    "adaptivecaching-tracer",
                    "tracing methods"
            ));
    Logger logger = LoggerFactory.getLogger(TracerAspect.class);
    private Repository repository;
    private List<String> allowedPackages;
    private List<String> notAllowedPackages;
    private CacheManager cacheManager;
    private Cache cache;
    private CacheMonitor cacheMonitor;
    private Analyzer analyzer;
    private AdaptiveCaching cachingConfig;
    private Properties properties;
    private UserGetter userGetter;
    private List<LogTrace> tempTraces;
    private int count;
    private long hashAndStructureTime;
    private long traceTime;

    //TODO Decouple repository from this
    public TracerAspect() throws IOException {
        initialize();
    }

    @Pointcut(
            //any execution except the own framework
            "(execution(!void *(..)) && !within(br.ufrgs.inf.prosoft.adaptivecaching..*) " +
                    //avoid calls from repository while serializing objects, it is necessary if a hash could not be used
                    "&& !cflow(call(* br.ufrgs.inf.prosoft.adaptivecaching.monitoring.storage..*(..)))" +
                    //TODO remove serialization for web pages: jsp, json, el
//                    "&& !cflow(call(* org.apache.jsp..*(..)))" +
//                    "&& !within(org.apache.jsp..*)" +
//                    "&& !cflowbelow(call(* javax.el.BeanELResolver.getValue(..)))" +
                    //conditional to enable and disable at runtime
                    "&& if())"
    )
    public static boolean anyCall() {
        return enabled;
    }

    private void initialize() throws IOException {

        //TODO Decouple
        Reflections reflections = new Reflections(
                new ConfigurationBuilder()
                        .setUrls(ClasspathHelper.forClassLoader())
                        .setScanners(new SubTypesScanner(false), new ResourcesScanner(), new TypeAnnotationsScanner()));

        //loading @AdaptiveCaching
        Set<Class<?>> configurations =
                reflections.getTypesAnnotatedWith(AdaptiveCaching.class);
        if (configurations.isEmpty()) {
            logger.error("@AdaptiveCaching not found, adaptive caching disabled.");
            enabled = false;
            return;
        }
        if (configurations.size() > 1) {
            logger.error("@AdaptiveCaching has too many definitions, adaptive caching disabled.");
            enabled = false;
            return;
        }
        Class<?> configClass = configurations.iterator().next();
        AdaptiveCaching adaptiveConfig = configClass.getAnnotation(AdaptiveCaching.class);
        cachingConfig = adaptiveConfig;
        logger.info("AdaptiveCaching found, loading options...");

        if (!cachingConfig.enabled()) {
            logger.error("Adaptive caching disabled manually on @AdaptiveCaching.");
            return;
        }

        //getting allowed packages from @ComponentScan
        ComponentScan componentScanConfig = configClass.getAnnotation(ComponentScan.class);
        if (componentScanConfig == null) {
            //TODO if not specified, it assumes the same package where @AdaptiveCaching were declared
            logger.error("ComponenScan for AdaptiveCaching not found, adaptive caching disabled.");
            enabled = false;
            return;
        }

        allowedPackages = Arrays.asList(componentScanConfig.allowed());
        notAllowedPackages = Arrays.asList(componentScanConfig.denied());
        logger.info("@AdaptiveCaching will trace and cache methods into {} packages...", allowedPackages);

        if (adaptiveConfig.tracerEnabled()) {
            //setting up the repository
            properties = new Properties();
            //file location:
            //System.out.println(getClass().getClassLoader().getResource("adaptivecaching.properties").getPath());
            properties.load(getClass().getClassLoader().getResourceAsStream("adaptivecaching.properties"));
            if (properties == null || properties.isEmpty()) {
                logger.error("adaptivecaching.properties is not defined, adaptive caching disabled.");
                enabled = false;
            } else
                logger.info("adaptivecaching.properties found, loading properties...");

            switch (cachingConfig.logRepository()) {
                case MONGODB:
                    //TODO decoupĺe this with a factory
                    try {
                        MongoClient mongo = new MongoClient(properties.getProperty("adaptivecaching.monitoring.db.address"), Integer.parseInt(properties.getProperty("adaptivecaching.monitoring.db.port")));
                        MongoDatabase database = mongo.getDatabase(properties.getProperty("adaptivecaching.monitoring.db.scheme"));
                        repository = new MongoRepository<LogTrace>(database.getCollection(properties.getProperty("adaptivecaching.monitoring.db.name")), LogTrace.class);
                        logger.info("Repository is configured to MongoDB: " + repository.toString());
                    } catch (MongoTimeoutException e) {
                        logger.error("Cannot connect with MongoDB with the defined properties, adaptive caching disabled.", e);
                        tracerEnabled = false;
                    }

                    break;
                case REDIS:
                    //TODO decouple it from redis and get of properties
                    repository = new RedisRepository<LogTrace>();
                    logger.debug("Repository is configured to Redis.");
                    break;
                case TEXTFILE:
                    AsyncFileWriter as = new AsyncFileWriter(new File("traces.txt"));
                    as.open();
                    repository = as;
                    break;
                case MEMORY:
                    //TODO log a warning because memory can take too much from the application...
                    repository = new MemoryRepository<LogTrace>();
                    logger.debug("Repository is configured to save logs in Memory.");
                    break;
            }

            if (cachingConfig.clearMonitoringDataOnStart()) {
                repository.removeAll();
                logger.debug("Repository starting cleaned.");
            }

            //TODO switch and build
            try {
                userGetter = new SpringUserGetter();
            } catch (Exception e) {
                userGetter = new AnonymousUserGetter();
            }

        } else tracerEnabled = false;

        switch (cachingConfig.cacheProvider()) {
            case GUAVA:
                this.cacheManager = new GuavaCacheManager();
                this.cache = cacheManager.getCache("test");
                this.cacheMonitor = new GuavaMonitor((GuavaCache) this.cache);
                logger.debug("Cache provider is configured to Guava.");
                break;
            case MEMCACHED:
                break;
            case REDIS:
                //redis cache manager
//        JedisConnectionFactory cf = new JedisConnectionFactory();
//        cf.setHostName("127.0.0.1");
//        cf.setPort(6379);
//        RedisTemplate<String, String> redisTemplate = new RedisTemplate<String, String>();
//        redisTemplate.setConnectionFactory(cf);
//        RedisCacheManager redisCacheManager = new RedisCacheManager(redisTemplate);
//        redisCacheManager.setDefaultExpiration(300);
//        this.cacheManager = redisCacheManager;
//        this.cache = redisCacheManager.getCache("test");
                break;
            case CAFFEINE:
                break;
            case EHCACHE:
                net.sf.ehcache.CacheManager cm = net.sf.ehcache.CacheManager.newInstance();

                //monitoring hit and miss ratio:
                MBeanServer mBeanServer = ManagementFactory.getPlatformMBeanServer();
                ManagementService.registerMBeans(cm, mBeanServer, false, false, false, true);

                //Create a Cache specifying its configuration.
                net.sf.ehcache.Cache adaptiveCache = new net.sf.ehcache.Cache(
                        new CacheConfiguration("adaptivecaching", 10000)
                                .memoryStoreEvictionPolicy(MemoryStoreEvictionPolicy.LRU)
                                .eternal(true)
                                .overflowToOffHeap(false)
                                .persistence(new PersistenceConfiguration().strategy(PersistenceConfiguration.Strategy.NONE)));
                cm.addCache(adaptiveCache);
                this.cacheManager = new EhCacheCacheManager(cm);
                this.cache = cacheManager.getCache("adaptivecaching");
                this.cacheMonitor = new EhCacheMonitor((EhCacheCache) this.cache);
                logger.debug("Cache provider is configured to EhCache.");
                break;
        }

        //TODO load another options from @AdaptiveCaching

        //TODO get time from properties or see the external process
        if (adaptiveConfig.analyzerEnabled()) {
            TracerAspect.analyzerEnabled = true;
            //TODO trigger by time
            //maybe we should kill it or delete the object?
            analyzerExecutor.scheduleWithFixedDelay(new Analyzer(repository, cacheMonitor.getCacheInfo(), adaptiveConfig), 2, 5, TimeUnit.MINUTES);
        }

        //TODO trigger by time
        this.expirationExecutor.scheduleWithFixedDelay(
                new VerboseRunnable(
                        new Runnable() {
                            @Override
                            public void run() {
                                TracerAspect.this.clean();
                            }
                        }
                ),
                6, adaptiveConfig.expiry(), TimeUnit.MINUTES
        );

//        if (adaptiveConfig.analyzerEnabled()) {
//            this.loaderExecutor.scheduleWithFixedDelay(
//                    new VerboseRunnable(() -> TracerAspect.this.cacheableMethodsloader()),
//                    1, 10000, TimeUnit.SECONDS
//            );
//        }

//        this.statsExecutor.scheduleWithFixedDelay(
//                new VerboseRunnable(() -> TracerAspect.this.stats()),
//                10, 5, TimeUnit.SECONDS
//        );

        methodBlackList = new ArrayList<>();
        tempTraces = Collections.synchronizedList(new ArrayList<>());
    }

    @Around("anyCall()")
    public Object aroundMethods(ProceedingJoinPoint joinPoint) throws Throwable {

        //see if a method is being caught
//        if (joinPoint.getSignature().toLongString().contains("findOwnerById")) {
//            System.out.println("pointcut: " + joinPoint);
//            for (StackTraceElement st : new Throwable().getStackTrace()){
//                System.out.println(st.getClassName() + ":" + st.getMethodName());
//            }
//        }

        //when method cached, no trace will be generated
        //caching methods
        //todo optimize this logic to build and compare methods
        if (cacheableMethods != null) {
            for (MethodEntry methodAnalysis : cacheableMethods) {
                if (joinPoint.getSignature().toLongString().equals(methodAnalysis.getMethodInfo().getSignature())) {

                    MethodInfo methodInfo = new MethodInfo(joinPoint.getSignature().toLongString(), joinPoint.getArgs());

                    //TODO hash or not???
                    //        methodInfo.setArguments(HashCodeBuilder.reflectionHashCode(joinPoint.getArgs()));
                    //        methodInfo.setReturnedValue(HashCodeBuilder.reflectionHashCode(result));

                    if (methodAnalysis.getMethodInfo().equalsWithoutReturnedValue(methodInfo)) {
                        //todo we should trace cached methods in order to provide the runtime, otherwise the cached method
                        //will not be cacheable on the second time
                        return cache(joinPoint);
                    }
                }
            }
        }

        for (String p : notAllowedPackages) {
            //toString to avoid get the return class as a false positive
            if (joinPoint.getSignature().toString().contains(p)) {
                return joinPoint.proceed();
            }
        }

        //trace only allowed packages
        boolean shouldTrace = false;
        for (String p : allowedPackages) {
            if (joinPoint.getSignature().toLongString().contains(p)
                    && !methodBlackList.contains(joinPoint.getSignature().toLongString()))
                shouldTrace = true;
            break;
        }
        if (!shouldTrace)
            return joinPoint.proceed();

        if (tracerEnabled)
            return trace(joinPoint);
        else
            return joinPoint.proceed();
    }

    private Object trace(ProceedingJoinPoint joinPoint) throws Throwable {
        long startTime = currentTimeMillis();
        Object result = joinPoint.proceed();
        long endTime = currentTimeMillis();

        //we do not cache null returns, but we trace them
        //maybe the method can sometimes return null
//        if (result == null)
//            return null;

        LogTrace logTrace = new LogTrace();
        logTrace.setStartTime(startTime);
        logTrace.setEndTime(endTime);

        //TODO declare on properties the class name which implements UserGetter, parse and initialize
        logTrace.setUserId(userGetter.getCurrentUser());

        MethodInfo methodInfo = new MethodInfo(joinPoint.getSignature().toLongString(), joinPoint.getArgs(), result);
        logTrace.setMethodInfo(methodInfo);

        //in case of batch processing...
//        traces.add(logTrace);

//        count++;
//        System.out.println(count);

//        temp = currentTimeMillis();
        //could not trace async due to database sessions, when saving the session could be closed already.
        //TODO solutions: configure jackson to avoid empty attributes / configure hibernate version as a datatype module
//        Tracer tracer = new Tracer(repository, logTrace);
//        if (cachingConfig.traceAsync()) {
//            tracerExecutor.execute(tracer);
//        } else {
//            tracer.run();
//        }

//        long la = System.currentTimeMillis();
        try {
            repository.save(logTrace);
            logger.debug("New trace entry: " + logTrace);// " serialization and save time: " + (System.currentTimeMillis() - la));
        } catch (Exception e) {
            logger.debug("Couldn't trace " + logTrace.getMethodInfo().getSignature() + " due to: " + e.getMessage());// + " process time: " + (System.currentTimeMillis() - la), e);
            logger.debug("Adding " + logTrace.getMethodInfo().getSignature() + " to blacklist");
            TracerAspect.methodBlackList.add(logTrace.getMethodInfo().getSignature());
        }

//        traceTime += System.currentTimeMillis() - temp;

        return result;
    }

    private Object cache(ProceedingJoinPoint joinPoint) throws Throwable {

        final Key key = new Key(joinPoint);
        ValueWrapper value = cache.get(key);

        if (value != null) {
            logger.debug(key + " with value: " + value.get() + " got from cachemanager");
            return value.get();
        }

        Object result;

        try {
            result = joinPoint.proceed();
        } catch (Exception ex) {
            throw ex;
        }

        if (result != null) {
            //todo maximize such decision
            //maybe getcache info a lot can lead to a performance problem
//            if(cacheMonitor.getCacheInfo().getFreeSpace() > 0) {
            cache.put(key, result);
            logger.debug(key + " with value: " + result + " set in cachemanager");
//            }
        }

        return result;
    }

    /**
     * Clean cacheable methods expired.
     */
    private void clean() {
        cache.clear();
        logger.info("Expired cached values were removed.");
    }

    private void stats() {
        logger.debug("Tempo hash: " + hashAndStructureTime + " Tempo save: " + traceTime + " Count: " + count);
        logger.debug("Mean Tempo hash: " + (hashAndStructureTime / count) + " Tempo save: " + (traceTime / count));
    }

    /**
     * Load cacheable methods
     */
    private void cacheableMethodsloader() {
        try {
            //TODO get db info from properties
            MongoClient mongo = new MongoClient("localhost", 27017);
            MongoDatabase database = mongo.getDatabase("cachemonitoring");
            Repository cacheableRepository = new MongoRepository<MethodEntry>(database.getCollection("petclinicCacheable"), MethodEntry.class);

            TracerAspect.cacheableMethods = (Set<MethodEntry>) cacheableRepository.findAll();
            if (!TracerAspect.cacheableMethods.isEmpty()) {
                logger.info(TracerAspect.cacheableMethods.size() + " cacheable methods loaded. Starting to work with them...");
                if (cachingConfig.disableMonitoringAfterAnalysis()) {
                    TracerAspect.enabled = false;
                    logger.info("Tracer disabled after analysis...");
                }
            }
        } catch (MongoTimeoutException e) {
            logger.error("Cannot connect with MongoDB to get the cacheable methods.", e);
        }
    }

    public void traceBatch() {
        if (tempTraces.size() > 50000) {
            logger.debug("Maximum number of traces in memory achieved. Analyzing them...");
            Thread.currentThread().setPriority(Thread.MAX_PRIORITY);

            TracerAspect.tracerEnabled = false;
            logger.debug("Disabling traces...");

            Analyzer analyzer = new Analyzer(repository, repository, cacheMonitor.getCacheInfo(), cachingConfig);

//            List<LogTrace> toSave = new ArrayList<LogTrace>(traces.size());
            synchronized (tempTraces) {
                Set<MethodEntry> cacheable = analyzer.analyzeAndReturn(tempTraces);
                for (MethodEntry ma : cacheable)
                    TracerAspect.cacheableMethods = cacheable;

//                logger.info("Trying to save the traces: " + traces.size());
//                repository.saveAll(traces);
                tempTraces.clear();
                logger.debug("Traces list clear: " + tempTraces.size());
            }
            //TODO get the future and reenables the trace?
        }
        TracerAspect.tracerEnabled = true;
    }

    //see if a method is being caught
//        if (joinPoint.getSignature().toLongString().contains("showVetList")) {
//            System.out.println("pointcut: " + joinPoint);
//            for (StackTraceElement st : new Throwable().getStackTrace()){
//                System.out.println(st.getClassName());
//            }
//        }
}
