package br.ufrgs.inf.prosoft.adaptivecaching.cachemanager.cacher.extensions.provided;

import br.ufrgs.inf.prosoft.adaptivecaching.cachemanager.cacher.key.Key;
import br.ufrgs.inf.prosoft.adaptivecaching.cachemanager.cacher.MethodCacher;
import br.ufrgs.inf.prosoft.adaptivecaching.cachemanager.cacher.UpdateMethodCacher;
import br.ufrgs.inf.prosoft.adaptivecaching.cachemanager.util.Mnemos;
import org.aspectj.lang.JoinPoint;
import org.aspectj.lang.ProceedingJoinPoint;
import org.aspectj.lang.reflect.MethodSignature;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.lang.reflect.Method;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.LinkedBlockingQueue;

/**
 * Cache method results.
 * <p>
 * <p>The class is thread-safe.
 *
 * @author Yegor Bugayenko (yegor@tpc2.com)
 * @version $Id$
 * @since 0.8
 */
public final class MyMethodCacher implements MethodCacher {

    /**
     * Calling tunnels.
     */
    private final transient
    ConcurrentMap<Key, Tunnel> tunnels;
    /**
     * Save the keys of caches which need update.
     */
    private final transient BlockingQueue<Key> updatekeys;
    Logger logger = LoggerFactory.getLogger(UpdateMethodCacher.class);

    /**
     * Public ctor.
     */
    public MyMethodCacher() {
        this.tunnels =
                new ConcurrentHashMap<Key, Tunnel>(0);
        this.updatekeys =
                new LinkedBlockingQueue<Key>();
        new UpdateMethodCacher(this.tunnels, this.updatekeys).start();
    }

    /**
     * Call the method or fetch from model.
     * <p>
     * <p>Try NOT to change the signature of this method, in order to keep
     * it backward compatible.
     *
     * @param point Joint point
     * @return The result of call
     * @throws Throwable If something goes wrong inside
     * @checkstyle IllegalThrows (4 lines)
     */
//    @Around("execution(* *(..))")
    public Object cache(final ProceedingJoinPoint point) throws Throwable {
        final Key key = new Key(point);
        Tunnel tunnel;
        final Method method = MethodSignature.class
                .cast(point.getSignature())
                .getMethod();

        synchronized (this.tunnels) {

            tunnel = this.tunnels.get(key);
            if (this.isCreateTunnel(tunnel)) {
                tunnel = new Tunnel(
                        point, key, false//annot.asyncUpdate()
                );
                this.tunnels.put(key, tunnel);
            }
            if (tunnel.expired() && tunnel.asyncUpdate()) {
                this.updatekeys.offer(key);
            }
        }
        return tunnel.through();
    }

    /**
     * Flush model.
     * <p>
     * <p>Try NOT to change the signature of this method, in order to keep
     * it backward compatible.
     *
     * @param point Joint point
     * @since 0.7.14
     */
//    @Before
//            (
//                    // @checkstyle StringLiteralsConcatenation (3 lines)
//                    "execution(* *(..))"
//                            + " && (@configuration(com.jcabi.aspects.Cacheable.Flush)"
//                            + " || @configuration(com.jcabi.aspects.Cacheable.FlushBefore))"
//            )
    public void preflush(final JoinPoint point) {
        this.flush(point, "before the call");
    }

    /**
     * Flush model after method execution.
     * <p>
     * <p>Try NOT to change the signature of this method, in order to keep
     * it backward compatible.
     *
     * @param point Joint point
     * @since 0.7.18
     */
//    @After
//            (
//                    // @checkstyle StringLiteralsConcatenation (2 lines)
//                    "execution(* *(..))"
//                            + " && @configuration(com.jcabi.aspects.Cacheable.FlushAfter)"
//            )
    public void postflush(final JoinPoint point) {
        this.flush(point, "after the call");
    }

    /**
     * Flush model.
     *
     * @param point Joint point
     * @param when  When it happens
     * @since 0.7.18
     */
    private void flush(final JoinPoint point, final String when) {
        synchronized (this.tunnels) {
            for (final Key key : this.tunnels.keySet()) {
                if (!key.sameTarget(point)) {
                    continue;
                }
                final Tunnel removed = this.tunnels.remove(key);
                final Method method = MethodSignature.class
                        .cast(point.getSignature())
                        .getMethod();
//                if (LogHelper.enabled(
//                        key.getLevel(), method.getDeclaringClass()
//                )) {
                logger.debug(
                        "%s: %s:%s removed from model %s",
                        Mnemos.toText(method, point.getArgs(), true, false),
                        key,
                        removed,
                        when
                );
//                }
            }
        }
    }

    /**
     * Whether create a new Tunnel.
     *
     * @param tunnel Tunnel
     * @return Boolean
     */
    private boolean isCreateTunnel(final Tunnel tunnel) {
        return tunnel == null || (tunnel.expired() && !tunnel.asyncUpdate());
    }
}

