package br.ufrgs.inf.prosoft.adaptivecaching.cachemanager.cacher;

import br.ufrgs.inf.prosoft.adaptivecaching.cachemanager.cacher.extensions.provided.Tunnel;
import br.ufrgs.inf.prosoft.adaptivecaching.cachemanager.cacher.key.Key;
import br.ufrgs.inf.prosoft.adaptivecaching.cachemanager.util.threads.NamedThreads;
import br.ufrgs.inf.prosoft.adaptivecaching.cachemanager.util.threads.VerboseRunnable;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.concurrent.*;

/**
 * Copyright (c) 2012-2015, jcabi.com
 * All rights reserved.
 * <p>
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met: 1) Redistributions of source code must retain the above
 * copyright notice, this list of conditions and the following
 * disclaimer. 2) Redistributions in binary form must reproduce the above
 * copyright notice, this list of conditions and the following
 * disclaimer in the documentation and/or other materials provided
 * with the distribution. 3) Neither the name of the jcabi.com nor
 * the names of its contributors may be used to endorse or promote
 * products derived from this software without specific prior written
 * permission.
 * <p>
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT
 * NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL
 * THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
 * OF THE POSSIBILITY OF SUCH DAMAGE.
 */

/**
 * For updating model and cleaning expired model.
 *
 * @author Peng Chuanjiang (pengchuanjiang@souyidai.com)
 * @version $Id$
 */
@SuppressWarnings("PMD.DoNotUseThreads")
public final class UpdateMethodCacher {

    /**
     * Calling tunnels.
     */
    private final transient
    ConcurrentMap<Key, Tunnel> tunnels;
    /**
     * Service that cleans model.
     */
    private final transient ScheduledExecutorService cleaner;
    /**
     * Save the keys of caches which need update.
     */
    private final transient BlockingQueue<Key> updatekeys;
    /**
     * Service that update model.
     */
    private final transient ScheduledExecutorService updater;
    Logger logger = LoggerFactory.getLogger(UpdateMethodCacher.class);

    /**
     * Public ctor.
     *
     * @param tls   ConcurrentMap
     * @param ukeys BlockingQueue
     */
    public UpdateMethodCacher(
            final ConcurrentMap<Key, Tunnel> tls,
            final BlockingQueue<Key> ukeys) {
        this.tunnels = tls;
        this.updatekeys = ukeys;
        this.cleaner = Executors.newSingleThreadScheduledExecutor(
                new NamedThreads(
                        "cacheable-clean",
                        "automated cleaning of expired cacheable values"
                )
        );
        this.updater = Executors.newSingleThreadScheduledExecutor(
                new NamedThreads(
                        "cacheable-update",
                        "async update of expired cacheable values"
                )
        );
    }

    /**
     * Staring two threads(cheaner and updater) are used for
     * updating model and cleaning expired model.
     */
    public void start() {
        this.cleaner.scheduleWithFixedDelay(
                new VerboseRunnable(
                        new Runnable() {
                            @Override
                            public void run() {
                                UpdateMethodCacher.this.clean();
                            }
                        }
                ),
                1L, 1L, TimeUnit.SECONDS
        );
        this.updater.schedule(
                new VerboseRunnable(
                        new Runnable() {
                            @Override
                            public void run() {
                                UpdateMethodCacher.this.update();
                            }
                        }
                ),
                0L, TimeUnit.SECONDS
        );
    }

    /**
     * Clean model.
     */
    private void clean() {
        synchronized (this.tunnels) {
            for (final Key key : this.tunnels.keySet()) {
                if (this.tunnels.get(key).expired()
                        && !this.tunnels.get(key).asyncUpdate()) {
                    final Tunnel tunnel = this.tunnels.remove(key);
                    logger.debug(
                            "%s:%s expired in model",
                            key,
                            tunnel
                    );
                }
            }
        }
    }

    /**
     * Update model.
     */
    @SuppressWarnings("PMD.AvoidCatchingThrowable")
    private void update() {
        while (true) {
            try {
                final Key key = this.updatekeys.take();
                final Tunnel tunnel = this.tunnels.get(key);
                if (tunnel != null && tunnel.expired()) {
                    final Tunnel newTunnel = tunnel.copy();
                    newTunnel.through();
                    this.tunnels.put(key, newTunnel);
                }
            } catch (final InterruptedException ex) {
                logger.error(
                        "%s:%s",
                        ex.getMessage(),
                        ex
                );
                // @checkstyle IllegalCatch (1 line)
            } catch (final Throwable ex) {
                logger.error(
                        "Exception message is %s, Exception is %s",
                        ex.getMessage(),
                        ex
                );
            }
        }
    }

}
