package br.ufrgs.inf.prosoft.adaptivecaching.cachemanager.cacher.extensions.provided;

import br.ufrgs.inf.prosoft.adaptivecaching.cachemanager.cacher.UpdateMethodCacher;
import br.ufrgs.inf.prosoft.adaptivecaching.cachemanager.cacher.key.Key;
import br.ufrgs.inf.prosoft.adaptivecaching.cachemanager.util.Mnemos;
import org.aspectj.lang.ProceedingJoinPoint;
import org.aspectj.lang.reflect.MethodSignature;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.lang.ref.SoftReference;
import java.lang.reflect.Method;

/**
 * Mutable caching/calling tunnel, it is thread-safe.
 */
public final class Tunnel {

    /**
     * Proceeding join point.
     */
    private final transient ProceedingJoinPoint point;
    /**
     * Key related to this tunnel.
     */
    private final transient Key key;
    /**
     * Whether asynchronous update.
     */
    private final transient boolean async;
    Logger logger = LoggerFactory.getLogger(UpdateMethodCacher.class);
    /**
     * Was it already executed?
     */
    private transient boolean executed;
    /**
     * When will it expire (moment in time).
     */
    private transient long lifetime;
    /**
     * Has non-null result?
     */
    private transient boolean hasresult;
    /**
     * Cached value.
     */
    @SuppressWarnings("PMD.AvoidFieldNameMatchingMethodName")
    private transient SoftReference<Object> cached;

    /**
     * Public ctor.
     *
     * @param pnt  ProceedingJoinPoint
     * @param akey MethodCacher.Key
     * @param asy  Boolean
     */
    public Tunnel(final ProceedingJoinPoint pnt,
                  final Key akey, final boolean asy) {
        this.point = pnt;
        this.key = akey;
        this.async = asy;
    }

    @Override
    public String toString() {
        return Mnemos.toText(this.cached.get(), true, false);
    }

    /**
     * Get a new instance.
     *
     * @return MethodCacher.Tunnel
     */
    public Tunnel copy() {
        return new Tunnel(
                this.point, this.key, this.async
        );
    }

    /**
     * Get a result through the tunnel.
     *
     * @return The result
     * @throws Throwable If something goes wrong inside
     * @checkstyle IllegalThrows (5 lines)
     */
    @SuppressWarnings("PMD.AvoidSynchronizedAtMethodLevel")
    public synchronized Object through() throws Throwable {
        if (!this.executed) {
            final long start = System.currentTimeMillis();
            final Object result = this.point.proceed();
            this.hasresult = result != null;
            this.cached = new SoftReference<Object>(result);
            final Method method = MethodSignature.class
                    .cast(this.point.getSignature())
                    .getMethod();
//            final Cacheable annot = method.getAnnotation(Cacheable.class);
            final String suffix;
//            if (annot.forever()) {
//                this.lifetime = Long.MAX_VALUE;
//                suffix = "valid forever";
//            } else if (annot.lifetime() == 0) {
//                this.lifetime = 0L;
//                suffix = "invalid immediately";
//            } else {
//                final long msec = annot.unit().toMillis(
//                        (long) annot.lifetime()
//                );
//                this.lifetime = start + msec;
//                suffix = Logger.format("valid for %[ms]s", msec);
//            }
            final Class<?> type = method.getDeclaringClass();
//            if (LogHelper.enabled(this.key.getLevel(), type)) {
            logger.debug(
                    "%s: %s cached in %[ms]s, %s",
                    Mnemos.toText(
                            method, this.point.getArgs(), true, false
                    ),
                    Mnemos.toText(this.cached.get(), true, false),
                    System.currentTimeMillis() - start
//                        suffix
            );
//            }
            this.executed = true;
        }
        return this.key.through(this.cached.get());
    }

    /**
     * Is it expired already?
     *
     * @return TRUE if expired
     */
    public boolean expired() {
        final boolean expired = this.lifetime < System.currentTimeMillis();
        final boolean collected = this.executed
                && this.hasresult
                && this.cached.get() == null;
        return this.executed && (expired || collected);
    }

    /**
     * Whether asynchronous update.
     *
     * @return TRUE if asynchronous update
     */
    public boolean asyncUpdate() {
        return this.async;
    }

    /**
     * Soft reference to cached object.
     * Visible only for testing. Do not use directly.
     *
     * @return Soft reference to cached object.
     */
    @SuppressWarnings("PMD.DefaultPackage")
    public SoftReference<Object> cached() {
        return this.cached;
    }
}