package br.ufrgs.inf.prosoft.adaptivecaching.analysis.decision.flowchart;

import br.ufrgs.inf.prosoft.adaptivecaching.analysis.decision.CacheDecider;
import br.ufrgs.inf.prosoft.adaptivecaching.analysis.decision.Processer;
import br.ufrgs.inf.prosoft.adaptivecaching.analysis.decision.flowchart.model.MethodEntry;
import br.ufrgs.inf.prosoft.adaptivecaching.analysis.decision.flowchart.model.MethodStats;
import br.ufrgs.inf.prosoft.adaptivecaching.analysis.decision.flowchart.stats.CacheabilityMetrics;
import br.ufrgs.inf.prosoft.adaptivecaching.monitoring.application.aspects.TracerAspect;
import br.ufrgs.inf.prosoft.adaptivecaching.monitoring.application.metadata.LogTrace;
import br.ufrgs.inf.prosoft.adaptivecaching.monitoring.application.metadata.MethodInfo;
import br.ufrgs.inf.prosoft.adaptivecaching.monitoring.cache.CacheInfo;
import org.apache.commons.lang3.builder.EqualsBuilder;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.PrintWriter;
import java.math.BigDecimal;
import java.math.RoundingMode;
import java.util.*;
import java.util.stream.Collectors;

public class FlowchartWorkFlow implements Processer {

    private final long population;
    protected HashMap<MethodInfo, MethodStats> methodsInfoMap;
    Logger logger = LoggerFactory.getLogger(FlowchartWorkFlow.class);

    private CacheDecider decider;
    private CacheInfo cacheInfo;

    private double sumMissRatio;
    private double sumHitRatio;
    private double sumExecutionTime;
    private double sumShareability;
    private double sumFrequency;
    private List<Double> hitRatios = new ArrayList<>();
    private List<Double> missRatios = new ArrayList<>();
    private List<Long> executionTimes = new ArrayList<>();
    private List<Double> shareabilities = new ArrayList<>();
    private List<Long> frequencies = new ArrayList<>();

    public FlowchartWorkFlow(CacheDecider decider, CacheInfo cacheInfo, List<LogTrace> logList) {
        this.decider = decider;
        this.cacheInfo = cacheInfo;
        this.methodsInfoMap = countOccurrences(logList);
        this.population = logList.size();

        logger.debug(methodsInfoMap.size() + " unique method calls identified from " + logList.size() + " original traces");

        //sorting to get threshold, also excluding duplicates
        Collections.sort(executionTimes);
        hitRatios = hitRatios.stream().distinct().collect(Collectors.toList());
        Collections.sort(hitRatios);
        missRatios = missRatios.stream().distinct().collect(Collectors.toList());
        Collections.sort(missRatios);
        shareabilities = shareabilities.stream().distinct().collect(Collectors.toList());
        Collections.sort(shareabilities);

        logger.debug("Average ExecutionTime: " + getAverageExecutionTime());
        logger.debug("Average HitRatio: " + getAverageHitRatio());
        logger.debug("Average MissRatio: " + getAverageMissRatio());
        logger.debug("Average shareability: " + getAverageShareability());
        logger.debug("StdDv ExecutionTime: " + getStdDevExecutionTimeRatio());
        logger.debug("StdDv HitRatio: " + getStdDevHitRatio());
        logger.debug("StdDv MissRatio: " + getStdDevMissRatio());
        logger.debug("StdDv shareability: " + getStdDevShareability());
        logger.debug("StdDv frequency: " + getStdDevFrequency());

        int k = 0;
        logger.debug("Using " + k + " stdDev to calculate thresholds...");
        logger.debug("Threshold ExecutionTime: " + expensivenessThreshold(k));
        logger.debug("Threshold HitRatio: " + hitThreshold(k));
        logger.debug("Threshold MissRatio: " + missThreshold(k));
        logger.debug("Threshold Shareability: " + shareabilityThreshold(k));
        logger.debug("Threshold frequency: " + frequencyThreshold(k));
    }


    public Set<MethodEntry> filterCacheableMethods(long expiryTime) {
        logger.debug("Deciding if methods are cacheable...");

        Set<MethodEntry> cacheableMethods = getMethodsInfoMap().keySet().stream()
                .filter(mi -> decider.isCacheable(cacheInfo, mi, getMethodsInfoMap().get(mi), this))
                .map(mi -> new MethodEntry(mi, getMethodsInfoMap().get(mi), System.currentTimeMillis() + expiryTime))
                .collect(Collectors.toSet());

        logger.info(cacheableMethods.size() + " cacheable methods detected. Printing files...");

        //TODO remove: print all unique methods and metrics to csv file
        try {
            final PrintWriter pw = new PrintWriter(new File("allmethods.csv"));
            pw.write("isStaticData,changeMoreThanUsed,usedByManyRequests,isUserSpecific,isCacheSizeLarge,isDataSizeLarge,isExpensive,signature,numberOfSameOccurrences,numberOfDifferentReturnOccurrences,totalOccurrences,sameOccurrencesAverageExecutionTime,sameOccurrencesTotalExecutionTime,hitRatio,missRatio\n");
            getMethodsInfoMap().keySet().stream().forEach(mi -> pw.write(CacheabilityMetrics.allMetricsToString(cacheInfo, mi, getMethodsInfoMap().get(mi), this, getMethodsInfoMap().size()) + "," + new MethodEntry(mi, getMethodsInfoMap().get(mi), System.currentTimeMillis() + expiryTime).getMethodInfo().getSignature() + "," + new MethodEntry(mi, getMethodsInfoMap().get(mi), System.currentTimeMillis() + expiryTime).getMethodStats().toCSV() + '\n'));
            pw.close();
        } catch (FileNotFoundException e) {
            e.printStackTrace();
        }

        //TODO remove: print cacheable methods to csv file
        try {
            final PrintWriter pw = new PrintWriter(new File("cacheablemethods.csv"));
            pw.write("isStaticData,changeMoreThanUsed,usedByManyRequests,isUserSpecific,isCacheSizeLarge,isDataSizeLarge,isExpensive,signature,numberOfSameOccurrences,numberOfDifferentReturnOccurrences,totalOccurrences,sameOccurrencesAverageExecutionTime,sameOccurrencesTotalExecutionTime,hitRatio,missRatio\n");
            cacheableMethods.stream().forEach(ma -> pw.write(CacheabilityMetrics.allMetricsToString(cacheInfo, ma.getMethodInfo(), ma.getMethodStats(), this, getMethodsInfoMap().size()) + "," + ma.getMethodInfo().getSignature() + "," + ma.getMethodStats().toCSV() + '\n'));
            pw.close();
        } catch (FileNotFoundException e) {
            e.printStackTrace();
        }

        //TODO remove: print black list to csv file
        try {
            if (TracerAspect.methodBlackList != null && !TracerAspect.methodBlackList.isEmpty()) {
                final PrintWriter pw = new PrintWriter(new File("backlisted.csv"));
                pw.write("signature\n");
                TracerAspect.methodBlackList.stream().forEach(ma -> pw.write(ma + '\n'));
                pw.close();
            }
        } catch (FileNotFoundException e) {
            e.printStackTrace();
        }

        return cacheableMethods;
    }

    public HashMap<MethodInfo, MethodStats> countOccurrences(List<LogTrace> logs) {

        sumExecutionTime = 0;
        sumHitRatio = 0;
        sumMissRatio = 0;
        sumShareability = 0;
        sumFrequency = 0;

        HashMap<MethodInfo, MethodStats> methodInfoMap = new HashMap<>();

        for (int i = 0; i < logs.size(); i++) {
            LogTrace logTrace = logs.get(i);

            if (methodInfoMap.containsKey(logTrace.getMethodInfo()))
                continue;

            MethodStats methodStats = new MethodStats(logTrace);

            for (int j = 0; j < logs.size(); j++) {
                LogTrace traceCompare = logs.get(j);

                if(i == j)
                    continue;

                //if similar methods: same signature and params, different return
//                if(logTrace.getMethodInfo().getHashedArguments())
                if (traceCompare.getMethodInfo().equalsWithoutReturnedValue(logTrace.getMethodInfo())
//                      || traceCompare.getMethodInfo().equalsHashedWithoutReturnedValue(logTrace.getMethodInfo())
                        ) {

                    //if identical methods
                    if (EqualsBuilder.reflectionEquals(traceCompare.getMethodInfo().getReturnedValue(), logTrace.getMethodInfo().getReturnedValue())
//                            || Objects.equals(traceCompare.getMethodInfo().getHashedReturnedValue(), logTrace.getMethodInfo().getHashedReturnedValue())
)
                        methodStats.addSameOccurrence(traceCompare);

                    else methodStats.addDifferentReturnOccurrence();
                }
            }

            methodInfoMap.put(logTrace.getMethodInfo(), methodStats);

            sumExecutionTime += methodStats.getSameOccurrencesTotalExecutionTime();
            executionTimes.add(methodStats.getSameOccurrencesTotalExecutionTime());

            sumHitRatio += methodStats.hitRatio();
            hitRatios.add(methodStats.hitRatio());

            sumMissRatio += methodStats.missRatio();
            missRatios.add(methodStats.missRatio());

            sumShareability += methodStats.shareability();
            shareabilities.add(methodStats.shareability());

            sumFrequency += methodStats.getNumberOfSameOccurrences();
            frequencies.add(methodStats.getNumberOfSameOccurrences());
        }

        return methodInfoMap;
    }

    /**
     * General mean hit ratio of all calls
     *
     * @return
     */
    public double getAverageHitRatio() {
        return new BigDecimal(sumHitRatio).divide(new BigDecimal(population), 5, RoundingMode.HALF_UP).doubleValue();
    }

    public double getAverageMissRatio() {
        return new BigDecimal(sumMissRatio).divide(new BigDecimal(population), 5, RoundingMode.HALF_UP).doubleValue();
    }

    public double getAverageExecutionTime() {
        return new BigDecimal(sumExecutionTime).divide(new BigDecimal(population), 5, RoundingMode.HALF_UP).doubleValue();
    }

    public double getAverageShareability() {
        return new BigDecimal(sumShareability).divide(new BigDecimal(population), 5, RoundingMode.HALF_UP).doubleValue();
    }

    public double getStdDevHitRatio() {
        //todo salvar em mémoria e não ficar recalculando
        double mean = getAverageHitRatio();
        double temp = 0;
        for (double a : hitRatios)
            temp += (a - mean) * (a - mean);
        return Math.sqrt(temp / population);
    }

    public double getStdDevMissRatio() {
        //todo salvar em mémoria e não ficar recalculando
        double mean = getAverageMissRatio();
        double temp = 0;
        for (double a : missRatios)
            temp += (a - mean) * (a - mean);
        return Math.sqrt(temp / population);
    }

    public double getStdDevExecutionTimeRatio() {
        //todo salvar em mémoria e não ficar recalculando
        double mean = getAverageExecutionTime();
        double temp = 0;
        for (Long a : executionTimes)
            temp += (a - mean) * (a - mean);
        return Math.sqrt(temp / population);
    }

    public double getStdDevFrequency() {
        //todo salvar em mémoria e não ficar recalculando
        double mean = getAverageFrequency();
        double temp = 0;
        for (Long a : frequencies)
            temp += (a - mean) * (a - mean);
        return Math.sqrt(temp / population);
    }

    public double getStdDevShareability() {
        //todo salvar em mémoria e não ficar recalculando
        double mean = getAverageShareability();
        double temp = 0;
        for (Double a : shareabilities)
            temp += (a - mean) * (a - mean);
        return Math.sqrt(temp / population);
    }

    public HashMap<MethodInfo, MethodStats> getMethodsInfoMap() {
        return methodsInfoMap;
    }

    //getting X% with most hits
    public double hitThreshold(int kStdDev) {
        return getAverageHitRatio() + (kStdDev * getStdDevHitRatio());
    }

    //getting X% with most misses
    public double missThreshold(int kStdDev) {
        return getAverageMissRatio() + (kStdDev * getStdDevMissRatio());
    }

    //getting X% most expensive methods
    public double expensivenessThreshold(int kStdDev) {
        return getAverageExecutionTime() + (kStdDev * getStdDevExecutionTimeRatio());
    }

    public double shareabilityThreshold(int kStdDev) {
        return getAverageShareability() + (kStdDev * getStdDevShareability());
    }

    //getting X% most frenquent
    public double frequencyThreshold(int kStdDev) {
        return getAverageFrequency() + (kStdDev * getStdDevFrequency());
    }

    /**
     * General miss ratio from a signature
     *
     * @param signature
     * @return
     */
    public double getMissRatio(String signature) {
        long occurrences = 0;
        long methods = 0;
        for (MethodInfo mi : methodsInfoMap.keySet()) {
            if (mi.getSignature().equals(signature)) {
                occurrences += methodsInfoMap.get(mi).getNumberOfSameOccurrences();
                methods++;
            }
        }
        return occurrences / methods;
    }

    /**
     * General hit ratio from a signature
     *
     * @param signature
     * @return
     */
    public double getHitRatio(String signature) {
        long occurrences = 0;
        long methods = 0;
        for (MethodInfo mi : methodsInfoMap.keySet()) {
            if (mi.getSignature().equals(signature)) {
                occurrences += methodsInfoMap.get(mi).getNumberOfSameOccurrences();
                methods++;
            }
        }
        return occurrences / methods;
    }

    public double getAverageFrequency() {
        return new BigDecimal(sumFrequency).divide(new BigDecimal(frequencies.size()), 5, RoundingMode.HALF_UP).doubleValue();
    }
}
