package br.ufrgs.inf.prosoft.adaptivecaching.sampling.metrics;

import org.aspectj.lang.JoinPoint;
import org.aspectj.lang.ProceedingJoinPoint;
import org.aspectj.lang.annotation.AfterThrowing;
import org.aspectj.lang.annotation.Around;
import org.aspectj.lang.annotation.Aspect;
import org.aspectj.lang.annotation.Pointcut;

import java.util.HashMap;
import java.util.Map;

import static java.lang.System.currentTimeMillis;

@Aspect
public class LightweightMetricAspect {

    @Pointcut(
        //any execution except the own framework
        "execution(!void *(..)) && !within(br.ufrgs.inf.prosoft.adaptivecaching..*) " +
        //avoid calls from repository while serializing objects, it is necessary if a hash could not be used
        "&& !cflow(call(* br.ufrgs.inf.prosoft.adaptivecaching.monitoring.storage..*(..)))"
    )
    public void anyCall() {
    }

    public static Map<String, LightweightMetrics> metrics;

    public LightweightMetricAspect() {
        metrics = new HashMap<>();
    }

    @Around("anyCall()")
    public Object aroundMethods(ProceedingJoinPoint joinPoint) throws Throwable {

        String signature = joinPoint.getSignature().toString();

        LightweightMetrics metric = metrics.get(signature);
        if(metric != null)
            metric.incOccurrence();
        else {
            metric = new LightweightMetrics();
            metric.incOccurrence();
            metrics.put(signature, metric);
        }

        //values are given in bytes, divide by 1000000 to get values in MB.
        Runtime runtime = Runtime.getRuntime();
        long startMemory = runtime.totalMemory() - runtime.freeMemory();
        long startTime = currentTimeMillis();
        long startThread = java.lang.Thread.activeCount();

        Object result = joinPoint.proceed();

        long endTime = currentTimeMillis();
        metric.addTime(startTime,endTime - startTime);
        long endMemory = runtime.totalMemory() - runtime.freeMemory();
        metric.addMemoryConsumption(endMemory - startMemory);
        long endThread = java.lang.Thread.activeCount();
        metric.addThreadNumber(endThread - startThread);



        return result;
    }

    @AfterThrowing(pointcut = "anyCall()", throwing = "e")
    public void myAfterThrowing(JoinPoint joinPoint, Throwable e) {

        String signature = joinPoint.getSignature().toString();

        LightweightMetrics metric = metrics.get(signature);
        if(metric != null)
            metric.incThrown();
        else {
            metric = new LightweightMetrics();
            metric.incThrown();
            metrics.put(signature, metric);
        }
    }
}
