package br.ufrgs.inf.prosoft.adaptivecaching.cachemanager.cacher.key;

import br.ufrgs.inf.prosoft.adaptivecaching.cachemanager.cacher.UpdateMethodCacher;
import br.ufrgs.inf.prosoft.adaptivecaching.cachemanager.util.Mnemos;
import org.aspectj.lang.JoinPoint;
import org.aspectj.lang.reflect.MethodSignature;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.lang.reflect.Method;
import java.lang.reflect.Modifier;
import java.util.Arrays;
import java.util.concurrent.atomic.AtomicInteger;

/**
 * Key of a callable target.
 *
 * @checkstyle DesignForExtensionCheck (2 lines)
 */
public class Key {

    /**
     * When instantiated.
     */
    private final transient long start;
    /**
     * How many times the key was already accessed.
     */
    private final transient AtomicInteger accessed;
    /**
     * MethodInfo.
     */
    private final transient Method method;
    /**
     * Object callable (or class, if static method).
     */
    private final transient Object target;
    /**
     * Arguments.
     */
    private final transient Object[] arguments;

    Logger logger = LoggerFactory.getLogger(Key.class);

    /**
     * Log level.
     */
//    private final int level;

    /**
     * Public ctor.
     *
     * @param point Joint point
     */
    public Key(final JoinPoint point) {
        this.start = System.currentTimeMillis();
        this.accessed = new AtomicInteger();
        this.method = MethodSignature.class
                .cast(point.getSignature()).getMethod();
        this.target = Key.targetize(point);
        this.arguments = point.getArgs();
//        if (this.method.isAnnotationPresent(Loggable.class)) {
//            this.level = this.method.getAnnotation(Loggable.class).value();
//        } else {
//            this.level = Loggable.DEBUG;
//        }
    }

//    /**
//     * Get log level.
//     * @return Log level of current method.
//     */
//    public final int getLevel() {
//        return this.level;
//    }

    /**
     * Calculate its target.
     *
     * @param point Proceeding point
     * @return The target
     */
    private static Object targetize(final JoinPoint point) {
        final Object tgt;
        final Method method = MethodSignature.class
                .cast(point.getSignature()).getMethod();
        if (Modifier.isStatic(method.getModifiers())) {
            tgt = method.getDeclaringClass();
        } else {
            tgt = point.getTarget();
        }
        return tgt;
    }

    @Override
    public final String toString() {
        return Mnemos.toText(this.method, this.arguments, true, false);
    }

    @Override
    public final int hashCode() {
        return this.method.hashCode();
    }

    @Override
    public final boolean equals(final Object obj) {
        final boolean equals;
        if (this == obj) {
            equals = true;
        } else if (obj instanceof Key) {
            final Key key = Key.class.cast(obj);
            equals = key.method.equals(this.method)
                    && this.target.equals(key.target)
                    && Arrays.deepEquals(key.arguments, this.arguments);
        } else {
            equals = false;
        }
        return equals;
    }

    /**
     * Send a result through, with necessary logging.
     *
     * @param result The result to send through
     * @return The same result/object
     * @checkstyle DesignForExtensionCheck (2 lines)
     */
    public Object through(final Object result) {
        final int hit = this.accessed.getAndIncrement();
        final Class<?> type = this.method.getDeclaringClass();
//        if (hit > 0 && LogHelper.enabled(this.level, type)) {
        logger.debug(
                "%s: %s from model (hit #%d, %[ms]s old)",
                this,
                Mnemos.toText(result, true, false),
                hit,
                System.currentTimeMillis() - this.start
        );
//        }
        return result;
    }

    /**
     * Is it related to the same target?
     *
     * @param point Proceeding point
     * @return True if the target is the same
     */
    public final boolean sameTarget(final JoinPoint point) {
        return Key.targetize(point).equals(this.target);
    }

}
