package br.ufrgs.inf.prosoft.tigrisframework.sampling;

import org.apache.commons.math3.distribution.BinomialDistribution;
import org.apache.commons.math3.random.RandomDataGenerator;
import org.apache.commons.math3.random.RandomGenerator;
import org.apache.commons.math3.stat.inference.TestUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.security.SecureRandom;
import java.util.Random;

/**
 * The type Sampling decision.
 */
public class Sampling implements Runnable {

    private boolean samplingEnabled = true;
    private int samplingRate = 50; // in percentage, 1 to 100
    private FrequencyDataSet population = new FrequencyDataSet(), sample = new FrequencyDataSet();

    Logger logger = LoggerFactory.getLogger(Sampling.class);

    /**
     * z confidence value, ex: 1.96 for 95%
     * p proportion of the population, 0.5 is default
     * e margin of error, ex: 0.05 for 5%
     */
    private double z = 1.96, p = 0.5, e = 0.05;

    private static SecureRandom rand = new SecureRandom();

    public Sampling(int initialSamplingRate) {
        samplingRate = initialSamplingRate;
    }

    public boolean samplingDecision(Granularity granularity) {
        population.addItem(granularity);
//
//        RandomDataGenerator randomDataGenerator = new RandomDataGenerator();
//        RandomGenerator
//        BinomialDistribution binomialDistribution = new BinomialDistribution(1, samplingRate);
//        binomialDistribution.sample();
//
//        UniformRandomProvider rg = RandomUtils.asUniformRandomProvider(new java.security.SecureRandom());

        boolean decision = samplingEnabled
                && ((rand.nextInt(100) + 1) > (100 - samplingRate)) // sampling rate evaluation
                && population.getProportion(granularity) > sample.getProportion(granularity); // sample has not enough items of that granularity compared to the population

        if (decision)
            sample.addItem(granularity);

        return decision;
    }

    public boolean isReady() {
        return
                // margin of error is lower than threshold
                getSampleSizeErrorMargin() < e
                // the sample has the min sample size based on the population
                && sample.getTotalItems() > getMinimumSampleSize()
                // proportion test
                && isSameProportion()
                // t-test
                && tTestEvaluation();
    }

    private boolean tTestEvaluation() {
        //To test the (2-sided) hypothesis sample mean = mu at the 95% level
        return TestUtils.tTest(population.getAsDescriptiveStatistics().getMean(),
                sample.getAsDescriptiveStatistics(),
                0.05);
    }

    //sample proportion is the same as population
    public boolean isSameProportion() {
        return population.getGranularities().stream().allMatch(granularity -> population.getProportion(granularity) == sample.getProportion(granularity));
    }

    /**
     * @return the minimum sample size for the population
     */
    public long getMinimumSampleSize() {
        long n_inf = (long) ((Math.pow(z, 2) * p * (1 - p)) / Math.pow(e, 2));
        return n_inf / (1 + ((n_inf - 1) / population.getTotalItems()));
    }

    public double getSampleSizeErrorMargin() {
        double e_n_inf = Math.sqrt((Math.pow(z, 2) * p * (1 - p)) / sample.getTotalItems());
        return e_n_inf * Math.sqrt((population.getTotalItems() - sample.getTotalItems()) / (population.getTotalItems() - 1));
    }

    public void disable() {
        samplingEnabled = false;
    }

    public void enable() {
        samplingEnabled = true;
    }

    public boolean isSamplingEnabled() {
        return samplingEnabled;
    }

    public int getSamplingRate() {
        return samplingRate;
    }

    public void adaptSamplingRate() {
        //TODO
    }

    @Override
    public void run() {
        logger.info("Running sampling adaptation.");

        if (isReady()) {
            logger.info("Sample is ready, releasing for analysis and resetting");
            //TODO
            releaseForAnalysis();
            reset();
        }
        adaptSamplingRate();
    }

    public void releaseForAnalysis() {
    }

    public void reset() {
    }
}
