package br.ufrgs.inf.prosoft.adaptivecaching.cachemanager.extensions.caffeine;

import br.ufrgs.inf.prosoft.adaptivecaching.cachemanager.model.Cache;
import br.ufrgs.inf.prosoft.adaptivecaching.cachemanager.model.CacheManager;
import com.github.benmanes.caffeine.cache.CacheLoader;
import com.github.benmanes.caffeine.cache.Caffeine;
import com.github.benmanes.caffeine.cache.CaffeineSpec;
import org.springframework.util.Assert;
import org.springframework.util.ObjectUtils;

import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;

/**
 * {@link CacheManager} implementation that lazily builds {@link CaffeineCache}
 * instances for each {@link #getCache} request. Also supports a 'static' mode
 * where the set of cachemanager names is pre-defined through {@link #setCacheNames},
 * with no dynamic creation of further cachemanager regions at runtime.
 * <p>
 * <p>The configuration of the underlying cachemanager can be fine-tuned through a
 * {@link Caffeine} builder or {@link CaffeineSpec}, passed into this
 * CacheManager through {@link #setCaffeine}/{@link #setCaffeineSpec}.
 * A {@link CaffeineSpec}-compliant expression value can also be applied
 * via the {@link #setCacheSpecification "cacheSpecification"} bean property.
 * <p>
 * <p>Requires Caffeine 2.1 or higher.
 */
public class CaffeineCacheManager implements CacheManager {

    private final ConcurrentMap<String, Cache> cacheMap = new ConcurrentHashMap<String, Cache>(16);

    private boolean dynamic = true;

    private Caffeine<Object, Object> cacheBuilder = Caffeine.newBuilder();

    private CacheLoader<Object, Object> cacheLoader;

    private boolean allowNullValues = true;


    /**
     * Construct a dynamic CaffeineCacheManager,
     * lazily creating cachemanager instances as they are being requested.
     */
    public CaffeineCacheManager() {
    }

    /**
     * Construct a static CaffeineCacheManager,
     * managing caches for the specified cachemanager names only.
     *
     * @param cacheNames the cache names
     */
    public CaffeineCacheManager(String... cacheNames) {
        setCacheNames(Arrays.asList(cacheNames));
    }

    /**
     * Set the Caffeine to use for building each individual
     * {@link CaffeineCache} instance.
     *
     * @param caffeine the caffeine
     * @see #createNativeCaffeineCache #createNativeCaffeineCache
     * @see com.github.benmanes.caffeine.cache.Caffeine#build() com.github.benmanes.caffeine.cache.Caffeine#build()
     */
    public void setCaffeine(Caffeine<Object, Object> caffeine) {
        Assert.notNull(caffeine, "Caffeine must not be null");
        doSetCaffeine(caffeine);
    }

    /**
     * Set the {@link CaffeineSpec} to use for building each individual
     * {@link CaffeineCache} instance.
     *
     * @param caffeineSpec the caffeine spec
     * @see #createNativeCaffeineCache #createNativeCaffeineCache
     * @see com.github.benmanes.caffeine.cache.Caffeine#from(CaffeineSpec) com.github.benmanes.caffeine.cache.Caffeine#from(CaffeineSpec)
     */
    public void setCaffeineSpec(CaffeineSpec caffeineSpec) {
        doSetCaffeine(Caffeine.from(caffeineSpec));
    }

    /**
     * Set the Caffeine cachemanager specification String to use for building each
     * individual {@link CaffeineCache} instance. The given value needs to
     * comply with Caffeine's {@link CaffeineSpec} (see its javadoc).
     *
     * @param cacheSpecification the cache specification
     * @see #createNativeCaffeineCache #createNativeCaffeineCache
     * @see com.github.benmanes.caffeine.cache.Caffeine#from(String) com.github.benmanes.caffeine.cache.Caffeine#from(String)
     */
    public void setCacheSpecification(String cacheSpecification) {
        doSetCaffeine(Caffeine.from(cacheSpecification));
    }

    /**
     * Set the Caffeine CacheLoader to use for building each individual
     * {@link CaffeineCache} instance, turning it into a LoadingCache.
     *
     * @param cacheLoader the cache loader
     * @see #createNativeCaffeineCache #createNativeCaffeineCache
     * @see com.github.benmanes.caffeine.cache.Caffeine#build(CacheLoader) com.github.benmanes.caffeine.cache.Caffeine#build(CacheLoader)
     * @see com.github.benmanes.caffeine.cache.LoadingCache
     */
    public void setCacheLoader(CacheLoader<Object, Object> cacheLoader) {
        if (!ObjectUtils.nullSafeEquals(this.cacheLoader, cacheLoader)) {
            this.cacheLoader = cacheLoader;
            refreshKnownCaches();
        }
    }

    /**
     * Return whether this cachemanager manager accepts and converts {@code null} values
     * for all of its caches.
     *
     * @return the boolean
     */
    public boolean isAllowNullValues() {
        return this.allowNullValues;
    }

    /**
     * Specify whether to accept and convert {@code null} values for all caches
     * in this cachemanager manager.
     * <p>Default is "true", despite Caffeine itself not supporting {@code null} values.
     * An internal holder object will be used to store user-level {@code null}s.
     *
     * @param allowNullValues the allow null values
     */
    public void setAllowNullValues(boolean allowNullValues) {
        if (this.allowNullValues != allowNullValues) {
            this.allowNullValues = allowNullValues;
            refreshKnownCaches();
        }
    }

    @Override
    public Collection<String> getCacheNames() {
        return Collections.unmodifiableSet(this.cacheMap.keySet());
    }

    /**
     * Specify the set of cachemanager names for this CacheManager's 'static' mode.
     * <p>The number of caches and their names will be fixed after a call to this method,
     * with no creation of further cachemanager regions at runtime.
     * <p>Calling this with a {@code null} collection argument resets the
     * mode to 'dynamic', allowing for further creation of caches again.
     *
     * @param cacheNames the cache names
     */
    public void setCacheNames(Collection<String> cacheNames) {
        if (cacheNames != null) {
            for (String name : cacheNames) {
                this.cacheMap.put(name, createCaffeineCache(name));
            }
            this.dynamic = false;
        } else {
            this.dynamic = true;
        }
    }

    @Override
    public Cache getCache(String name) {
        Cache cache = this.cacheMap.get(name);
        if (cache == null && this.dynamic) {
            synchronized (this.cacheMap) {
                cache = this.cacheMap.get(name);
                if (cache == null) {
                    cache = createCaffeineCache(name);
                    this.cacheMap.put(name, cache);
                }
            }
        }
        return cache;
    }

    /**
     * Create a new CaffeineCache instance for the specified cachemanager name.
     *
     * @param name the name of the cachemanager
     * @return the Spring CaffeineCache adapter (or a decorator thereof)
     */
    protected Cache createCaffeineCache(String name) {
        return new CaffeineCache(name, createNativeCaffeineCache(name), isAllowNullValues());
    }

    /**
     * Create a native Caffeine Cache instance for the specified cachemanager name.
     *
     * @param name the name of the cachemanager
     * @return the native Caffeine Cache instance
     */
    protected com.github.benmanes.caffeine.cache.Cache<Object, Object> createNativeCaffeineCache(String name) {
        if (this.cacheLoader != null) {
            return this.cacheBuilder.build(this.cacheLoader);
        } else {
            return this.cacheBuilder.build();
        }
    }

    private void doSetCaffeine(Caffeine<Object, Object> cacheBuilder) {
        if (!ObjectUtils.nullSafeEquals(this.cacheBuilder, cacheBuilder)) {
            this.cacheBuilder = cacheBuilder;
            refreshKnownCaches();
        }
    }

    /**
     * Create the known caches again with the current state of this manager.
     */
    private void refreshKnownCaches() {
        for (Map.Entry<String, Cache> entry : this.cacheMap.entrySet()) {
            entry.setValue(createCaffeineCache(entry.getKey()));
        }
    }

}
