'use strict';var lang_1 = require('angular2/src/facade/lang');
var exceptions_1 = require('angular2/src/facade/exceptions');
var collection_1 = require('angular2/src/facade/collection');
var async_1 = require('angular2/src/facade/async');
var route_recognizer_1 = require('./route_recognizer');
var route_config_impl_1 = require('./route_config_impl');
var async_route_handler_1 = require('./async_route_handler');
var sync_route_handler_1 = require('./sync_route_handler');
/**
 * `ComponentRecognizer` is responsible for recognizing routes for a single component.
 * It is consumed by `RouteRegistry`, which knows how to recognize an entire hierarchy of
 * components.
 */
var ComponentRecognizer = (function () {
    function ComponentRecognizer() {
        this.names = new collection_1.Map();
        // map from name to recognizer
        this.auxNames = new collection_1.Map();
        // map from starting path to recognizer
        this.auxRoutes = new collection_1.Map();
        // TODO: optimize this into a trie
        this.matchers = [];
        this.defaultRoute = null;
    }
    /**
     * returns whether or not the config is terminal
     */
    ComponentRecognizer.prototype.config = function (config) {
        var handler;
        if (lang_1.isPresent(config.name) && config.name[0].toUpperCase() != config.name[0]) {
            var suggestedName = config.name[0].toUpperCase() + config.name.substring(1);
            throw new exceptions_1.BaseException("Route \"" + config.path + "\" with name \"" + config.name + "\" does not begin with an uppercase letter. Route names should be CamelCase like \"" + suggestedName + "\".");
        }
        if (config instanceof route_config_impl_1.AuxRoute) {
            handler = new sync_route_handler_1.SyncRouteHandler(config.component, config.data);
            var path = config.path.startsWith('/') ? config.path.substring(1) : config.path;
            var recognizer = new route_recognizer_1.RouteRecognizer(config.path, handler);
            this.auxRoutes.set(path, recognizer);
            if (lang_1.isPresent(config.name)) {
                this.auxNames.set(config.name, recognizer);
            }
            return recognizer.terminal;
        }
        var useAsDefault = false;
        if (config instanceof route_config_impl_1.Redirect) {
            var redirector = new route_recognizer_1.RedirectRecognizer(config.path, config.redirectTo);
            this._assertNoHashCollision(redirector.hash, config.path);
            this.matchers.push(redirector);
            return true;
        }
        if (config instanceof route_config_impl_1.Route) {
            handler = new sync_route_handler_1.SyncRouteHandler(config.component, config.data);
            useAsDefault = lang_1.isPresent(config.useAsDefault) && config.useAsDefault;
        }
        else if (config instanceof route_config_impl_1.AsyncRoute) {
            handler = new async_route_handler_1.AsyncRouteHandler(config.loader, config.data);
            useAsDefault = lang_1.isPresent(config.useAsDefault) && config.useAsDefault;
        }
        var recognizer = new route_recognizer_1.RouteRecognizer(config.path, handler);
        this._assertNoHashCollision(recognizer.hash, config.path);
        if (useAsDefault) {
            if (lang_1.isPresent(this.defaultRoute)) {
                throw new exceptions_1.BaseException("Only one route can be default");
            }
            this.defaultRoute = recognizer;
        }
        this.matchers.push(recognizer);
        if (lang_1.isPresent(config.name)) {
            this.names.set(config.name, recognizer);
        }
        return recognizer.terminal;
    };
    ComponentRecognizer.prototype._assertNoHashCollision = function (hash, path) {
        this.matchers.forEach(function (matcher) {
            if (hash == matcher.hash) {
                throw new exceptions_1.BaseException("Configuration '" + path + "' conflicts with existing route '" + matcher.path + "'");
            }
        });
    };
    /**
     * Given a URL, returns a list of `RouteMatch`es, which are partial recognitions for some route.
     */
    ComponentRecognizer.prototype.recognize = function (urlParse) {
        var solutions = [];
        this.matchers.forEach(function (routeRecognizer) {
            var pathMatch = routeRecognizer.recognize(urlParse);
            if (lang_1.isPresent(pathMatch)) {
                solutions.push(pathMatch);
            }
        });
        // handle cases where we are routing just to an aux route
        if (solutions.length == 0 && lang_1.isPresent(urlParse) && urlParse.auxiliary.length > 0) {
            return [async_1.PromiseWrapper.resolve(new route_recognizer_1.PathMatch(null, null, urlParse.auxiliary))];
        }
        return solutions;
    };
    ComponentRecognizer.prototype.recognizeAuxiliary = function (urlParse) {
        var routeRecognizer = this.auxRoutes.get(urlParse.path);
        if (lang_1.isPresent(routeRecognizer)) {
            return [routeRecognizer.recognize(urlParse)];
        }
        return [async_1.PromiseWrapper.resolve(null)];
    };
    ComponentRecognizer.prototype.hasRoute = function (name) { return this.names.has(name); };
    ComponentRecognizer.prototype.componentLoaded = function (name) {
        return this.hasRoute(name) && lang_1.isPresent(this.names.get(name).handler.componentType);
    };
    ComponentRecognizer.prototype.loadComponent = function (name) {
        return this.names.get(name).handler.resolveComponentType();
    };
    ComponentRecognizer.prototype.generate = function (name, params) {
        var pathRecognizer = this.names.get(name);
        if (lang_1.isBlank(pathRecognizer)) {
            return null;
        }
        return pathRecognizer.generate(params);
    };
    ComponentRecognizer.prototype.generateAuxiliary = function (name, params) {
        var pathRecognizer = this.auxNames.get(name);
        if (lang_1.isBlank(pathRecognizer)) {
            return null;
        }
        return pathRecognizer.generate(params);
    };
    return ComponentRecognizer;
})();
exports.ComponentRecognizer = ComponentRecognizer;
//# sourceMappingURL=data:application/json;base64,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