import { isPresent } from 'angular2/src/facade/lang';
import { DebugNode, DebugElement, EventListener, getDebugNode, indexDebugNode, removeDebugNodeFromIndex } from 'angular2/src/core/debug/debug_node';
export class DebugDomRootRenderer {
    constructor(_delegate) {
        this._delegate = _delegate;
    }
    renderComponent(componentProto) {
        return new DebugDomRenderer(this, this._delegate.renderComponent(componentProto));
    }
}
export class DebugDomRenderer {
    constructor(_rootRenderer, _delegate) {
        this._rootRenderer = _rootRenderer;
        this._delegate = _delegate;
    }
    renderComponent(componentType) {
        return this._rootRenderer.renderComponent(componentType);
    }
    selectRootElement(selector) {
        var nativeEl = this._delegate.selectRootElement(selector);
        var debugEl = new DebugElement(nativeEl, null);
        indexDebugNode(debugEl);
        return nativeEl;
    }
    createElement(parentElement, name) {
        var nativeEl = this._delegate.createElement(parentElement, name);
        var debugEl = new DebugElement(nativeEl, getDebugNode(parentElement));
        debugEl.name = name;
        indexDebugNode(debugEl);
        return nativeEl;
    }
    createViewRoot(hostElement) { return this._delegate.createViewRoot(hostElement); }
    createTemplateAnchor(parentElement) {
        var comment = this._delegate.createTemplateAnchor(parentElement);
        var debugEl = new DebugNode(comment, getDebugNode(parentElement));
        indexDebugNode(debugEl);
        return comment;
    }
    createText(parentElement, value) {
        var text = this._delegate.createText(parentElement, value);
        var debugEl = new DebugNode(text, getDebugNode(parentElement));
        indexDebugNode(debugEl);
        return text;
    }
    projectNodes(parentElement, nodes) {
        var debugParent = getDebugNode(parentElement);
        if (isPresent(debugParent) && debugParent instanceof DebugElement) {
            nodes.forEach((node) => { debugParent.addChild(getDebugNode(node)); });
        }
        return this._delegate.projectNodes(parentElement, nodes);
    }
    attachViewAfter(node, viewRootNodes) {
        var debugNode = getDebugNode(node);
        if (isPresent(debugNode)) {
            var debugParent = debugNode.parent;
            if (viewRootNodes.length > 0 && isPresent(debugParent)) {
                var debugViewRootNodes = [];
                viewRootNodes.forEach((rootNode) => debugViewRootNodes.push(getDebugNode(rootNode)));
                debugParent.insertChildrenAfter(debugNode, debugViewRootNodes);
            }
        }
        return this._delegate.attachViewAfter(node, viewRootNodes);
    }
    detachView(viewRootNodes) {
        viewRootNodes.forEach((node) => {
            var debugNode = getDebugNode(node);
            if (isPresent(debugNode) && isPresent(debugNode.parent)) {
                debugNode.parent.removeChild(debugNode);
            }
        });
        return this._delegate.detachView(viewRootNodes);
    }
    destroyView(hostElement, viewAllNodes) {
        viewAllNodes.forEach((node) => { removeDebugNodeFromIndex(getDebugNode(node)); });
        return this._delegate.destroyView(hostElement, viewAllNodes);
    }
    listen(renderElement, name, callback) {
        var debugEl = getDebugNode(renderElement);
        if (isPresent(debugEl)) {
            debugEl.listeners.push(new EventListener(name, callback));
        }
        return this._delegate.listen(renderElement, name, callback);
    }
    listenGlobal(target, name, callback) {
        return this._delegate.listenGlobal(target, name, callback);
    }
    setElementProperty(renderElement, propertyName, propertyValue) {
        var debugEl = getDebugNode(renderElement);
        if (isPresent(debugEl) && debugEl instanceof DebugElement) {
            debugEl.properties.set(propertyName, propertyValue);
        }
        return this._delegate.setElementProperty(renderElement, propertyName, propertyValue);
    }
    setElementAttribute(renderElement, attributeName, attributeValue) {
        var debugEl = getDebugNode(renderElement);
        if (isPresent(debugEl) && debugEl instanceof DebugElement) {
            debugEl.attributes.set(attributeName, attributeValue);
        }
        return this._delegate.setElementAttribute(renderElement, attributeName, attributeValue);
    }
    /**
     * Used only in debug mode to serialize property changes to comment nodes,
     * such as <template> placeholders.
     */
    setBindingDebugInfo(renderElement, propertyName, propertyValue) {
        return this._delegate.setBindingDebugInfo(renderElement, propertyName, propertyValue);
    }
    /**
     * Used only in development mode to set information needed by the DebugNode for this element.
     */
    setElementDebugInfo(renderElement, info) {
        var debugEl = getDebugNode(renderElement);
        debugEl.setDebugInfo(info);
        return this._delegate.setElementDebugInfo(renderElement, info);
    }
    setElementClass(renderElement, className, isAdd) {
        return this._delegate.setElementClass(renderElement, className, isAdd);
    }
    setElementStyle(renderElement, styleName, styleValue) {
        return this._delegate.setElementStyle(renderElement, styleName, styleValue);
    }
    invokeElementMethod(renderElement, methodName, args) {
        return this._delegate.invokeElementMethod(renderElement, methodName, args);
    }
    setText(renderNode, text) { return this._delegate.setText(renderNode, text); }
}
//# sourceMappingURL=data:application/json;base64,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