/*
 * Copyright 2016 Red Hat, Inc. and/or its affiliates
 * and other contributors as indicated by the @author tags.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.keycloak.models.sessions.infinispan.stream;

import org.keycloak.models.sessions.infinispan.changes.SessionEntityWrapper;
import org.keycloak.models.sessions.infinispan.entities.SessionEntity;
import org.keycloak.models.sessions.infinispan.entities.UserSessionEntity;

import org.keycloak.models.sessions.infinispan.util.KeycloakMarshallUtil;
import java.io.IOException;
import java.io.ObjectInput;
import java.io.ObjectOutput;
import java.util.Map;
import java.util.function.Predicate;
import org.infinispan.commons.marshall.Externalizer;
import org.infinispan.commons.marshall.MarshallUtil;
import org.infinispan.commons.marshall.SerializeWith;

/**
 * @author <a href="mailto:sthorger@redhat.com">Stian Thorgersen</a>
 */
@SerializeWith(UserSessionPredicate.ExternalizerImpl.class)
public class UserSessionPredicate implements Predicate<Map.Entry<String, SessionEntityWrapper<UserSessionEntity>>> {

    private final String realm;

    private String user;

    private String client;

    private Integer expired;

    private Integer expiredRefresh;

    private String brokerSessionId;
    private String brokerUserId;

    private UserSessionPredicate(String realm) {
        this.realm = realm;
    }

    public static UserSessionPredicate create(String realm) {
        return new UserSessionPredicate(realm);
    }

    public UserSessionPredicate user(String user) {
        this.user = user;
        return this;
    }

    public UserSessionPredicate client(String clientUUID) {
        this.client = clientUUID;
        return this;
    }

    public UserSessionPredicate expired(Integer expired, Integer expiredRefresh) {
        this.expired = expired;
        this.expiredRefresh = expiredRefresh;
        return this;
    }

    public UserSessionPredicate brokerSessionId(String id) {
        this.brokerSessionId = id;
        return this;
    }

    public UserSessionPredicate brokerUserId(String id) {
        this.brokerUserId = id;
        return this;
    }

    @Override
    public boolean test(Map.Entry<String, SessionEntityWrapper<UserSessionEntity>> entry) {
        SessionEntity e = entry.getValue().getEntity();

        UserSessionEntity entity = (UserSessionEntity) e;

        if (!realm.equals(entity.getRealmId())) {
            return false;
        }

        if (user != null && !entity.getUser().equals(user)) {
            return false;
        }

        if (client != null && (entity.getAuthenticatedClientSessions() == null || !entity.getAuthenticatedClientSessions().containsKey(client))) {
            return false;
        }

        if (brokerSessionId != null && !brokerSessionId.equals(entity.getBrokerSessionId())) {
            return false;
        }

        if (brokerUserId != null && !brokerUserId.equals(entity.getBrokerUserId())) {
            return false;
        }

        if (expired != null && expiredRefresh != null && entity.getStarted() > expired && entity.getLastSessionRefresh() > expiredRefresh) {
            return false;
        }

        if (expired == null && expiredRefresh != null && entity.getLastSessionRefresh() > expiredRefresh) {
            return false;
        }

        return true;
    }

    public static class ExternalizerImpl implements Externalizer<UserSessionPredicate> {

        private static final int VERSION_1 = 1;

        @Override
        public void writeObject(ObjectOutput output, UserSessionPredicate obj) throws IOException {
            output.writeByte(VERSION_1);

            MarshallUtil.marshallString(obj.realm, output);
            MarshallUtil.marshallString(obj.user, output);
            MarshallUtil.marshallString(obj.client, output);
            KeycloakMarshallUtil.marshall(obj.expired, output);
            KeycloakMarshallUtil.marshall(obj.expiredRefresh, output);
            MarshallUtil.marshallString(obj.brokerSessionId, output);
            MarshallUtil.marshallString(obj.brokerUserId, output);

        }

        @Override
        public UserSessionPredicate readObject(ObjectInput input) throws IOException, ClassNotFoundException {
            switch (input.readByte()) {
                case VERSION_1:
                    return readObjectVersion1(input);
                default:
                    throw new IOException("Unknown version");
            }
        }

        public UserSessionPredicate readObjectVersion1(ObjectInput input) throws IOException, ClassNotFoundException {
            UserSessionPredicate res = new UserSessionPredicate(MarshallUtil.unmarshallString(input));
            res.user(MarshallUtil.unmarshallString(input));
            res.client(MarshallUtil.unmarshallString(input));
            res.expired(KeycloakMarshallUtil.unmarshallInteger(input), KeycloakMarshallUtil.unmarshallInteger(input));
            res.brokerSessionId(MarshallUtil.unmarshallString(input));
            res.brokerUserId(MarshallUtil.unmarshallString(input));
            return res;
        }
    }
}
