package azkaban.project.validator;

import java.util.HashSet;
import java.util.Set;

/**
 * The result of a project validation generated by a {@link ProjectValidator}. It contains
 * an enum of type {@link ValidationStatus} representing whether the validation passes,
 * generates warnings, or generates errors. Accordingly, three sets of String are also
 * maintained, storing the messages generated by the {@link ProjectValidator} at each of
 * the 3 {@link ValidationStatus} levels, i.e., {@link ValidationStatus#PASS},
 * {@link ValidationStatus#WARN}, and {@link ValidationStatus#ERROR}.
 */
public class ValidationReport {

  protected ValidationStatus _status;
  protected Set<String> _passMsgs;
  protected Set<String> _warningMsgs;
  protected Set<String> _errorMsgs;

  public ValidationReport() {
    _status = ValidationStatus.PASS;
    _passMsgs = new HashSet<String>();
    _warningMsgs = new HashSet<String>();
    _errorMsgs = new HashSet<String>();
  }

  /**
   * Add a message with status level being {@link ValidationStatus#PASS}
   *
   * @param msgs
   */
  public void addPassMsgs(Set<String> msgs) {
    if (msgs != null) {
      _passMsgs.addAll(msgs);
    }
  }

  /**
   * Add a message with status level being {@link ValidationStatus#WARN}
   *
   * @param msgs
   */
  public void addWarningMsgs(Set<String> msgs) {
    if (msgs != null) {
      _warningMsgs.addAll(msgs);
      if (!msgs.isEmpty() && _errorMsgs.isEmpty()) {
        _status = ValidationStatus.WARN;
      }
    }
  }

  /**
   * Add a message with status level being {@link ValidationStatus#ERROR}
   *
   * @param msgs
   */
  public void addErrorMsgs(Set<String> msgs) {
    if (msgs != null) {
      _errorMsgs.addAll(msgs);
      if (!msgs.isEmpty()) {
        _status = ValidationStatus.ERROR;
      }
    }
  }

  /**
   * Retrieve the status of the report.
   *
   * @return
   */
  public ValidationStatus getStatus() {
    return _status;
  }

  /**
   * Retrieve the messages associated with status level {@link ValidationStatus#PASS}
   *
   * @return
   */
  public Set<String> getPassMsgs() {
    return _passMsgs;
  }

  /**
   * Retrieve the messages associated with status level {@link ValidationStatus#WARN}
   *
   * @return
   */
  public Set<String> getWarningMsgs() {
    return _warningMsgs;
  }

  /**
   * Retrieve the messages associated with status level {@link ValidationStatus#ERROR}
   *
   * @return
   */
  public Set<String> getErrorMsgs() {
    return _errorMsgs;
  }

}
